/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Abstract superclass for optimization methods
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public abstract class Optimization
	{
        /// <summary>
        /// Event handler for Iteration notifications.
        /// </summary>
        /// <param name="sender">object that sent the event</param>
        /// <param name="e">event arguments</param>
        public delegate void IterationEventHandler(object sender, IterationEventArgs e);

        /// <summary>
        /// Event that is sent when one iteration of optimization method is passed.
        /// </summary>
        public event IterationEventHandler IterationNotify;

        /// <summary>
        /// Stop condition delegate. These are called after each iteration, passing the
        /// current 'best' parameters reached in the optimization as an argument.
        /// </summary>
        /// <param name="v">Current parameters</param>
        /// <returns>True, if stop condition has been met</returns>
        public delegate bool StopCondition(Vector v);

        /// <summary>
        /// Optimization constraint delegate. These functions are called before calling the target function
        /// to check that the current parameters are within limits. The parameters are passed by reference
        /// and the constraint methods may modify them, typically moving them back within the limits. The
        /// target function is called only with these modified parameters, but the result obtained is
        /// considered to be the target functions value at the original, unmodified, point.
        /// </summary>
        /// <param name="v">Parameters to check</param>
        /// <returns>Should return negative value, if within limits, positive, if outside.</returns>
        public delegate double Constraint(ref Vector v);

        /// <summary>
        /// Number of iterations performed so far
        /// </summary>
		private int iterations;

        /// <summary>
        /// Number of target function calls done so far
        /// </summary>
        private int functioncalls;

        /// <summary>
        /// Parameter change in last iteration
        /// </summary>
        private Vector delta;

        /// <summary>
        /// Target function. Not to be directly assigned, use SetTargetFunction instead
        /// </summary>
		private RealFunction targetfunc;

        /// <summary>
        /// List of stop conditions
        /// </summary>
		private List<StopCondition> Stops;

        /// <summary>
        /// List of constraints
        /// </summary>
		private List<Constraint> Constraints;

        /// <summary>
        /// The best parameters found so far
        /// </summary>
		private Vector currentParams;

        /// <summary>
        /// Initial parameters
        /// </summary>
        private Vector initialParams;

        /// <summary>
        /// For calling the target function outside this class.
        /// Masks the original target function and the modified, constrained function, as a single function
        /// </summary>
        public RealFunction TargetFunction
        {
            set { this.targetfunc = value; }
            get { return this.ConstrainedFunction; }
        }

        /// <summary>
        /// Get/Set optimization starting parameters.
        /// Note: setting new initial parameters between iterations may result in
        /// unpredictable behavior.
        /// </summary>
		public Vector InitialParams
		{
			get { return initialParams;}
			set
			{
				initialParams = value;
				currentParams = initialParams;
			}
		}

        /// <summary>
        /// Number of iterations made so far
        /// </summary>
        public int Iterations
        {
            get { return iterations; }
            set { }
        }

        /// <summary>
        /// Number of target function calls made so far
        /// </summary>
        public int FunctionCalls
        {
            get { return functioncalls; }
            set { }
        }

        /// <summary>
        /// Constrained target function. If function parameters are outside the allowed range,
        /// penalty function is applied.
        /// </summary>
        /// <param name="par">Function parameters</param>
        /// <returns>Modified target function value</returns>
        private double ConstrainedFunction(Vector par)
        {
            // Count function calls
            functioncalls++;

            // If there are defined constraints, apply them.
            if (Constraints.Count > 0)
            {
	            Vector limited = new Vector(par);
                double sum = 0.0;
                double penalty;

                // Add each constraint functions square to the original target function
                foreach (Constraint c in Constraints)
                {
                    penalty = c(ref limited);
                    if (penalty > 0.0)
                        sum += penalty;
                }
                
                // Return the modified target function value
                return targetfunc(limited) + sum;
            }
            // If there are no constraints, use original target function
            else return targetfunc(par);
        }

        /// <summary>
        /// Check the stop conditions
        /// </summary>
        /// <returns>True, if at least one stop condition is reached</returns>
        private bool Stop()
        {
            if (Stops.Count > 0)
            {
                foreach (StopCondition s in Stops)
                    if (s(currentParams)) return true;
            }
            return false;
        }

        /// <summary>
        /// Abstract method to be overriden in optimization method implementations.
        /// Performs a single iteration of the optimization process. Definition of 'single iteration,
        /// is left open, but it must return a single, 'best' point found so far.
        /// </summary>
        protected abstract Vector Step();

        /// <summary>
        /// Creates optimization object for use
        /// </summary>
        /// <param name="TargetFunction">goodness of fit (GoF)</param>
        /// <param name="Optimization_Info">optimization method info</param>
        /// <param name="c">ceiling values for parameters</param>
        /// <param name="f">floor values for parameters</param>
        /// <param name="init">initial model parameter values</param>
        /// <returns>created optimization object</returns>
        public static Optimization Create(
            RealFunction TargetFunction,
            OptimizationInfo Optimization_Info,
            Vector c,
            Vector f,
            Vector init)
        {
            // amount of parameters
            int length = c.Length;

            // floor, ceil and initial values must have equal length
            if( c.Length != f.Length || f.Length != init.Length ) 
                throw new OptimizationException("Parameter lengths must be equal.");

            // Optimization class that will be returned.
            Optimization method;

            // We try to create instance of given object:
            try
            {
                method = (Optimization)System.Activator.CreateInstance(Type.GetType( "TPClib.Model." + Optimization_Info.MethodName ) );
            }
            catch (Exception e)
            {
                throw new OptimizationException(e.Message);
            }

            // Initing the created object
            method.InitMethod( TargetFunction, Optimization_Info, c, f, init );
            return method;
        }

        /// <summary>
        /// Returns Info object of all methods currently in the library
        /// </summary>
        /// <returns>Method info in list</returns>
        public static List<OptimizationInfo> GetInfoOfAllMethods()
        {
            List<OptimizationInfo> result = new List<OptimizationInfo>();
            result.Add(PowellBrent.GetInfo());
            result.Add(RandomDirections.GetInfo());
            result.Add(ITGO.GetInfo());
            result.Add(NelderMead.GetInfo());
            return result;
        }

        /// <summary>
        /// Inits the created method.
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="info">Info of method (Not used in PowellBrent)</param>
        /// <param name="c">parameter ceiling values</param>
        /// <param name="f">parameter floor values</param>
        /// <param name="init">initial parameter values</param>
        protected abstract void InitMethod(
            RealFunction TargetFunction,
            OptimizationInfo info,
            Vector c,
            Vector f,
            Vector init);

        /// <summary>
        /// Default constructor
        /// </summary>
        public Optimization()
        {
            Stops = new List<StopCondition>();
            Constraints = new List<Constraint>();
        }

        /// <summary>
        /// Returns the dimension of this optimization
        /// </summary>
        /// <returns>Dimension, i.e. the number of optimized parameters</returns>
        public int Dim()
        {
            return initialParams.Dim;
        }

		/// <summary>
		/// Add new constraint(s)
		/// </summary>
		/// <param name="cns">
        /// Constraint function(s)
		/// </param>
		public void AddConstraint(params Constraint[] cns)
		{
            foreach (Constraint c in cns)
            {
                Constraints.Add(c);
            }
		}

		/// <summary>
		/// Remove constraint(S)
		/// </summary>
		/// <param name="cns">
        /// Constraint function(s) to remove
		/// </param>
		public void RemoveConstraint(params Constraint[] cns)
		{
            foreach (Constraint c in cns)
            {
                Constraints.Remove(c);
            }
		}

		/// <summary>
		/// Clear all constraints
		/// </summary>
		public void ClearConstraints()
		{
			Constraints.Clear();
		}

		/// <summary>
		/// Add stop condition(s)
		/// </summary>
		/// <param name="scs">
        /// Stop condition function(s)
		/// </param>
		public void AddStop(params StopCondition[] scs)
		{
            foreach (StopCondition s in scs)
            {
                Stops.Add(s);
            }
		}

		/// <summary>
		/// Remove stop condition(s)
		/// </summary>
		/// <param name="scs">
        /// Stop condition(s) to remove
		/// </param>
		public void RemoveStop(params StopCondition[] scs)
		{
            foreach (StopCondition s in scs)
            {
                Stops.Remove(s);
            }
		}

		/// <summary>
		/// Clear all stop conditions
		/// </summary>
		public void ClearStops()
		{
			Stops.Clear();
		}

        /// <summary>
        /// Get all stop conditions
        /// </summary>
        /// <returns>Array of stop condition functions</returns>
        public StopCondition[] GetStops()
        {
            return Stops.ToArray();
        }

        /// <summary>
        /// Get all Constraint functions
        /// </summary>
        /// <returns>Array of Constraint functions</returns>
        public Constraint[] GetConstraints()
        {
            return Constraints.ToArray();
        }

        /// <summary>
        /// Set target function to optimize
        /// </summary>
        /// <param name="function">Real function</param>
        public void SetTargetFunction(RealFunction function)
		{
			targetfunc = function;
		}

		/// <summary>
		/// Performs a single iteration of optimization.
		/// </summary>
		/// <returns>
		/// Returns true, if any supplied stop condition is fulfilled.
		/// </returns>
        public bool Iterate()
        {
            // Update current parameters
            Vector newparams = Step();
            delta = currentParams - newparams;
            currentParams = newparams;

            // Update iteration count
            iterations++;

            // notify clients
            if(IterationNotify != null)
                IterationNotify.Invoke(this, new IterationEventArgs());

            // Check if stop conditions are fulfilled and return
            return Stop();
        }

		/// <summary>
		/// Runs n iterations of optimization, or until a stop condition is reached
		/// </summary>
		/// <param name="n">
        /// Maximum iterations
		/// </param>
		/// <returns>
        /// True, if stop condition was reached, false otherwise.
		/// </returns>
		public bool Iterate(int n)
		{
			bool stop_reached = false;
			for(int i = 0; i<n; i++)
			{
				stop_reached = Iterate();
				if( stop_reached ) break;
			}
			return stop_reached;
		}
		
		/// <summary>
		/// Iterates until a stop condition is reached
		/// </summary>
		public void IterateUntilStop()
		{
            while (!Iterate())
            {
                // Loop until stop condition reached
            }
		}

        /// <summary>
        /// Get the current candidate for minimum
        /// </summary>
        /// <returns>Current minimum</returns>
        public Vector GetMinimum()
        {
            return currentParams;
        }

        /// <summary>
        /// Target function value at current minimum
        /// </summary>
        /// <returns>Target function value at minimum</returns>
        public double AtMinimum()
        {
            return targetfunc(GetMinimum());
        }

        /// <summary>
        /// Set upper limit for iterations
        /// </summary>
        /// <param name="n">Upper limit of iterations to run</param>
        public void SetMaxIterations(int n)
        {
            Stops.Add(delegate(Vector v) { return this.iterations >= n; });
        }

        /// <summary>
        /// Set a stop condition: optimization stops at first point p,
        /// where TargetFunction(p) is less than 'd'
        /// </summary>
        /// <param name="d">Upper limit for acceptable funtion value at minimum</param>
        public void SetTolerance(double d)
        {
            Stops.Add(delegate(Vector v) { return this.targetfunc(v) < d; });
        }

        /// <summary>
        /// Set stop condition: if the parameter vector has changed less than 'd'
        /// (measured in euclidean distance) in one iteration, optimization stops.
        /// </summary>
        /// <param name="d">Minimal change of parameters</param>
        public void SetConvergence(double d)
        {
            if (d < 0) throw new OptimizationException("Can't set negative convergence criterion");
            Stops.Add(delegate(Vector c) { return Vector.Norm(delta) < d; });
        }

        /// <summary>
        /// Set constraint: upper limit (inclusive) for a single parameter.
        /// This is a "hard" limit in the sense that it guarantees target function
        /// is not called with parameters outside this limit. Instead target function is
        /// called with value at the limit, plus a penalty function.
        /// The default penalty function rises very rapidly to Double.Max, but is not a "vertical"
        /// wall to enable all optimization methods to work correctly near the limits of the allowed
        /// area. If needed, different limit types can be supplied via delegates to the constraint
        /// mecahnism.
        /// (See the source code for an example)
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="limit">Upper limit (inclusive)</param>
        public void SetCeiling(int parameter, double limit)
        {
            Constraints.Add(
                delegate(ref Vector v)
                {
                    // If dif < 0, v is inside the limits
                    double dif = v[parameter] - limit;

                    // Define a penalty function in two parts.
                    // This penalty function will get values between 0 and Double.Max outside
                    // the limits, but it's derivative is always > 0 to ensure that all methods
                    // work correctly.
                    if (dif > 1.0)
                    {
                        v[parameter] = limit;
                        dif = Double.MaxValue * (1.0 - 1.0 / (dif * dif + 1));
                    }
                    else if (dif > 0.0)
                    {
                        v[parameter] = limit;
                        dif = 0.5 * dif * dif * Double.MaxValue;
                    }
                    return dif;
                }
            );
        }

        /// <summary>
        /// Set constraint: upper limits (inclusive) for all parameters.
        /// See SetCeiling(int parameter, double limit) for details.
        /// </summary>
        /// <param name="limits">Vector of upper limits</param>
        public void SetCeiling(Vector limits)
        {
            for (int i = 0; i < limits.Dim; i++)
            {
                SetCeiling(i, limits[i]);
            }
        }

        /// <summary>
        /// Set constraint: lower limit (inclusive) for a single parameter.
        /// See SetCeiling(int parameter, double limit) for details.
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="limit">Lower limit (inclusive)</param>
        public void SetFloor(int parameter, double limit)
        {
            Constraints.Add(
                delegate(ref Vector v)
                {
                    double dif = limit - v[parameter];

                    // Define a penalty function in two parts.
                    // Thi penalty function will get values between 0 and Double.Max outside
                    // the limits, but it's derivative is always > 0 to ensure that all methods
                    // work correctly.
                    if (dif > 1.0)
                    {
                        v[parameter] = limit;
                        dif = Double.MaxValue * (1.0 - 1.0 / (dif * dif + 1));
                    }
                    else if (dif > 0.0)
                    {
                        v[parameter] = limit;
                        dif = 0.5 * dif * dif * Double.MaxValue;
                    }
                    return dif;
                }
            );
        }

        /// <summary>
        /// Set constraint: lower limits (inclusive) for all parameters.
        /// See SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="limits">Vector of lower limits</param>
        public void SetFloor(Vector limits)
        {
            for (int i = 0; i < limits.Dim; i++)
            {
                SetFloor(i, limits[i]);
            }
        }

        /// <summary>
        /// Set a constraint: allowed parameter range (inclusive)
        /// See SetCeiling(int parameter, double limit) and
        /// SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="parameter">Index of constrained parameter</param>
        /// <param name="floor">Lower limit</param>
        /// <param name="ceiling">Upper limit</param>
        public void SetRange(int parameter, double floor, double ceiling)
        {
            SetCeiling(parameter, ceiling);
            SetFloor(parameter, floor);
        }

        /// <summary>
        /// Set a constraint: allowed parameter ranges (inclusive)
        /// See SetCeiling(int parameter, double limit) and
        /// SetFloor(int parameter, double limit) for details.
        /// </summary>
        /// <param name="floor">Vector of lower limits</param>
        /// <param name="ceiling">Vector of upper limits</param>
        public void SetRange(Vector floor, Vector ceiling)
        {
            SetCeiling(ceiling);
            SetFloor(floor);
        }

        /// <summary>
        /// Set a constraint: limit parameters to 'radius' (euclidean) distance from 'center'.
        /// This is an example of "soft" limits; parameters may move outside the radius, but
        /// a penalty is added to the target function value.
        /// </summary>
        /// <param name="center">Center point</param>
        /// <param name="radius">Max distance allowed (inclusive)</param>
        public void SetRadius(Vector center, double radius)
        {
            Constraints.Add(
                delegate(ref Vector v) {
                    Vector diff = v - center;
                    return radius * radius - Vector.Dot(diff, diff);
                });
        }
    }
}
