﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Runtime.InteropServices;
using System.Collections.Generic;


namespace TPClib.Model
{
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class Patlak : MultipleTimeGraphicalAnalysis
    {
        /// <summary>
        /// Calculation's result
        /// </summary>
        private double[] result;
        /// <summary>
        /// calculated x-values
        /// </summary>
        private double[] theta;
        /// <summary>
        /// calculated y-values
        /// </summary>
        private double[] dv;
        /// <summary>
        /// time points from acquisition start in ms (X1)
        /// </summary>
        private Vector plasma_start_times;
        /// <summary>
        /// time points from acquisition start in ms (X1)
        /// </summary>
        private Vector plasma_end_times;
        /// <summary>
        /// time points from acquisition start in ms (X1)
        /// </summary>
        private Vector plasma_times;
        /// <summary>
        /// arterial plasma (blood) (Y1) (reference)
        /// </summary>
        private Vector plasma; 
        /// <summary>
        /// time points from acquisition start in ms (X2)
        /// </summary>
        private Vector tissue_start_times;
        /// <summary>
        /// time points from acquisition end in ms (X2)
        /// </summary>
        private Vector tissue_end_times;
        /// <summary>
        /// time points from acquisition mid in ms (X2)
        /// </summary>
        private Vector tissue_mid_times;
        /// <summary>
        /// Tissue (Y2) (tac)
        /// </summary>
        private Vector tissue;
        /// <summary>
        /// Frame, where calculation starts
        /// </summary>
        private int start;
        /// <summary>
        /// Frame, where calculation stops
        /// </summary>
        private int stop;
        /// <summary>
        /// fixed ic value
        /// </summary>
        private double fixed_ic;
        /// <summary>
        /// True, if class is using frame mid times
        /// </summary>
        private bool using_mid_times = false;
        /// <summary>
        /// True, if plasma start end end times are given
        /// </summary>
        private bool plasma_start_and_end = false;
        
        
        /// <summary>
        /// Default constructor for Patlak, using tissue start and end times
        /// </summary>
        /// <param name="pl_tim">Plasma times</param>
        /// <param name="pl">Plasma values</param>
        /// <param name="ti_stim">Tissue start times</param>
        /// <param name="ti_etim">Tissue end times</param>
        /// <param name="ti">Tissue values</param>
        /// <param name="f_ic">fixed ic value, default double.MinValue</param>
        /// <param name="start_frame">Starting frame's number</param>
        /// <param name="stop_frame">Ending frame's number</param>
        public Patlak(Vector pl_tim, Vector pl, Vector ti_stim, Vector ti_etim, Vector ti, double f_ic, int start_frame, int stop_frame)
        {
            plasma_times = pl_tim;
            plasma = pl;
            tissue_start_times = ti_stim;
            tissue_end_times = ti_etim;
            tissue = ti;
            fixed_ic = f_ic;
            start = start_frame;
            stop = stop_frame;
        }
        /// <summary>
        /// Constructor for Patlak, using tissue start and end times and plasma start and end times
        /// </summary>
        /// <param name="f_ic">fixed ic value, default double.MinValue</param>
        /// <param name="pl">Plasma values</param>
        /// <param name="ti_stim">Tissue start times</param>
        /// <param name="ti_etim">Tissue end times</param>
        /// <param name="ti">Tissue values</param>
        /// <param name="start_frame">Starting frame's number</param>
        /// <param name="stop_frame">Ending frame's number</param>
        public Patlak(Vector pl_st, Vector pl_end, Vector pl, Vector ti_stim, Vector ti_etim, Vector ti, double f_ic, int start_frame, int stop_frame)
        {
            plasma_start_times = pl_st;
            plasma_end_times = pl_end;
            plasma = pl;
            tissue_start_times = ti_stim;
            tissue_end_times = ti_etim;
            tissue = ti;
            fixed_ic = f_ic;
            plasma_start_and_end = true;
            start = start_frame;
            stop = stop_frame;
        }
        
        /// <summary>
        /// Default constructor for Patlak, using tissue start and end times
        /// </summary>
        /// <param name="pl_tim">Plasma times</param>
        /// <param name="pl">Plasma values</param>
        /// <param name="ti_midtim">Tissue mid times</param>
        /// <param name="ti">Tissue values</param>
        /// <param name="f_ic">fixed ic value, default double.MinValue</param>
        /// <param name="start_frame">Starting frame's number</param>
        /// <param name="stop_frame">Ending frame's number</param>
        public Patlak(Vector pl_tim, Vector pl, Vector ti_midtim, Vector ti, double f_ic, int start_frame, int stop_frame)
        {
            plasma_times = pl_tim;
            plasma = pl;
            tissue_mid_times = ti_midtim;
            tissue = ti;
            fixed_ic = f_ic;
            using_mid_times = true;
            start = start_frame;
            stop = stop_frame;
        }
        
        /// <summary>
        /// Calculates model value
        /// </summary>
        /// <returns>Model parameter values, array {k, kSD, b, bSD, r, ySD}</returns>
        public override double[] Calculate() {
            
            if (!using_mid_times)
                if (tissue_start_times.Length != tissue_end_times.Length) throw new TPCException("Tissue start times length and tissue end times length must be the same.");
            
            // Interpolating plasma values
            double[] interpolatedPlasma;
            double[] integrated1_Plasma;
            double[] integrated2_Plasma;

            if (plasma_start_and_end)
            {
                LinearIntegration.IntegrateToFrameMidTimes(plasma_start_times/60, plasma_end_times/60, plasma, out integrated1_Plasma, out integrated2_Plasma);
                interpolatedPlasma = plasma;
            }
            else
            {
                if (using_mid_times)
                    LinearInterpolation.Interpolate(plasma_times / 60, plasma, tissue_mid_times / 60, out interpolatedPlasma, out integrated1_Plasma, out integrated2_Plasma);
                else
                    LinearInterpolation.InterpolateToPETFrames(plasma_times / 60, plasma, tissue_start_times / 60, tissue_end_times / 60, out interpolatedPlasma, out integrated1_Plasma, out integrated2_Plasma);
            }
                
            
	        // Calculating Ki for this region
            theta = new double[interpolatedPlasma.Length];
            dv = new double[interpolatedPlasma.Length];
            double[] ci = interpolatedPlasma;
            double[] ici = integrated1_Plasma;
            double[] ics = integrated2_Plasma;
            Vector ct = tissue;
            double[] wx = new double[tissue.Length];
            double[] wy = new double[tissue.Length];
            
	        // Calculating Patlak plot data
	        for(int fi = ct.Length-1; fi >= 0; fi--) 
                if(ci[fi] != 0.0) {
		            // theta (x axis)
		            theta[fi]=ici[fi]/ci[fi]; 
                    wx[fi]=1.0;
		            // dv (y axis)
		            dv[fi]=ct[fi]/ci[fi]; wy[fi]=1.0;
                    // check the close-to-zeroes in first frames
                    if(fi < tissue.Length/2) { 
			            if(theta[fi]>theta[tissue.Length-1] || dv[fi]>dv[tissue.Length-1])
				            theta[fi]=dv[fi]=wx[fi]=wy[fi]=0.0;
		            }
	            }
                else if(fixed_ic > double.MinValue) {
		        theta[fi]=ici[fi]; 
		        dv[fi]=ct[fi]; 
		        wx[fi]=wy[fi]=1.0;
	            }
                else theta[fi]=dv[fi]=wx[fi]=wy[fi]=0.0;

	        // Set x weight to 0, if integral is still <= 0, whether weighted or not
            bool cont = true;
            int fii = ct.Length - 1;
            while (cont)
            {
                if (fii < 0) cont = false;
                else if (cont == true && integrated1_Plasma[fii] <= 0.0) cont = false;
                    else
                    {
                       cont = true;
                        fii--;
                    }
            }
            for (; fii >= 0; fii--) wx[fii] = 0.0;

            double KiSD,Ki,Ic,IcSD,SWSS;
            KiSD = Ki = Ic = IcSD = SWSS = 0.0;
            double xm, xs, ym, ys;
            // check if y axis is constrained to fixed_ic
            if (fixed_ic > double.MinValue)
            {
                // Calculate the means and SDs of plot data
                Mean(theta, dv, start, stop, out xm, out xs, out ym, out ys);
                // Calculate the slope through constrained intercept and plot mean
                if (fixed_ic > double.MinValue) Ic = fixed_ic; 
                else Ic = 0.0;
                Ki = (ym - Ic) / xm; 
                if (xm != 0.0) KiSD = ys / xm; 
                SWSS = 1.0;
                return new double[] {Ki, KiSD, Ic, IcSD, SWSS};
            }
            else
            {
                // Using linear regression method
                // Delete negative or zero values from theta and dv
                for (int iii = 0; iii < wx.Length; iii++)
                {
                    if (wx[iii] <= 0.0 || wy[iii] <= 0.0) theta[iii] = dv[iii] = double.MinValue;
                }

                LinearRegression linReg = new LinearRegression();
                List<double> theta_n = new List<double>();
                List<double> dv_n = new List<double>();
                for (int i = 0; i < theta.Length; i++)
                {
                    if (i >= start)
                    {
                        theta_n.Add(theta[i]);
                        dv_n.Add(dv[i]);
                    }
                }
                result = linReg.pearson3(theta_n.ToArray(), dv_n.ToArray());
                return result;
            }
        }
        /// <summary>
        /// Calculates Patlak value for given VOI and also calculates Metabolic rate value
        /// </summary>
        /// <param name="ca">Concentration of native substrate in arterial plasma (mM)</param>
        /// <param name="density">Tissue density in MR calculation</param>
        /// <param name="lc">Lumped Constant in MR calculation</param>
        /// <returns>Model parameter values, array {k, kSD, b, bSD, r, ySD, mr, mrSD}</returns>
        public double[] CalculateMR(double ca, double lc, double density)
        {
            double f = 100*ca/(density*lc);
            double[] oldResult = this.Calculate();
            double[] newResult = new double[8];
            oldResult.CopyTo(newResult, 0);
            newResult[6] = f * oldResult[0];
            newResult[7] = f * oldResult[1];
            return newResult;
        }
        /// <summary>
        /// Returns region's theta values
        /// </summary>
        /// <returns>Array containing theta-values</returns>
        public double[] GetTheta()
        {
            this.Calculate();
            return this.theta;
        }
        /// <summary>
        /// Returns region's dv-values
        /// </summary>
        /// <returns>Array containing dv-values</returns>
        public double[] GetDv()
        {
            this.Calculate();
            return this.dv;
        }
        /// <summary>
        /// Calculates the mean and SD of data. Data (y data) may contain NA's.
        /// </summary>
        /// <param name="x">Data x values</param>
        /// <param name="y">Data y values</param>
        /// <param name="start">Starting index</param>
        /// <param name="stop">Ending index</param>
        /// <param name="xmean">Calculated x mean</param>
        /// <param name="xsd">Calculated SD of x mean</param>
        /// <param name="ymean">Calculated y mean</param>
        /// <param name="ysd">Calculated SD of y mean</param>
        private void Mean(double[] x, double[] y, int start, int stop, out double xmean, out double xsd, out double ymean, out double ysd)
        {
            double sumsqr, sqrsum;
            sumsqr = sqrsum = 0.0;
            int n = 0;

            for (int i = start; i <= stop; i++) 
                if (x[i] != double.NaN && y[i] != double.NaN)
                {
                    sumsqr += x[i] * x[i]; 
                    sqrsum += x[i]; 
                    n++;
                }
            if (n <= 0) throw new TPCException("All X and Y coordinates are NaN-values.");
            xmean = sqrsum / n;
            if (n == 1) xsd = 0.0;
            else
            {
                sqrsum *= sqrsum;
                xsd = System.Math.Sqrt((sumsqr - sqrsum / n) / (n - 1));
            }
            sumsqr = sqrsum = 0.0;
            n = 0;
            for (int i = start; i <= stop; i++)
            {
                if (x[i] != double.NaN && y[i] != double.NaN)
                {
                    sumsqr += y[i] * y[i];
                    sqrsum += y[i];
                    n++;
                }
            }
            if (n <= 0) throw new TPCException("All X and Y coordinates are NaN-values.");
            ymean = sqrsum / n;
            if (n == 1) ysd = 0.0;
            else
            {
                sqrsum *= sqrsum;
                ysd = System.Math.Sqrt((sumsqr - sqrsum / n) / (n - 1));
            }
        }// Mean
    }// Patlak
}
