﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections.Generic;

namespace TPClib.Model
{
    /// <summary>
    /// Generalization of polygon to n dimensions.
    /// </summary>
    class Polytope
    {
        /// <summary>
        /// List of vertices of this polytope and function values at vertices.
        /// Both lists are kept sorted.
        /// </summary>
        private List<Vector> vertices;
        private List<double> funcvals;

        /// <summary>
        /// Constructor
        /// </summary>
        public Polytope()
        {
            vertices = new List<Vector>();
            funcvals = new List<double>();
        }

        /// <summary>
        /// Number of vertices in this polytope
        /// </summary>
        /// <returns>Number of vertices</returns>
        public int Dim() { return vertices.Count; }

        /// <summary>
        /// Add a vertex
        /// </summary>
        /// <param name="v">Vertex point</param>
        /// <param name="val">Function value at vertex</param>
        public void Add(Vector v, double val)
        {
            // Add the point to its proper place;
            // function values are kept in increasing order
            int index = 0;
            List<double>.Enumerator e = funcvals.GetEnumerator();
            while (e.MoveNext())
            {
                if (e.Current < val) { index++; }
                else { break; }
            }
            vertices.Insert(index, v);
            funcvals.Insert(index, val);
        }

        /// <summary>
        /// Centroid of the vertices of the polytope, excluding the worst vertex
        /// </summary>
        /// <returns>Centroid point</returns>
        public Vector GetCentroid()
        {
            Vector sum = Vector.Sum(vertices.ToArray()) - GetMaximum();
            return sum / (vertices.Count - 1);
        }

        /// <summary>
        /// Reflection of v through the centroid
        /// </summary>
        /// <param name="v">Point to reflect</param>
        /// <param name="scale">Coefficient applied to the reflection distance</param>
        /// <returns>Reflected point</returns>
        public Vector GetReflection(Vector v, double scale)
        {
            Vector centr = this.GetCentroid();
            return (centr + scale * (centr - v));
        }

        /// <summary>
        /// The vertex associated with minimal function value
        /// </summary>
        /// <returns>Minimal vertex</returns>
        public Vector GetMinimum()
        {
            return vertices[0];
        }

        /// <summary>
        /// The vertex associated with maximal function value
        /// </summary>
        /// <returns>Maximal vertex</returns>
        public Vector GetMaximum()
        {
            return vertices[vertices.Count - 1];
        }

        /// <summary>
        /// Maximal function value at vertex points
        /// </summary>
        /// <returns>Maximal function value</returns>
        public double AtMaximum()
        {
            return funcvals[funcvals.Count - 1];
        }

        /// <summary>
        /// Minimal function value at vertex points
        /// </summary>
        /// <returns>Minimal function value</returns>
        public double AtMinimum()
        {
            return funcvals[0];
        }

        /// <summary>
        /// Replace the "worst" vertex (i.e. the one with maximal function value) with a new vertex
        /// </summary>
        /// <param name="v">New vertex</param>
        /// <param name="val">Function value at the new vertex</param>
        public void ReplaceWorst(Vector v, double val)
        {
            vertices.RemoveAt(vertices.Count - 1);
            funcvals.RemoveAt(funcvals.Count - 1);
            this.Add(v, val);
        }

        /// <summary>
        /// Check if 'val' is smaller than the largest function value in this polytope
        /// </summary>
        /// <param name="val">Function value to check</param>
        /// <returns>True, if 'val' is an improvement</returns>
        public bool IsBetter(double val)
        {
            return (val < this.AtMaximum());
        }

        /// <summary>
        /// Is a value better than the best in this polytope?
        /// </summary>
        /// <param name="val">Value to check</param>
        /// <returns>True, if value is better than any in this polytope</returns>
        public bool IsBest(double val)
        {
            return (val < this.AtMinimum());
        }

        /// <summary>
        /// Get a list of vertices in this polytope
        /// </summary>
        /// <returns>Vertex list</returns>
        public Vector[] GetVertices()
        {
            return vertices.ToArray();
        }
    }
}