﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// Direction Set or Powell's Method in Multidimensions.
    /// Minimization routine.
    /// This is the method of choice when you cannot easily
    /// calculate derivatives, and is not necessarily to be sneered at even if you can. 
    /// Although derivatives are not needed, the method does require
    /// a one-dimensional minimization sub-algorithm.
    /// This algorithm utilizes Brent’s method.
    /// Storage is of order N^2.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class PowellBrent : Optimization
    {
        /// <summary>
        /// The direction to go.
        /// </summary>
        private Vector dir;

        /// <summary>
        /// if delta is below this, the parameter is fixed
        /// </summary>
        private double fixedTolerance = 1.0e-20;

        /// <summary>
        /// The minimum difference between last optimum value.
        /// if current iteration and last iteration difference is 
        /// below this, the algorithm stops.
        /// </summary>
        private double minimumTolerance = 1e-10;

        /// <summary>
        /// true if parameter index in direction is fixed
        /// </summary>
        private bool[] fiXed;

        /// <summary>
        /// The direction NxN matrix.
        /// </summary>
        private double[,] dirmat;

        /// <summary>
        /// Current optimal point.
        /// </summary>
        private Vector p;

        /// <summary>
        /// Previus optimal point
        /// </summary>
        private Vector pprev;

        /// <summary>
        /// Previus previus optimal point.
        /// </summary>
        private Vector avg;

        /// <summary>
        /// Saves function value at initial point. 
        /// </summary>
        private double fvalue;

        /// <summary>
        /// Gets a multivariable funtion as a one dimensional "cut"
        /// </summary>
        private Linemethod linemin;

        /// <summary>
        /// Contructor with parameters. 
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="initial">initial parameter values</param>
        public PowellBrent(RealFunction function, Vector initial)
        {
            // generates automaticly delta.
            Vector delta = Vector.Fill(initial.Length, 0.001);

            Init(function, delta, initial);
        }

        /// <summary>
        /// Constructor. InitMethod must be called if this is used
        /// </summary>
        public PowellBrent() { }


        /// <summary>
        /// Gets info of Method and its parameters
        /// </summary>
        /// <returns>Info of method</returns>
        public static OptimizationInfo GetInfo()
        {
            OptimizationInfo info = new OptimizationInfo();
            info.MethodName = "PowellBrent";
            info.ParameterName = new System.Collections.Generic.List<string>();
            info.ParameterValue = new System.Collections.Generic.List<double>();
            return info;
        }

        /// <summary>
        /// Inits PowellBrent. This method is called from Parent class (optimization)
        /// Create() -method. 
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="info">Info of method (Not used in PowellBrent)</param>
        /// <param name="c">parameter ceiling values</param>
        /// <param name="f">parameter floor values</param>
        /// <param name="initial">initial parameter values</param>
        protected override void InitMethod(
            RealFunction function,
            OptimizationInfo info,
            Vector c,
            Vector f,
            Vector initial)
        {
            // Powell is not using min and max values. Values must be converted to
            // deltas
            int length = c.Length;
            double[] delta = new double[length];
            for (int i = 0; i < length; i++) delta[i] = 0.001; //Math.Abs(c[i] - f[i]) * 0.1;

            Init(function, delta, initial);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="func">optimized function</param>
        /// <param name="delta">Used to initialize Direction Matrix</param>
        /// <param name="initial">initial parameter values</param>
        public PowellBrent(RealFunction func, Vector delta, Vector initial)
        {
            Init(func, delta, initial);
        }

        /// <summary>
        /// Initialization of Powell-Brent optimization.
        /// </summary>
        /// <param name="func">optimized function</param>
        /// <param name="powellDelta">Used to initialize Direction Matrix</param>
        /// <param name="initial">initial parameter values</param>
        private void Init(RealFunction func, double[] powellDelta, double[] initial)
        {
            if (func == null || powellDelta == null || initial == null)
                throw new NotImplementedException();

            if (initial.Length < 1)
                throw new NotImplementedException();

            int parNr = initial.Length;
            this.TargetFunction = func;
            this.dir = new double[parNr];
            this.pprev = new double[parNr];
            this.fiXed = new bool[parNr];
            this.dirmat = new double[parNr, parNr];

            // Checks which paratemers are fixed.
            for (int i = 0; i < parNr; i++)
            {
                this.fiXed[i] = (Math.Abs(powellDelta[i]) < fixedTolerance);
            }

            // Initiates matrix for directions.
            for (int i = 0; i < parNr; i++)
            {
                for (int j = 0; j < parNr; j++)
                {
                    dirmat[i, j] = (i == j) ? powellDelta[i] : 0.0;
                }
            }

            // Saves the initial point
            pprev = initial;

            this.p = initial;
            this.InitialParams = this.p;
            this.fvalue = TargetFunction(p);
            this.linemin = new Linemethod(delegate(double t)
            {
                return TargetFunction(p + t * dir);
            }, 2.0e-4);
        }

        /// <summary>
        /// Single minimization step; called from Optimization.Iterate()
        /// </summary>
        /// <returns>Best found parameters.</returns>
        protected override Vector Step()
        {
            // current function value
            double fprev1 = fvalue, fprev2;
            // the best index in direction set 
            int ibig = 0;
            // the best difference
            double newdel, del = 0.0;

            // Loops over all directions (matrix row) in the set.
            // After we know the best (ibig) direction to go.
            for (int row = 0; row < p.Length; row++)
            {
                // Leave the fixed parameters alone.
                if (!fiXed[row])
                {
                    for (int col = 0; col < p.Length; col++)
                    {
                        // yes, we don't move along fixed component.
                        dir[col] = fiXed[col] ? 0.0 : dirmat[col, row];
                    }

                    // Store the latest.
                    fprev2 = fvalue;

                    // minimazes along direction and get new minimum value
                    fvalue = linemin.LineMinimum(ref p, ref dir);

                    // Stores the best index
                    newdel = Math.Abs(fprev2 - fvalue);
                    if (newdel > del)
                    {
                        del = newdel;
                        ibig = row;
                    }
                }
            }

            // Check if direction isn't any better
            double c1 = 2.0 * Math.Abs(fprev1 - fvalue);
            double c2 = minimumTolerance * (Math.Abs(fprev1) + Math.Abs(fvalue));
            if (c1 <= c2)
            {
                return p;
            }

            // and the average direction moved.
            avg = 2 * p - pprev;
            dir = p - pprev;

            // saves the old starting point
            pprev = p;

            // gets the extrapolated function value.
            fprev2 = TargetFunction(avg);

            // if we found better minimum.
            if (fprev2 < fprev1)
            {
                double sq1 = fprev1 - fvalue - del;
                double sq2 = fprev1 - fprev2;
                double t = 2.0 * (fprev1 - 2.0 * fvalue + fprev2) * sq1 * sq1 - del * sq2 * sq2;

                //
                if (t < 0.0)
                {
                    //
                    this.fvalue = linemin.LineMinimum(ref p, ref dir);

                    // updates the direction matrix 
                    // move to the minimum of the new direction,
                    // and save the new direction.
                    for (int i = 0; i < p.Length; i++)
                    {
                        dirmat[i, ibig] = dirmat[i, p.Length - 1];
                        dirmat[i, p.Length - 1] = dir[i];
                    }
                }
            }

            // the next iteration
            return p;
        }
    }
}