﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// RTCM model
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class RTCMModel : Model
    {
        /// <summary>
        /// Vectors for storing time intervals and reference values
        /// </summary>
        private Vector times;
        private Vector refintegral;
        private Vector refvals;

        /// <summary>
        /// Tissue compartment data
        /// </summary>
        private Vector cb;
        private Vector cf;

        /// <summary>
        /// Number of samples simulated
        /// </summary>
        public override int Samples
        {
            get { return times.Length; }
        }


        /// <summary>
        /// Return info of Model
        /// </summary>
        /// <returns>Info of model</returns>
        protected override ModelInfo GetInfo()
        {
            ModelInfo info = new ModelInfo("RTCMModel", "This is RTCMModel description...", true);

            List<ModelParameter> parameters = new List<ModelParameter>();
            parameters.Add(new ModelParameter("r1", 0, 10, 1));
            parameters.Add(new ModelParameter("k2", 0, 10, 1));
            parameters.Add(new ModelParameter("k3", 0, 10, 1));
            parameters.Add(new ModelParameter("bp", 0.1, 2, 1));
            info.Parameters = parameters.ToArray();
            info.ClassType = this.GetType();

            return info;
        }
        
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Reference value vector</param>
        public override void Create(Vector t, params Vector[] r)
        {
            if (r[0].Length != t.Length) throw new ModelException("Different number of sample times and values");

            // Calculate the time intervals and the reference integral
            Info = GetInfo();
            times = HalfTimeDifs(t);
            if (r.Length > 0) refvals = r[0];
            refintegral = CalcRefIntegral(refvals, times);
            cb = new Vector(Samples);
            cf = new Vector(Samples);
        }

        /// <summary>
        /// Constructor
        /// </summary>
        public RTCMModel() { Info = GetInfo(); }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Reference value vector</param>
        public RTCMModel(Vector t, Vector r) { Create(t, r); }

        /// <summary>
        /// Wrapper function that overrides virtual method Model.Simulate
        /// </summary>
        /// <param name="par">Model parameters</param>
        /// <returns>Vector of simulated values</returns>
        public override Vector Simulate(params double[] par)
        {
            if (par.Length != 4) throw new ModelException("Wrong number of parameters");
            return Simulate(par[0], par[1], par[2], par[3]);
        }

        /// <summary>
        /// Simulates tissue TAC using reference tissue compartment model (original) and reference region TAC, at reference region TAC times.
        /// The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
        /// </summary>
        /// <param name="r1">Ratio K1/K1'</param>
        /// <param name="k2">Rate constant of the model</param>
        /// <param name="k3">Rate constant of the model</param>
        /// <param name="bp">K3/K4</param>
        /// <returns>Vector of simulated values</returns>
        public Vector Simulate(double r1, double k2, double k3, double bp)
        {
            double k4 = k3 / bp;
            if (Double.IsNaN(k4)) throw new ModelException("Parameter bp cannot be 0.0");

            Vector simval = new Vector(Samples);

            //Cumulative integral values
            double cbi = 0.0;
            double cfi = 0.0;
            double cbi_last = 0.0;
            double cfi_last = 0.0;

            double cb_last = 0.0;
            double cf_last = 0.0;

            // Variables for intermediate values
            double f, b, w, dt2, cr, cri;

            for (int i = 0; i < Samples; i++)
            {
                dt2 = times[i];
                cr = refvals[i];
                cri = refintegral[i];

                w = k2 + k3 + k2 * k4 * dt2;
                f = cfi_last + dt2 * cf_last;
                b = cbi_last + dt2 * cb_last;

                cf[i] = ((1.0 + k4 * dt2) * (r1 * cr + k2 * cri) + k4 * b - w * f) /
                    (1.0 + dt2 * (w + k4));
                cfi = cfi_last + dt2 * (cf_last + cf[i]);

                cb[i] = (k3 * cfi - k4 * b) / (1.0 + k4 * dt2);
                cbi = cbi_last + dt2 * (cb_last + cb[i]);

                simval[i] = cf[i] + cb[i];
                cb_last = cb[i];
                cf_last = cf[i];
                cfi_last = cfi;
                cbi_last = cbi;
            }

            return simval;
        }

        /// <summary>
        /// Get the first compartment TAC
        /// </summary>
        /// <returns>Vector of simulated 1st tissue compartment values</returns>
        public Vector GetCompartment1()
        {
            return cf;
        }

        /// <summary>
        /// Get the second compartment TAC
        /// </summary>
        /// <returns>Vector of simulated 2nd tissue compartment values</returns>
        public Vector GetCompartment2()
        {
            return cb;
        }
    }
}
