﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Model
{
    /// <summary>
    /// RandomDirections optimization method. Generates a random direction (from normal
    /// distribution) and searches along this direction as long as there is improvement,
    /// repeating the process minimum is found. Self adjusts step length.
    /// Simple and converges to local minimum with probability of 1, but not very efficient.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class RandomDirections : Optimization
    {
        /// <summary>
        /// Current step length
        /// </summary>
        private double dt;

        /// <summary>
        /// Current direction
        /// </summary>
        private Vector dir;

        /// <summary>
        /// Random number generator
        /// </summary>
        private Random rand;

        /// <summary>
        /// Number of succesfull steps needed before lengthtening the step length
        /// </summary>
        private const int L=3;

        /// <summary>
        /// Number of failed steps needed before shortening the step length
        /// </summary>
        private const int K = 2;

        /// <summary>
        /// Failed steps
        /// </summary>
        private int failcount;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="tf">Function to optimize</param>
        /// <param name="initial">Initial parameters</param>
        public RandomDirections(RealFunction tf, Vector initial)
        {
            InitialParams = initial;
            SetTargetFunction(tf);
            rand = new Random();
            failcount = 0;
            dt = 1;
        }

        /// <summary>
        /// Constructor. InitMethod must be called if this is used
        /// </summary>
        public RandomDirections() { }

        /// <summary>
        /// Gets info of Method and its parameters
        /// </summary>
        /// <returns>Info of method</returns>
        public static OptimizationInfo GetInfo()
        {
            OptimizationInfo info = new OptimizationInfo();
            info.MethodName = "RandomDirections";
            info.ParameterName = new System.Collections.Generic.List<string>();
            info.ParameterValue = new System.Collections.Generic.List<double>();
            return info;
        }

        /// <summary>
        /// Inits RandomDirections. This method is called from Parent class (optimization)
        /// Create() -method. 
        /// </summary>
        /// <param name="function">optimized function</param>
        /// <param name="info">Info of method (Not needed with NelderMean)</param>
        /// <param name="c">parameter ceiling values</param>
        /// <param name="f">parameter floor values</param>
        /// <param name="initial">initial parameter values</param>
        protected override void InitMethod(
            RealFunction function,
            OptimizationInfo info,
            Vector c,
            Vector f,
            Vector initial)
        {
            InitialParams = initial;
            SetTargetFunction(function);
            rand = new Random();
            failcount = 0;
            dt = 1;
        }


        /// <summary>
        /// Get the last search vector
        /// </summary>
        /// <returns>Last search vector</returns>
        public Vector GetSearchVector()
        {
            return dt * dir;
        }

        /// <summary>
        /// Single iteration
        /// </summary>
        /// <returns>Best point found</returns>
        protected override Vector Step()
        {
            // Search fails, until proven otherwise
            failcount++;

            // Generate new random search vector (of length 'dt')
            Vector d = dt * RandomDirection();

            // Start from the current minimum found
            Vector x = GetMinimum();

            // Calculate function values
            double fval1 = TargetFunction(x);
            double fval2 = TargetFunction(x + d);

            // If better point is found at generated direction, use that direction
            if (fval2 < fval1)
            {
                dir = d;
            }
            // If it is worse, assume that the opposite direction is better
            else
            {
                dir = -d;
                fval2 = TargetFunction(dir);
            }

            // Step to the next point in the chosen direction
            Vector y = x + dir;

            // Steps taken succesfully
            int success = 0;

            // Loop as long as better points are found
            while (fval2 < fval1)
            {
                // Progress made, so failcount down to zero and success up
                failcount = 0;
                success++;

                // Take the next step
                x = y;
                y = y + dir;

                // Calculate function values
                fval1 = fval2;
                fval2 = TargetFunction(y);

                // If succefull enough in this direction, double the step length
                // Zero the success counter
                if (success == L)
                {
                    dt *= 2;
                    dir = dir * 2;
                    success = 0;
                }
            }
            // end while loop

            // If no success at all in the generated direction, failcount > 0
            // If it reaches K...
            if (failcount == K)
            {
                // ... halve the step length and zero the counter
                dt /= 2;
                failcount = 0;
            }

            // Return the result of last succesfull search
            return x;
        }

        /// <summary>
        /// Generate random (normally distributed) direction vector;
        /// </summary>
        /// <returns>Random unit vector</returns>
        private Vector RandomDirection()
        {
            int n = Dim();
            Vector v = new Vector(n);

            // If a_i are uniformly distributed in [0,1], then
            // sqrt(12/n)*(a_1 + a_2 + ... + a_n - n/2)
            // approximates N(0,1)
            // (Alternatively fix n=12 to get simpler formula of
            // (a_1 +...+ a_12 -6) )
            double sq12n = Math.Sqrt(12.0 / (double)n);

            for (int k = 0; k < n; k++)
            {
                double sum = -n * 0.5;
                for (int i = 0; i < n; i++)
                {
                    sum += rand.NextDouble();
                }

                sum *= sq12n;
                v[k] = sum;
            }

            // Normalize the vector and return
            return v/Vector.Norm(v);
        }
    }
}
