﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Model
{
    /// <summary>
    /// Runge-Kutta method for solving ordinary differential equations.
    /// These methods propagate a solution over an interval by combining
    /// the information from several elementary steps, and then using
    /// the information obtained to match a Taylor series
    /// expansion up to some higher order.
    /// </summary>
    public class RungeKutta : ODESolver
    {
        /// <summary>
        /// Ordinaray differential equaion to be solved.
        /// </summary>
        private ODE ode;

        /// <summary>
        /// current input state of ODE solution.
        /// </summary>
        private double[] input;

        /// <summary>
        /// Four estimate points for calculating next step.
        /// </summary>
        private double[] K1, K2, K3, K4;

        /// <summary>
        /// general index value
        /// </summary>
        private int i;

        /// <summary>
        /// Initializes the method.
        /// </summary>
        /// <param name="ode">the equations to be solved</param>
        public RungeKutta(ODE ode)
        {
            this.ode = ode;
            double[] vars = ode.GetVars();
            int N = vars.Length;
            input = new double[N];
            K1 = new double[N];
            K2 = new double[N];
            K3 = new double[N];
            K4 = new double[N];
        }

        /// <summary>
        /// Iterates one step.
        /// </summary>
        /// <param name="stepSize">step size</param>
        public void Step(double stepSize)
        {
            // array of variables
            double[] vars = ode.GetVars();

            // number of variables in x array
            int N = vars.Length;

            // evaluate at time t
            ode.Evaluate(vars, K1);

            // set up input to diffeqs
            for (i = 0; i < N; i++)
                input[i] = vars[i] + K1[i] * stepSize / 2;

            // evaluate at time t+stepSize/2
            ode.Evaluate(input, K2);

            // set up input to diffeqs
            for (i = 0; i < N; i++)
                input[i] = vars[i] + K2[i] * stepSize / 2;

            // evaluate at time t+stepSize/2
            ode.Evaluate(input, K3);

            // set up input to diffeqs
            for (i = 0; i < N; i++)
                input[i] = vars[i] + K3[i] * stepSize;

            // evaluate at time t+stepSize
            ode.Evaluate(input, K4);

            // modify the variables
            for (i = 0; i < N; i++)
            {
                vars[i] = vars[i] + (K1[i] + 2 * K2[i] + 2 * K3[i] + K4[i]) * stepSize / 6;
            }
        }
    }
}
