﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.IO;
using System.Drawing;

namespace TPClib.Model
{
    /// <summary>
    /// Scalable Vector Graphics File
    /// </summary>
    /// <seealso cref="http://www.w3.org/Graphics/SVG/"/>
    public class SVGFile : TPCFile
    {
        /// <summary>
        /// Class containing possible plot types
        /// </summary>
        public enum PlotType
        {
            SYMBOLS, LINE, LINE_AND_SYMBOLS
        }
        /// <summary>
        /// Enum class containing all possible symbol types
        /// </summary>
        public enum SvgSymbolType
        {
            RECTANGLE,
            CIRCLE,
            UPTRIANGLE,
            DOWNTRIANGLE,
            DIAMOND,
            LEFTTRIANGLE,
            RIGHTTRIANGLE
        }
        /// <summary>
        /// Class, which contains all information for one plot
        /// </summary>
        public class PlotParameters
        {
            public PlotType plot_type;
            public string plot_name;
            public Color plot_color;
            public SvgSymbolType plot_symbol;
            public double[] plot_x_points;
            public double[] plot_y_points;
            public double stroke_width = 10.0;

            /// <summary>
            /// Constructor, no color, no symbol, no stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = Color.Black;
                plot_symbol = SvgSymbolType.RECTANGLE;
            }
            /// <summary>
            /// Constructor, no color, no symbol, stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="sw">Plot's width (uses value 10.0 if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, double sw)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = Color.Black;
                plot_symbol = SvgSymbolType.RECTANGLE;
                stroke_width = sw;
            }
            /// <summary>
            /// Constructor, no symbol, no stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="color">Plot's color (black, if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, Color color)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = color;
                plot_symbol = SvgSymbolType.RECTANGLE;
            }
            /// <summary>
            /// Constructor, no symbol, stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="color">Plot's color (black, if not given)</param>
            /// <param name="sw">Plot's width (uses value 10.0 if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, Color color, double sw)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = color;
                plot_symbol = SvgSymbolType.RECTANGLE;
                stroke_width = sw;
            }
            /// <summary>
            /// Constructor, no color, no stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="symbol">Plot's symbol (rectangle, if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, SvgSymbolType symbol)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = Color.Black;
                plot_symbol = symbol;
            }
            /// <summary>
            /// Constructor, no color, stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="symbol">Plot's symbol (rectangle, if not given)</param>
            /// <param name="sw">Plot's width (uses value 10.0 if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, SvgSymbolType symbol, double sw)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = Color.Black;
                plot_symbol = symbol;
                stroke_width = sw;
            }
            /// <summary>
            /// Constructor, color and symbol, no stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="color">Plot's color (black, if not given)</param>
            /// <param name="symbol">Plot's symbol (rectangle, if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, Color color, SvgSymbolType symbol)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = color;
                plot_symbol = symbol;
            }
            /// <summary>
            /// Constructor, color and symbol and stroke width
            /// </summary>
            /// <param name="ptype">Plot type</param>
            /// <param name="pname">Plot name (appears, when user holds mouse on the plot)</param>
            /// <param name="xp">Plot's x-coordinates</param>
            /// <param name="yp">Plot's y-coordinates</param>
            /// <param name="color">Plot's color (black, if not given)</param>
            /// <param name="symbol">Plot's symbol (rectangle, if not given)</param>
            /// <param name="sw">Plot's width (uses value 10.0 if not given)</param>
            public PlotParameters(PlotType ptype, string pname, double[] xp, double[] yp, Color color, SvgSymbolType symbol, double sw)
            {
                plot_type = ptype;
                plot_name = pname;
                plot_x_points = xp;
                plot_y_points = yp;
                plot_color = color;
                plot_symbol = symbol;
                stroke_width = sw;
            }
        }// class LineParameters

        /// <summary>
        /// SVG file name
        /// </summary>
        private string file_name;
        /// <summary>
        /// All plots to be drawn
        /// </summary>
        public PlotParameters[] plots;
        /// <summary>
        /// SVG's main title
        /// </summary>
        private string main_title;
        /// <summary>
        /// title of x-axis
        /// </summary>
        private string x_title;
        /// <summary>
        /// title of y-axis
        /// </summary>
        private string y_title;

        public SVGFile(string fname, PlotParameters[] plot_params, string mTitle, string xTitle, string yTitle)
        {
            file_name = fname;
            plots = plot_params;
            main_title = mTitle;
            x_title = xTitle;
            y_title = yTitle;
        }

        /// <summary>
        /// Method for writing SVG-plot to file. Data and file name are given with constructor.
        /// </summary>
        public override void WriteFile()
        {
            StreamWriter fp_svg = File.CreateText(file_name);
            Svg_viewports viewports = new Svg_viewports();

            double maxPlotX = 0;
            for (int i = 0; i < plots.Length; i++)
                for (int j = 0; j < plots[i].plot_x_points.Length; j++)
                    if (plots[i].plot_x_points[j] > maxPlotX) maxPlotX = plots[i].plot_x_points[j];
            
            double maxy = 0;
            for (int i = 0; i < plots.Length; i++)
                for (int j = 0; j < plots[i].plot_y_points.Length; j++)
                    if (plots[i].plot_y_points[j] > maxy) maxy = plots[i].plot_y_points[j];

            //Calculate the axis tics
            viewports.x.min = 0.0;
            viewports.x.max = maxPlotX;
            viewports.y.min = 0.0;
            viewports.y.max = maxy;
            SvgCalculateAxes(viewports);
            
            // Set the plot window and window area sizes
            int is_label = 0;
            viewports = SvgDefineViewports(0, 0, main_title.Length, y_title.Length, x_title.Length, is_label, viewports);
            
            // Initiate graphics file
            SvgInitiate(fp_svg, 0, 0, viewports);

            // Put the graphic titles into their own viewports
            SvgCreateMainTitle(fp_svg, main_title, "", viewports);
            SvgCreateYAxisTitle(fp_svg, y_title, viewports);
            SvgCreateXAxisTitle(fp_svg, x_title, viewports);

            // Put the plot into its own viewport
            SvgStartPlotViewport(fp_svg, viewports);

            // Start coordinate area viewport
            SvgStartCoordinateViewport(fp_svg, viewports);

            // Write plot axes
            SvgWriteAxes(fp_svg, viewports);

            // Draw the plots
            
            string tac_id;
            string tac_title;
            int ln = 0;
            int sn = 0;
            int lsn = 0;

            for (int ri = 0; ri < plots.Length; ri++)
            {
                tac_title = plots[ri].plot_name;

                switch (plots[ri].plot_type)
                {
                    case PlotType.LINE:
                        tac_id = "line_" + ln;
                        ln++;
                        break;
                    case PlotType.SYMBOLS:
                        tac_id = "plot_" + sn;
                        sn++;
                        break;

                    case PlotType.LINE_AND_SYMBOLS:
                    default:
                        tac_id = "plot_and_line_" + lsn;
                        lsn++;
                        break;
                }
                
                SvgWriteTac(fp_svg, viewports, plots[ri].plot_type, tac_id, tac_title, plots[ri].plot_x_points, plots[ri].plot_y_points, plots[ri].plot_color, plots[ri].plot_symbol, plots[ri].stroke_width); 
            }
            
            // Close the coordinate viewport
            SvgEndCoordinateViewport(fp_svg);

            // Write the axis tics
            SvgWriteXTicks(fp_svg, viewports);
            SvgWriteYTicks(fp_svg, viewports);
            
            // Close the Plot Viewport
            SvgEndPlotViewport(fp_svg, viewports);

            //Close the SVG file
            SvgClose(fp_svg);
            fp_svg.Close();
            Console.WriteLine("SVG File Created SucacessFully");
        }
        
        /// <summary>
        /// Method ReadFile not implemented, because not needed.
        /// </summary>
        public override void ReadFile()
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Data structure for defining SVG-viewport position
        /// </summary>
        private class Svg_viewport_pos {
            public int iss, x, y, w, h, chr_size;

            /// <summary>
            /// Default constructor
            /// </summary>
            public Svg_viewport_pos() { }
        }
        /// <summary>
        /// Data structure for defining SVG-viewport coordinates
        /// </summary>
        private class Svg_coord {
            public double min, max, scale, origo;
            public int tick_nr;
            public double[] tick; 
            public double tickscale;
            public int tick_decimals;
            public int upper_margin;

            /// <summary>
            /// Default constructor
            /// </summary>
            public Svg_coord() { }
        }
        /// <summary>
        /// Data structure for defining SFG-viewports
        /// </summary>
        private class Svg_viewports {
            public Svg_viewport_pos main_viewport;
            public Svg_viewport_pos main_title_viewport;
            public Svg_viewport_pos yaxis_title_viewport;
            public Svg_viewport_pos xaxis_title_viewport;
            public Svg_viewport_pos label_area_viewport;
            public Svg_viewport_pos plot_area_viewport;
            public Svg_viewport_pos coordinate_area_viewport;
            /* Coordinate area contents */
            public Svg_coord x;
            public Svg_coord y;

            /// <summary>
            /// Default constructor
            /// </summary>
            public Svg_viewports() {
                main_viewport = new Svg_viewport_pos();
                main_title_viewport = new Svg_viewport_pos(); ;
                yaxis_title_viewport = new Svg_viewport_pos(); ;
                xaxis_title_viewport = new Svg_viewport_pos(); ;
                label_area_viewport = new Svg_viewport_pos(); ;
                plot_area_viewport = new Svg_viewport_pos(); ;
                coordinate_area_viewport = new Svg_viewport_pos(); 
                x = new Svg_coord();
                y = new Svg_coord();
            }
        }

        /// <summary>
        /// Calculate the axis tick positions.
        /// Before calling this, viewport must be filled with curve min and max values.
        /// This routine checks that max>min, changing the values if necessary.
        /// </summary>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgCalculateAxes(Svg_viewports vp) {
            
            /* x axis */
            double[] new_range = AxisCheckRange(vp.x.min, vp.x.max);
            vp.x.min = new_range[0];
            vp.x.max = new_range[1];
            vp.x.tick_nr = 6;
            vp.x.tick = new double[7];

            int actual_tick_nr;
            AxisTickPositions(vp.x.min, vp.x.max, vp.x.tick, vp.x.tick_nr, out actual_tick_nr, out vp.x.tickscale, out vp.x.tick_decimals);
            vp.x.tick_nr = actual_tick_nr;
            
            /* y axis */
            new_range = AxisCheckRange(vp.y.min, vp.y.max);
            vp.y.min = new_range[0];
            vp.y.max = new_range[1];
            vp.y.tick_nr = 10;
            vp.y.tick = new double[11];

            AxisTickPositions(vp.y.min, vp.y.max, vp.y.tick, vp.y.tick_nr, out actual_tick_nr, out vp.y.tickscale, out vp.y.tick_decimals);
            
        } //SvgCalculateAxes

        /// <summary>
        /// Check and if necessary correct axis range: min must be smaller than max.
        /// Also, if the range is close to zero, then make it larger.
        /// Also, if the range is relatively large, then change it to start from zero.
        /// </summary>
        /// <param name="b">axis range minimum</param>
        /// <param name="e">axis range maximum</param>
        private double[] AxisCheckRange(double b, double e)
        {
            double begin;
            double end;
            double temp;
            begin = b;
            end = e;

            if (begin < end)
            {
                // fine now
            }
            else if (begin > end)
            {
                temp = end; end = begin; begin = temp;
            }
            else
            { // this means that begin==end
                if (begin == 0.0)
                {
                    begin = -1.0; end = 1.0;
                }
                else if (begin < 0.0)
                {
                    begin = 2.0; end = 0.0;
                }
                else
                {
                    begin = 0.0; end = 2.0;
                }
                
            }
            /* If range is very small */
            if ((end - begin) < 1.0e-98)
            {
                if (begin >= 1.0e-98 || begin < 0.0) begin -= 1.0e-98;
                else if (begin >= 0.0) begin = 0.0;
                if (end <= -1.0e-98 || end > 0.0) end += 1.0e-98;
                else if (end < 0.0) end = 0.0;
            }
            /* If range is relatively large */
            if (begin > 0.0 && end > 0.0)
            {
                if ((end - begin) > 3.3 * (begin)) begin = 0.0;
            }
            else if (begin < 0.0 && end < 0.0)
            {
                if ((end - begin) > 3.3 * (-end)) end = 0.0;
            }
            /* If data range is relatively small (compared to level) */
            temp = (end - begin) * 2.0 / (Math.Abs(end) + Math.Abs(begin));
            if (temp < 0.01)
            {
                temp = 0.5 * (end - begin) * 0.01 / temp;
                if (begin < 0.0 || begin > temp) begin -= temp; else begin = 0.0;
                if (end > 0.0 || end < -temp) end += temp; else end = 0.0;
            }
            return new double[] { begin, end };
        }//AxisCheckRange

        /// <summary>
        /// Define suitable tick positions for XY plot
        /// </summary>
        /// <param name="begin">axis range minimum</param>
        /// <param name="end">axis range maximum</param>
        /// <param name="tics">Pointer to array where tick values will be written</param>
        /// <param name="tick_nr">Input: Max allowed nr of ticks</param>
        /// <param name="actual_tick_nr">Output: actual nr of ticks</param>
        /// <param name="scale_factor">Output: Suggested scale factor (x10^sf). NULL, if not needed</param>
        /// <param name="tick_decimals">Output: tick value precision (nr of decimals). NULL, if not needed.</param>
        private void AxisTickPositions(double begin, double end, double[] ticks, int tick_nr, out int actual_tick_nr, out double scale_factor, out int tick_decimals)
        {
            int ti;
            double step, scale;
            scale_factor = double.MinValue;
            tick_decimals = int.MinValue;

            /* Check input */
            if (ticks == null || tick_nr < 1) throw new TPCException("ticks must contain more than 0 values");
            if (end <= begin)
            {
                actual_tick_nr = 0;
                scale_factor = 1.0;
                tick_decimals = 0;
            }
            else
            {
                // Calculate the requested tick step size
                step = (end - begin) / (double)(tick_nr);

                /* Calculate a feasible step size and scale */
                scale = 1.0;
                while (step <= 0.5) { step *= 10.0; scale /= 10.0; }
                while (step > 5.0) { step /= 10.0; scale *= 10.0; }
                if (step < 1.0) { step = 1.0; }
                else if (step < 2.0) { step = 2.0; }
                else { step = 5.0; }
                
                ti = 0; ticks[ti] = step * scale * Math.Ceiling(begin / (step * scale));
                while (ticks[ti] <= (end + (end - begin) * 0.00001) && ti < tick_nr)
                {
                    ti++; ticks[ti] = ticks[0] + step * scale * (double)ti;
                }
                actual_tick_nr = ti;

                /* Quit, if user did not like scales etc */
                if (scale_factor != double.NaN || tick_decimals != double.NaN)
                {

                    /* Calculate step scale */
                    double step_scale;
                    step_scale = Math.Log10(scale);
                    
                    /* Find the highest tick label */
                    double tick_high;
                    int tick_scale;
                    double scale_dif;
                    if (Math.Abs(ticks[0]) > Math.Abs(ticks[actual_tick_nr - 1])) tick_high = ticks[0];
                    else tick_high = ticks[actual_tick_nr - 1];
                    
                    tick_scale = 0;
                    while (Math.Abs(tick_high) < 1.0) { tick_high *= 10.0; tick_scale--; }
                    while (Math.Abs(tick_high) >= 10.0) { tick_high /= 10.0; tick_scale++; }
                    

                    /* Calculate the difference between tick and step scales */
                    scale_dif = tick_scale - step_scale;
                    
                    /* That determines the tick number precision */
                    if (tick_decimals != double.NaN)
                    {
                        tick_decimals = (int)(1 + scale_dif);
                    }

                    /* Calculate the preferred tick scale factor, or quit, if it is not needed */
                    if (scale_factor != double.NaN)
                    {
                        scale_factor = tick_scale;
                    }
                }
            }
        }//AxisTickPositions

        /// <summary>
        /// Define the viewport positions for furher use. All measures are in pixels.
        /// Returns pointer to the file if successful and NULL in case of an error.
        /// </summary>
        /// <param name="main_viewport_width">Main viewport width (zero if default is used)</param>
        /// <param name="main_viewport_height">Main viewport height (zero if default is used)</param>
        /// <param name="is_main_title">Is there main title? no=0, yes<>0</param>
        /// <param name="is_yaxis_title">Is there y axis title? no=0, yes<>0</param>
        /// <param name="is_xaxis_title">Is there x axis title? no=0, yes<>0</param>
        /// <param name="is_label_area">Is there label area? no=0, yes<>0</param>
        /// <param name="v_ports">structure which will be filled with viewport positions and sizes</param>
        private Svg_viewports SvgDefineViewports(int main_viewport_width, int main_viewport_height, int is_main_title, int is_yaxis_title, int is_xaxis_title, int is_label_area, Svg_viewports v_ports) 
        {
            int SVG_VIEWPORT_WIDTH = 1000;
            int SVG_VIEWPORT_HEIGHT = 1000;
            if (v_ports == null) throw new TPCException("Viewport is null");
            Svg_viewports vp = v_ports;
            
            /* Set main viewport */
            vp.main_viewport.iss=1;
            vp.main_viewport.x = vp.main_viewport.y=0;
            if(main_viewport_width < 1) vp.main_viewport.w = SVG_VIEWPORT_WIDTH;
            if(main_viewport_height < 1) vp.main_viewport.h = SVG_VIEWPORT_HEIGHT;
            
            /* Set viewport for main title(s) */
            if(is_main_title == 0) {
                vp.main_title_viewport.iss=0;
                vp.main_title_viewport.x = vp.main_title_viewport.y=0;
                vp.main_title_viewport.w = vp.main_title_viewport.h=0;
                vp.main_title_viewport.chr_size=0;
            } else {
                vp.main_title_viewport.iss = 1;
                vp.main_title_viewport.x = vp.main_title_viewport.y = 0;
                vp.main_title_viewport.w = vp.main_viewport.w;
                vp.main_title_viewport.h = vp.main_viewport.w/12;
                vp.main_title_viewport.chr_size = 5*vp.main_title_viewport.h/10;
            }
            
            /* Set viewport for x axis title */
            if(is_xaxis_title == 0) {
                vp.xaxis_title_viewport.iss = 0;
                vp.xaxis_title_viewport.x = 0;
                vp.xaxis_title_viewport.y = vp.main_title_viewport.h;
                vp.xaxis_title_viewport.w = vp.main_title_viewport.w;
                vp.xaxis_title_viewport.h = 0;
                vp.xaxis_title_viewport.chr_size = 0;
            } else {
                vp.xaxis_title_viewport.iss = 1;
                vp.xaxis_title_viewport.x = 0;
                vp.xaxis_title_viewport.w = vp.main_viewport.w;
                vp.xaxis_title_viewport.h = vp.main_viewport.h/18;
                vp.xaxis_title_viewport.y = vp.main_viewport.h-vp.xaxis_title_viewport.h;
                vp.xaxis_title_viewport.chr_size = 7*vp.xaxis_title_viewport.h/10;
            }
            
            /* Set viewport for y axis title */
            if(is_yaxis_title == 0) {
                vp.yaxis_title_viewport.iss = 0;
                vp.yaxis_title_viewport.x = 0;
                vp.yaxis_title_viewport.y = vp.main_title_viewport.h;
                vp.yaxis_title_viewport.w = 0;
                vp.yaxis_title_viewport.h = vp.main_viewport.h - vp.main_title_viewport.h - vp.xaxis_title_viewport.h;
                vp.yaxis_title_viewport.chr_size=0;
            } else {
                vp.yaxis_title_viewport.iss = 1;
                vp.yaxis_title_viewport.x = 0;
                vp.yaxis_title_viewport.y = vp.main_title_viewport.h;
                vp.yaxis_title_viewport.w = vp.main_viewport.w/18;
                vp.yaxis_title_viewport.h = vp.main_viewport.h - vp.main_title_viewport.h - vp.xaxis_title_viewport.h;
                if(vp.xaxis_title_viewport.iss != 0)
                    vp.yaxis_title_viewport.chr_size = vp.xaxis_title_viewport.chr_size;
                else
                    vp.yaxis_title_viewport.chr_size = 7*vp.yaxis_title_viewport.w/10;
            }

            /* Set viewport for label area */
            if(is_label_area == 0) {
                vp.label_area_viewport.iss=0;
                vp.label_area_viewport.x=vp.main_viewport.w;
                vp.label_area_viewport.y=vp.main_title_viewport.h;
                vp.label_area_viewport.w=0;
                vp.label_area_viewport.h=vp.main_viewport.h - vp.main_title_viewport.h - vp.xaxis_title_viewport.h;
            } else {
                vp.label_area_viewport.iss = 1;
                vp.label_area_viewport.x = 3*vp.main_viewport.w/4;
                vp.label_area_viewport.y = vp.main_title_viewport.h;
                vp.label_area_viewport.w = vp.main_viewport.w - vp.label_area_viewport.x;
                vp.label_area_viewport.h = vp.main_viewport.h - vp.main_title_viewport.h - vp.xaxis_title_viewport.h;
            }
            /* Set viewport for plot area */
            vp.plot_area_viewport.iss=1;
            vp.plot_area_viewport.x=vp.yaxis_title_viewport.w;
            vp.plot_area_viewport.y=vp.main_title_viewport.h;
            vp.plot_area_viewport.w=vp.main_viewport.w-vp.yaxis_title_viewport.w-vp.label_area_viewport.w;
            vp.plot_area_viewport.h=vp.main_viewport.h-vp.main_title_viewport.h-vp.xaxis_title_viewport.h;
            
            /* Set plot area contents (inside plot area) */
            vp.coordinate_area_viewport.iss=1;
            if(vp.y.tickscale==0)
                vp.coordinate_area_viewport.x=vp.plot_area_viewport.w/10;
            else
                vp.coordinate_area_viewport.x=vp.plot_area_viewport.w/5;
            vp.coordinate_area_viewport.y=0;
            vp.coordinate_area_viewport.w=vp.plot_area_viewport.w-vp.coordinate_area_viewport.x;
            vp.coordinate_area_viewport.h=19 * vp.plot_area_viewport.h/20;
            
            /* Calculate the character size for tick labels etc */
            vp.plot_area_viewport.chr_size=vp.coordinate_area_viewport.chr_size = (int)Math.Ceiling(0.67*(double)(vp.plot_area_viewport.h - vp.coordinate_area_viewport.h));

            return vp;
        }//SvgDefineViewports

        /// <summary>
        /// Initiate a new SVG graphics file.
        /// Returns pointer to the file if successful and NULL in case of an error.
        /// </summary>
        /// <param name="filename">File name for SVG graphics</param>
        /// <param name="height">Plot height in cm; 0, if not predefined</param>
        /// <param name="width">Plot width in cm; 0, if not predefined</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgInitiate(StreamWriter fp, double height, double width, Svg_viewports vp) 
        {
            /* Check input */
            if(fp == null || vp == null) throw new TPCException("Streamwriter or viewport null");
            if(vp.main_viewport.w<3 || vp.main_viewport.h<3) throw new TPCException("Viewport height or width too low");

            fp.WriteLine("<?xml version=\"1.0\"?>");
            //fp.WriteLine("<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\"");
            //fp.WriteLine(" \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">");
            fp.WriteLine("<svg version=\"1.1\" baseProfile=\"full\"");
            fp.WriteLine("     xmlns=\"http://www.w3.org/2000/svg\"");
            fp.WriteLine("     xmlns:xlink=\"http://www.w3.org/1999/xlink\"");
            fp.WriteLine("     xmlns:ev=\"http://www.w3.org/2001/xml-events\"");
            fp.WriteLine("     viewBox=\"0 0 " + vp.main_viewport.w + " " + vp.main_viewport.h + "\"");
            fp.Write("     preserveAspectRatio=\"xMinYMin meet\"");
            
            if(width>0.0) {
                fp.WriteLine(("\n      width=\"" + width + "cm\"").Replace(',', '.'));
            }
            if(height>0.0) {
                fp.Write(("      height=\"" + height + "cm\"").Replace(',', '.'));
            }
            fp.WriteLine(">");
        }//SvgInitiate

        /// <summary>
        /// Create SVG plot main title.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="main_title_text">Text for main title</param>
        /// <param name="sub_title_text">Text for sub title</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgCreateMainTitle(StreamWriter fp, string main_title_text, string sub_title_text, Svg_viewports vp) 
        {
            double main_pos, sub_pos;

            /* Check the input */
            if(vp.main_title_viewport.iss != 0) 
            {
                if(fp == null) throw new TPCException("Filestream was closed too early");
                
                /* Create a new viewport for main title */
                fp.WriteLine("\n   <!-- Main title viewport -->");
                fp.WriteLine("  <svg x=\"" + vp.main_title_viewport.x + "px\" y=\"" + vp.main_title_viewport.y + "px\" width=\"" + vp.main_title_viewport.w + "px\" height=\"" + vp.main_title_viewport.h + "\""); 
                fp.WriteLine("      viewBox=\"0 0 " + vp.main_title_viewport.w + " " + vp.main_title_viewport.h + "\""); 
                fp.Write("      preserveAspectRatio=\"xMidYMid meet\"");
                fp.WriteLine(">");
                
                /* Determine the y positions for titles */
                
                if(main_title_text.Length>0) {
                    /* main title exists */
                    if(sub_title_text.Length==0) {
                        /* no subtitle */
                        main_pos=0.75*(double)vp.main_title_viewport.h;
                        sub_pos=vp.main_title_viewport.h;
                    } 
                    else { 
                        /* also subtitle */
                    main_pos=0.52*(double)vp.main_title_viewport.h;
                    sub_pos=0.9*(double)vp.main_title_viewport.h;
                    }
                } 
                else { 
                    /* no main title */
                    if(sub_title_text.Length==0) { 
                        /* no subtitle either */
                        main_pos=0.5*(double)vp.main_title_viewport.h;
                        sub_pos=vp.main_title_viewport.h;
                    } 
                    else { 
                        /* only subtitle */
                        main_pos=0.0;
                        sub_pos=0.4*(double)vp.main_title_viewport.h;
                    }
                }

                /* Set main title text */
                fp.WriteLine(("    <text x=\"" + vp.main_title_viewport.w/2 + "\" y=\"" + main_pos + "\"").Replace(',', '.'));
                fp.Write("        font-family=\"Verdana\"");
                fp.Write(" text-anchor=\"middle\"");
                fp.Write(" font-size=\"" + vp.main_title_viewport.chr_size + "\"");
                fp.Write(" fill=\"black\"");
                fp.WriteLine(">");
                fp.WriteLine("      " + main_title_text);
                fp.WriteLine("    </text>");

                /* Set sub title text */
                fp.WriteLine(("    <text x=\"" + vp.main_title_viewport.w / 2 + "\" y=\"" + sub_pos + "\"").Replace(',', '.'));
                fp.Write("        font-family=\"Verdana\"");
                fp.Write(" text-anchor=\"middle\"");
                fp.Write((" font-size=\"" + 2 * vp.main_title_viewport.chr_size / 3 + "\"").Replace(',', '.'));
                fp.Write(" fill=\"black\"");
                fp.WriteLine(">");
                fp.WriteLine("      " + sub_title_text);
                fp.WriteLine("    </text>");

                /* Close the view port */
                fp.WriteLine("  </svg>");
            }
        }// SvgCreateMainTitle

        /// <summary>
        /// Create SVG plot x axis title.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="title_text">Text for x axis title</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgCreateXAxisTitle(StreamWriter fp, string title_text, Svg_viewports vp) 
        {
            /* Check the input */
            if(vp.xaxis_title_viewport.iss !=0)
            {
                if(fp == null) throw new TPCException("Filestream was closed too early");
                
                /* Create a new viewport for x axis title */
                fp.WriteLine("\n  <!-- X axis title viewport -->");
                fp.WriteLine("  <svg x=\"" + vp.xaxis_title_viewport.x + "px\" y=\"" + vp.xaxis_title_viewport.y + "px\" width=\"" + vp.xaxis_title_viewport.w + "px\" height=\"" + vp.xaxis_title_viewport.h + "\"");
                fp.WriteLine("      viewBox=\"0 0 " + vp.xaxis_title_viewport.w +" " + vp.xaxis_title_viewport.h +"\"");
                fp.Write("      preserveAspectRatio=\"xMidYMid meet\"");
                fp.WriteLine(">");
                
                /* Set the text contents */
                fp.WriteLine(("    <text x=\"" + vp.xaxis_title_viewport.w / 2 + "\" y=\"" + 0.75 * (double)vp.xaxis_title_viewport.h + "\"").Replace(',', '.'));
                fp.Write("        font-family=\"Verdana\""); 
                fp.Write(" text-anchor=\"middle\""); 
                fp.Write(" font-size=\"" + vp.xaxis_title_viewport.chr_size + "\""); 
                fp.Write(" fill=\"black\""); 
                fp.WriteLine(">");
                fp.WriteLine("      " + title_text);
                fp.WriteLine("    </text>");
                
                /* Close the view port */
                fp.WriteLine("  </svg>");
            }
        }
        
        /// <summary>
        /// Create SVG plot y axis title
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="title_text">Text for y axis title</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgCreateYAxisTitle(StreamWriter fp, string title_text, Svg_viewports vp) 
        {
            double xpos, ypos;

            /* Check the input */
            if(vp.yaxis_title_viewport.iss !=0)
            {
                if(fp == null) throw new TPCException("Filestream was closed too early");
                
                /* Create a new viewport for y axis title */
                fp.WriteLine("\n  <!-- Y axis title viewport -->");
                fp.WriteLine("  <svg x=\"" + vp.yaxis_title_viewport.x + "px\" y=\"" + vp.yaxis_title_viewport.y + "px\" width=\"" + vp.yaxis_title_viewport.w + "px\" height=\"" + vp.yaxis_title_viewport.h + "\"");
                fp.WriteLine("      viewBox=\"0 0 " + vp.yaxis_title_viewport.w + " " + vp.yaxis_title_viewport.h + "\"");
                fp.Write("      preserveAspectRatio=\"xMidYMid meet\"");
                fp.WriteLine(">");
                
                /* Set the text contents */
                xpos=0.75*(double)vp.yaxis_title_viewport.w;
                ypos=0.5*(double)vp.yaxis_title_viewport.h;
                fp.WriteLine(("    <text x=\"" + xpos + "\" y=\"" + ypos + "\"").Replace(',', '.'));
                fp.Write("        font-family=\"Verdana\""); 
                fp.Write(" text-anchor=\"middle\""); 
                fp.Write(" font-size=\"" + vp.yaxis_title_viewport.chr_size + "\""); 
                fp.Write(" fill=\"black\"");
                fp.Write(" transform=\"rotate(270," + xpos.ToString().Replace(',', '.') + "," + ypos.ToString().Replace(',', '.') + ")\""); 
                fp.WriteLine(">");
                fp.WriteLine("      " + title_text); 
                fp.WriteLine("    </text>");
                
                /* Close the view port */
                fp.WriteLine("  </svg>");
            }
        }// SvgCreateYAxisTitle

        /// <summary>
        /// Start plot area viewport.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgStartPlotViewport(StreamWriter fp, Svg_viewports vp) 
        {
            /* Check the input */
            if(fp == null) throw new TPCException("Filestream was closed too early");

            /* Create a new viewport for plot area */
            fp.WriteLine("\n  <!-- Plot area viewport -->");
            fp.Write("  <svg");
            fp.WriteLine(" x=\"" + vp.plot_area_viewport.x + "px\" y=\"" + vp.plot_area_viewport.y + "px\" width=\"" + vp.plot_area_viewport.w + "px\" height=\"" + vp.plot_area_viewport.h + "\""); 
            fp.WriteLine("      viewBox=\"0 0 " + vp.plot_area_viewport.w + " " + vp.plot_area_viewport.h + "\"");
            fp.Write("      preserveAspectRatio=\"xMidYMid meet\"");
            fp.WriteLine(">");
            
            /* Create symbols for later use */
            SvgDefineSymbols(fp);
        }
        
        /// <summary>
        /// End plot viewport.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgEndPlotViewport(StreamWriter fp, Svg_viewports vp) 
        {
            /* Check the input */
            if(fp == null) throw new TPCException("Filestream was closed too early");

            /* Show the curve name when mouse-over event */
            fp.Write("\n<text x=\"" + (0.5 * vp.plot_area_viewport.w).ToString().Replace(',', '.') + "px\" y=\"" + vp.coordinate_area_viewport.chr_size + "px\" font-size=\"" + (0.8 * vp.coordinate_area_viewport.chr_size).ToString().Replace(',', '.') + "px\" fill=\"" + "black" + "\">");
            fp.Write("<tref id=\"title\" xlink:href=\"#\">");
            fp.WriteLine("</tref></text>");

            /* Write the end tag */
            fp.WriteLine("  </svg>");
        }
        
        /// <summary>
        /// Start coordinate area viewport.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgStartCoordinateViewport(StreamWriter fp, Svg_viewports vp) 
        {
            /* Check the input */
            if(fp == null) throw new TPCException("Filestream was closed too early");

            /* Create a new viewport for plot area */
            fp.WriteLine("\n  <!-- Coordinate area viewport -->");
            fp.WriteLine("  <svg x=\"" + vp.coordinate_area_viewport.x + "px\" y=\"" + vp.coordinate_area_viewport.y + "px\" width=\"" + vp.coordinate_area_viewport.w + "px\" height=\"" + vp.coordinate_area_viewport.h + "\"");
            fp.WriteLine("      viewBox=\"0 0 " + vp.coordinate_area_viewport.w + " " + vp.coordinate_area_viewport.h + "\"");
            fp.WriteLine("      preserveAspectRatio=\"xMidYMid meet\">");
        }
       
        /// <summary>
        /// End coordinate area viewport.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        private void SvgEndCoordinateViewport(StreamWriter fp) 
        {
            /* Check the input */
            if (fp == null) throw new TPCException("Filestream was closed too early");
            fp.WriteLine("  </svg>\n");
        }
        
        /// <summary>
        /// Define the curve symbols for SVG graphics file.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        private void SvgDefineSymbols(StreamWriter fp)
        {
            fp.WriteLine("  <defs>"); 
            fp.WriteLine("    <symbol id=\"sym-rect\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <rect x=\"10\" y=\"10\" width=\"100\" height=\"100\" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-circ\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <circle cx=\"60\" cy=\"60\" r=\"50\" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-uptr\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <polygon points=\" 10 17, 110 17, 60 103 \" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-dotr\" viewBox=\"0 -0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <polygon points=\" 10 103, 110 103, 60 17 \" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-letr\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <polygon points=\" 103 10, 103 110, -103 60 \" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-ritr\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <polygon points=\" 17 10, 17 110, 103 60 \" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("    <symbol id=\"sym-diam\" viewBox=\"0 0 120 120\" preserveAspectRatio=\"xMinYMin meet\" >"); 
            fp.WriteLine("      <polygon points=\" 60 10, 110 60, 60 110, 10 60 \" />"); 
            fp.WriteLine("    </symbol>"); 

            fp.WriteLine("  </defs>");
        }// SvgDefineSymbols

        /// <summary>
        /// Draw the axes into SVG plot coordinate area.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgWriteAxes(StreamWriter fp, Svg_viewports vp) 
        {
            int linew=2, coordw=1;
            double f;

            /* Check the input */
            if (fp == null) throw new TPCException("Filestream was closed too early");
            
            if(vp.x.min >= vp.x.max || vp.y.min >= vp.y.max) 
                throw new TPCException("Invalid plot range");
            
            /* Draw the lines around the plot */
            fp.WriteLine("    <polyline fill=\"none\" stroke=\"" + "black" + "\" stroke-width=\"" + linew + "\"");

            fp.WriteLine("      points=\"" + (linew / 2).ToString().Replace(',', '.') + "," + (linew / 2).ToString().Replace(',', '.') + " " + (linew / 2).ToString().Replace(',', '.') + "," + (vp.coordinate_area_viewport.h - linew / 2).ToString().Replace(',', '.') + " " + (vp.coordinate_area_viewport.w - linew / 2).ToString().Replace(',', '.') + "," + (vp.coordinate_area_viewport.h - linew / 2).ToString().Replace(',', '.') + " " + (vp.coordinate_area_viewport.w - linew / 2).ToString().Replace(',', '.') + "," + (linew / 2).ToString().Replace(',', '.') + " " + (linew / 2).ToString().Replace(',', '.') + "," + (linew / 2).ToString().Replace(',', '.') + "\" />");
            
            /* Set the upper margins for both axes */
            vp.x.upper_margin=(int)(0.08*(double)vp.coordinate_area_viewport.w);
            vp.y.upper_margin=(int)(0.04*(double)vp.coordinate_area_viewport.h);

            /* Calculate the scale factors */
            f=vp.coordinate_area_viewport.w-vp.x.upper_margin;
            vp.x.scale=f/(vp.x.max-vp.x.min);
            
            f=vp.coordinate_area_viewport.h-vp.y.upper_margin;
            vp.y.scale=f/(vp.y.max-vp.y.min);
            
            /* Calculate the origo in plot coordinates */
            vp.x.origo=-vp.x.scale*vp.x.min;
            vp.y.origo=-vp.y.scale*vp.y.min;

            /* Draw the x=0 line, if necessary */
            if(vp.x.origo>0 && vp.x.origo<vp.coordinate_area_viewport.w/*-vp.x.upper_margin*/) {
                fp.WriteLine("    <line fill=\"none\" stroke=\"" + "black" + "\" stroke-width=\"" + coordw + "\"");
                fp.WriteLine("      x1=\"" + vp.x.origo.ToString().Replace(',', '.') + "\" x2=\"" + vp.x.origo.ToString().Replace(',', '.') + "\" y1=\"0\" y2=\"" + vp.coordinate_area_viewport.h + "\" />"); 
            }

            /* Draw the y=0 line, if necessary */
            if(vp.y.origo>0 && vp.y.origo<vp.coordinate_area_viewport.h /*-vp.y.upper_margin*/) {
                fp.WriteLine("    <line fill=\"none\" stroke=\"" + "black" + "\" stroke-width=\"" + coordw + "\"");
                fp.WriteLine("      x1=\"0\" x2=\"" + vp.coordinate_area_viewport.w + "\" y1=\"" + (vp.coordinate_area_viewport.h - vp.y.origo).ToString().Replace(',', '.') + "\" y2=\"" + (vp.coordinate_area_viewport.h - vp.y.origo).ToString().Replace(',', '.') + "\" />");
            }
        }// SvgWriteAxes
       
        /// <summary>
        /// Draw sample curve in a SVG file.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        /// <param name="plot_type">Plot type: PlotType.LINE, PlotType.SYMBOLS or PlotType.LINE_AND_SYMBOLS</param>
        /// <param name="tac_id">Unique ID for the curve</param>
        /// <param name="tac_title">Title of the curve, which may be shown in the graph</param>
        /// <param name="x">polyline data x array (original quantities) </param>
        /// <param name="y">polyline data y array (original quantities)</param>
        /// <param name="color">SVG color name as a string, e.g. aqua,black,blue,fuchsia,gray,
        ///                     green,lime,maroon,navy,olive,purple,red,silver,teal,yellow.
        ///                     Note that this string is not tested </param>
        /// <param name="symbol_type">Symbol type: RECTANGLE,CIRCLE,UPTRIANGLE,DOWNTRIANGLE,DIAMOND,
        ///                           LEFTTRIANGLE, RIGHTTRIANGLE</param>
        /// <param name="stroke_width">Plot width value, using default value if 0 is given</param>                          
        private void SvgWriteTac(StreamWriter fp, Svg_viewports vp, PlotType plot_type, string tac_id, string tac_title, double[] x, double[] y, Color c, SvgSymbolType symbol_type, double stroke_width)
        {
            int i, j;
            double px, py, trgsize=14.0, circsize=12.0;
            double size = stroke_width;
            
            string color = (c.Name).ToLower();

            /* Check the input */
            if (fp == null) throw new TPCException("Filestream was closed too early");
            if (x.Length >= 1)
            {
                if (color == null || color.ToString().Length < 2) throw new TPCException("Invalid color");
                  
                /* Initiate the curve object group */
                fp.WriteLine("\n    <!-- " + tac_id + " : " + tac_title + " -->");
                fp.Write("    <g stroke=\"" + color + "\"");
                fp.Write((" stroke-width=\"" + 0.25 * size + "\"").Replace(',', '.'));
                fp.Write(" fill=\"" + color + "\"");
                fp.Write(" fill-opacity=\"0.67\"");
                fp.Write(" xlink:title=\"" + tac_title + "\"");
                fp.WriteLine(">");

                /* Set the curve title */
                fp.Write("      <title id=\"" + tac_id + "\">");
                fp.Write(tac_title);
                fp.WriteLine("</title>");
                fp.Write("      <set xlink:href=\"#title\" attributeType=\"XML\" attributeName=\"xlink:href\" begin=\"mouseover\" end=\"mouseout\"");
                fp.WriteLine(" to=\"#" + tac_id + "\" />");

                /* Plot the polyline, if required */

                if (plot_type.Equals(PlotType.LINE_AND_SYMBOLS) || plot_type.Equals(PlotType.LINE))
                {
                    fp.WriteLine("    <polyline fill=\"none\"");
                    fp.Write("      points=\"");

                    /* Write polyline coordinates */
                    for (i = j = 0; i < x.Length; i++, j++)
                    {
                        if (j == 5) j = 0; /* line end */
                        px = vp.x.origo + x[i] * vp.x.scale;
                        py = vp.coordinate_area_viewport.h - (vp.y.origo + vp.y.scale * y[i]);
                        if (j == 0)
                        { /* line start */
                            fp.Write("\n       ");
                        }
                        else fp.Write(" ");
                        fp.Write("" + px.ToString("F3").Replace(',', '.') + "," + py.ToString("F3").Replace(',', '.') + "");
                    }
                    /* Close polyline */
                    fp.WriteLine("\" />");
                }

                /* Plot the symbols, if required */
                if (plot_type.Equals(PlotType.LINE_AND_SYMBOLS) || plot_type.Equals(PlotType.SYMBOLS))
                {
                    for (i = 0; i < x.Length; i++)
                    {
                        px = vp.x.origo + x[i] * vp.x.scale;
                        py = vp.coordinate_area_viewport.h - (vp.y.origo + vp.y.scale * y[i]);
                        fp.Write("    <use ");
                        switch (symbol_type)
                        {
                            case SvgSymbolType.RECTANGLE:
                                fp.Write("xlink:href=\"#sym-rect\" x=\"" + (px - 0.5 * size).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * size).ToString("F3").Replace(',', '.') + "\" width=\"" + size.ToString("F3").Replace(',', '.') + "\" heigh=\"" + size.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.UPTRIANGLE:
                                fp.Write("xlink:href=\"#sym-uptr\" x=\"" + (px - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" width=\"" + trgsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + trgsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.DOWNTRIANGLE:
                                fp.Write("xlink:href=\"#sym-dotr\" x=\"" + (px - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" width=\"" + trgsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + trgsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.DIAMOND:
                                fp.Write("xlink:href=\"#sym-diam\" x=\"" + (px - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" width=\"" + trgsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + trgsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.LEFTTRIANGLE:
                                fp.Write("xlink:href=\"#sym-letr\" x=\"" + (px - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" width=\"" + trgsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + trgsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.RIGHTTRIANGLE:
                                fp.Write("xlink:href=\"#sym-ritr\" x=\"" + (px - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * trgsize).ToString("F3").Replace(',', '.') + "\" width=\"" + trgsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + trgsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                            case SvgSymbolType.CIRCLE:
                            default:
                                fp.Write("xlink:href=\"#sym-circ\" x=\"" + (px - 0.5 * circsize).ToString("F3").Replace(',', '.') + "\" y=\"" + (py - 0.5 * circsize).ToString("F3").Replace(',', '.') + "\" width=\"" + circsize.ToString("F3").Replace(',', '.') + "\" heigh=\"" + circsize.ToString("F3").Replace(',', '.') + "\"");
                                break;
                        }
                        fp.WriteLine(" />");
                    }
                }
                /* Close the curve object group */
                fp.WriteLine("    </g>");
            }
        }// SvgWriteTac

        /// <summary>
        /// Create SVG plot x axis ticks.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgWriteXTicks(StreamWriter fp, Svg_viewports vp)
        {
            int ti, prec;
            double pos, ypos, yheight, f;

            /* Check the input */
            if(!(vp.x.tick_nr < 1 || vp.plot_area_viewport.h == vp.coordinate_area_viewport.h))
            {
                if (fp == null) throw new TPCException("Filestream was closed too early");

                yheight=vp.plot_area_viewport.h-vp.coordinate_area_viewport.h;
                ypos=vp.coordinate_area_viewport.h;

                fp.WriteLine("\n    <!-- X axis ticks inside plot area -->");
                
                /* Write ticks */
                f = 1.0 / Math.Pow(10, vp.x.tickscale);
                for(ti = 0; ti < vp.x.tick_nr; ti++) {
                    pos = vp.x.origo + vp.x.scale * vp.x.tick[ti];
                    pos += vp.plot_area_viewport.w-vp.coordinate_area_viewport.w;
                    
                    fp.Write("    <line fill=\"none\" stroke=\"" + "black" + "\" stroke-width=\"" + 2 + "\"");
                    fp.WriteLine(" x1=\"" + pos.ToString("F3").Replace(',', '.') + "\" x2=\"" + pos.ToString("F3").Replace(',', '.') + "\" y1=\"" + ypos.ToString("F3").Replace(',', '.') + "\" y2=\"" + (ypos + yheight / 8).ToString("F3").Replace(',', '.') + "\" />");

                    fp.Write("    <text x=\"" + pos.ToString("F3").Replace(',', '.') + "\" y=\"" + (ypos + 0.92 * (double)yheight).ToString("F3").Replace(',', '.') + "\""); 
                    fp.Write(" font-family=\"Verdana\""); 
                    fp.Write(" text-anchor=\"middle\""); 
                    fp.Write(" font-size=\"" + vp.coordinate_area_viewport.chr_size + "\""); 
                    fp.Write(" fill=\"black\""); 
                    fp.Write(">"); 
                    prec = vp.x.tick_decimals - 1 - (int)vp.x.tickscale;
                    if(prec < 0) prec = 0;
                    if(vp.x.tickscale < -2 || vp.x.tickscale > 3)
                        //fp.WriteLine("%."+(vp.x.tick_decimals-1)+"E", vp.x.tick[ti]);
                        fp.Write(vp.x.tick[ti].ToString("E" + (vp.x.tick_decimals - 1)));
                    else
                        //if (vp.x.tickscale <= 0) fp.WriteLine("%.*f", prec, vp.x.tick[ti]);
                        if (vp.x.tickscale <= 0) fp.Write(vp.x.tick[ti].ToString("F" + prec));
                        //else fp.WriteLine("%.*f", prec, vp.x.tick[ti]);
                        else fp.Write(vp.x.tick[ti].ToString("F" + prec));
                    
                    fp.WriteLine("</text>");
                }
            }
        }// SvgWriteXTicks
            
        /// <summary>
        /// Create SVG plot y axis ticks.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        /// <param name="vp">Struct containing the viewport sizes</param>
        private void SvgWriteYTicks(StreamWriter fp, Svg_viewports vp)
        {
            int ti, prec;
            double pos, xpos, xwidth, f;

            /* Check the input */
            if(!(vp.y.tick_nr < 1 || vp.plot_area_viewport.w == vp.coordinate_area_viewport.w))
            {
                if (fp == null) throw new TPCException("Filestream was closed too early");

                xwidth = vp.plot_area_viewport.w-vp.coordinate_area_viewport.w;
                xpos = xwidth;

                fp.WriteLine("\n    <!-- Y axis ticks inside plot area -->");
                
                /* Write ticks */
                f = 1.0 / Math.Pow(10, vp.y.tickscale);
                
                for(ti = 0; ti < vp.y.tick_nr; ti++) {
                    pos = vp.coordinate_area_viewport.h - (vp.y.origo + vp.y.scale * vp.y.tick[ti]);
                    
                    /* tick line */
                    fp.Write("    <line fill=\"none\" stroke=\"" + "black" + "\" stroke-width=\"" + 2 + "\"");
                    fp.WriteLine(" x1=\"" + xwidth.ToString("F3").Replace(',', '.') + "\" x2=\"" + (xwidth - (double)vp.coordinate_area_viewport.chr_size / 8.0).ToString("F3").Replace(',', '.') + "\" y1=\"" + pos.ToString("F3").Replace(',', '.') + "\" y2=\"" + pos.ToString("F3").Replace(',', '.') + "\" />"); 
                    
                    /* tick label */
                    fp.Write("    <text x=\"" + (0.92 * xwidth).ToString("F3").Replace(',', '.') + "\" y=\"" + (pos + 0.4 * (double)vp.coordinate_area_viewport.chr_size).ToString("F3").Replace(',', '.') + "\"");
                    fp.Write(" font-family=\"Verdana\""); 
                    fp.Write(" text-anchor=\"end\""); 
                    fp.Write(" font-size=\"" + vp.coordinate_area_viewport.chr_size + "\""); 
                    fp.Write(" fill=\"black\""); 
                    fp.Write(">"); 
                    prec = vp.y.tick_decimals - 1 - (int)vp.y.tickscale; 
                    if (prec < 0) prec = 0;
                    if (vp.y.tickscale < -2 || vp.y.tickscale > 3)
                        //fp.WriteLine("%.*E", vp.y.tick_decimals-1, vp.y.tick[ti]);
                        fp.Write(vp.y.tick[ti].ToString("E" + (vp.y.tick_decimals-1)));
                    else 
                        //if (vp.y.tickscale <= 0) fp.WriteLine("%.*f", prec, vp.y.tick[ti]);
                        if (vp.y.tickscale <= 0) fp.Write(vp.y.tick[ti].ToString("F" + prec));
                        //else fp.WriteLine("%.*f", prec, vp.y.tick[ti]);
                        else fp.Write(vp.y.tick[ti].ToString("F" + prec));
                    
                    fp.WriteLine("</text>");
                }
            }
        }// SvgWriteYTicks

        /// <summary>
        /// Close SVG graphics file.
        /// </summary>
        /// <param name="fp">StreamWriter name for SVG graphics</param>
        private void SvgClose(StreamWriter fp)
        {
            fp.WriteLine("</svg>");
            //fclose(fp);
        }
    }
}
