﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{

    /// <summary>
    /// TRTM model
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class TRTMModel : Model
    {
        /// <summary>
        /// Vectors for storing time intervals and reference values
        /// </summary>
        private Vector times;
        private Vector refintegral;
        /// <summary>
        /// Reference values
        /// </summary>
        private Vector refvals;

        /// <summary>
        /// Number of samples simulated
        /// </summary>
        public override int Samples
        {
            get { return times.Length; }
        }


        /// <summary>
        /// Return info of Model
        /// </summary>
        /// <returns>Info of model</returns>
        protected override ModelInfo GetInfo()
        {
            ModelInfo info = new ModelInfo("TRTMModel", "This is TRTMModel description...", "Reference values");

            List<ModelParameter> parameters = new List<ModelParameter>();
            parameters.Add(new ModelParameter("r1", 0, 10, 5));
            parameters.Add(new ModelParameter("k2", 0, 10, 5));
            parameters.Add(new ModelParameter("k3", 0, 10, 5));
            info.Parameters = parameters.ToArray();
            info.ClassType = this.GetType();

            return info;
        }


        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Reference value vector</param>
        public override void Create(Vector t, params Vector[] r)
        {
            if (r[0].Length != t.Length) throw new ModelException("Different number of sample times and values");

            model_times = t;

            Info = GetInfo();
            times = HalfTimeDifs(t);
            if (r.Length > 0) refvals = r[0];
            refintegral = CalcRefIntegral(refvals, times);
        }

        /// <summary>
        /// Constructor
        /// </summary>
        public TRTMModel() { Info = GetInfo(); }
        
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="t">Vector of time values</param>
        /// <param name="r">Reference value vector</param>
        public TRTMModel(Vector t, Vector r) { Create(t, r); }

        /// <summary>
        /// Wrapper function. Overrides Model.Simulate.
        /// </summary>
        /// <param name="par">Model parameters</param>
        /// <returns>Vector of simulated values</returns>
        public override Vector Simulate(params double[] par)
        {
            if (par.Length != 3) throw new ModelException("Wrong number of parameters");
            return Simulate(par[0], par[1], par[2]);
        }

        /// <summary>
        /// Simulates tissue TAC using reference tissue compartment model (transport limited in ref region) and reference region TAC, at reference region TAC times.
        /// The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
        /// </summary>
        /// <param name="r1">Ratio K1/K1'</param>
        /// <param name="k2">Rate constant of the model</param>
        /// <param name="k3">Rate constant of the model</param>
        /// <returns>Vector of simulated values</returns>
        public Vector Simulate(double r1, double k2, double k3)
        {
            Vector simval = new Vector(Samples);

            double k23 = k2 + k3;

            //Cumulative integral value
            double cti = 0.0;

            double last_val = 0.0;
            double cr, cri, dt2;

            for (int i = 0; i < Samples; i++)
            {
                cr = refvals[i];
                cri = refintegral[i];
                dt2 = times[i];

                simval[i] = ((r1 * cr) + (r1 * k3 * cri) - k23 * (dt2 * last_val + cti)) /
                            (1.0 + dt2 * k23);

                cti += dt2 * (last_val + simval[i]);
                last_val = simval[i];
            }

            return simval;
        }
    }
}
