﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Model
{
    /// <summary>
    /// Homogenous Coordinate Transformation Matrices.
    /// </summary>
    public static class Transformation
    {
        /// <summary>
        /// 4x4 Rotation matrix generator.
        /// </summary>
        /// <param name="x_rot">rotation about the x-axis, pitch</param>
        /// <param name="y_rot">rotation about the y-axis, roll</param>
        /// <param name="z_rot">rotation about the z-axis, yaw</param>
        /// <returns></returns>
        public static Matrix Rotation4x4(double x_rot, double y_rot, double z_rot)
        {
            Matrix rot = new Matrix(4, 4);

            double cosx = Math.Cos(x_rot), sinx = Math.Sin(x_rot);
            double cosy = Math.Cos(y_rot), siny = Math.Sin(y_rot);
            double cosz = Math.Cos(z_rot), sinz = Math.Sin(z_rot);
            
            rot[0, 0] = cosz * cosy + sinz * sinx * siny;
            rot[0, 1] = sinz * cosy - cosz * sinx * siny;
            rot[0, 2] = cosx * siny;
            
            rot[1, 0] = -sinz * cosx;
            rot[1, 1] = cosz * cosx;
            rot[1, 2] = sinx;
            
            rot[2, 0] = sinz * sinx * cosy - cosz * siny;
            rot[2, 1] = -cosz * sinx * cosy - sinz * siny;
            rot[2, 2] = cosx * cosy;
            
            rot[3, 3] = 1.0f;
            
            return rot;
        }

        /// <summary>
        /// 4x4 Scale matrix generator
        /// </summary>
        /// <param name="sx">x-scale</param>
        /// <param name="sy">y-scale</param>
        /// <param name="sz">z-scale</param>
        /// <returns></returns>
        public static Matrix Scale4x4(double sx, double sy, double sz)
        {
            Matrix scale = new Matrix(4, 4);
            scale[0, 0] = sx; // x scale
            scale[1, 1] = sy; // y scale
            scale[2, 2] = sz; // z scale 
            scale[3, 3] = 1;
            return scale;
        }

        /// <summary>
        /// 4x4 Translation matrix generator
        /// </summary>
        /// <param name="dx">x shift</param>
        /// <param name="dy">y shift</param>
        /// <param name="dz">z shift</param>
        /// <returns></returns>
        public static Matrix Translation4x4(double dx, double dy, double dz)
        {
            Matrix tr = new Matrix(4, 4);
            tr[0, 3] = dx; 
            tr[1, 3] = dy; 
            tr[2, 3] = dz;
            tr[0, 0] = tr[1, 1] = 1;
            tr[2, 2] = tr[3, 3] = 1;
            return tr;
        }
        /// <summary>
        /// Inverts the first 3x3 block of a matrix.
        /// </summary>
        /// <param name="M">3x3 matrix of bigger</param>
        /// <returns>Matrix of which the first 3x3 block is inverted.</returns>
        public static Matrix Inverse3x3(Matrix M)
        {
            if (M.Columns < 3 || M.Rows < 3)
                throw new TPCException("Matrix is smaller than 3x3.");

            // 1. Find det(M)
            double det = M[0, 0] * (M[1, 1] * M[2, 2] - M[1, 2] * M[2, 1])
                       - M[0, 1] * (M[1, 0] * M[2, 2] - M[1, 2] * M[2, 0])
                       + M[0, 2] * (M[1, 0] * M[2, 1] - M[1, 1] * M[2, 0]);

            // 2. Transpose M to obtain M^T
            Matrix t = M.Transpose();

            // 3. Find the matrix of cofactors by first calculating the matrix of minors
            Matrix m = new Matrix(3);

            m[0, 0] = t[1, 1] * t[2, 2] - t[1, 2] * t[2, 1];
            m[0, 1] = t[1, 0] * t[2, 2] - t[1, 2] * t[2, 0];
            m[0, 2] = t[1, 0] * t[2, 1] - t[1, 1] * t[2, 0];

            m[1, 0] = t[0, 1] * t[2, 2] - t[0, 2] * t[2, 1];
            m[1, 1] = t[0, 0] * t[2, 2] - t[0, 2] * t[2, 0];
            m[1, 2] = t[0, 0] * t[2, 1] - t[0, 1] * t[2, 0];

            m[2, 0] = t[0, 1] * t[1, 2] - t[0, 2] * t[1, 1];
            m[2, 1] = t[0, 0] * t[1, 2] - t[0, 2] * t[1, 0];
            m[2, 2] = t[0, 0] * t[1, 1] - t[0, 1] * t[1, 0];

            // find the adjacent
            m[0, 1] = -m[0, 1];
            m[1, 0] = -m[1, 0];
            m[2, 1] = -m[2, 1];
            m[1, 2] = -m[1, 2];

            // 4. Find the inverse from the rule: M^-1 = 1/det(M)*Adj(M)
            Matrix result = 1 / det * m;

            return result;
        }

        private const double MATRIX_INVERSE_EPSILON = 0.0000001;

        /// <summary>
        /// Inverts the first 4x4 block of a matrix.
        /// Adapted from Id Software's Doom 3 GPL open source code
        /// </summary>
        /// <remarks>104 (84+4+16) multiplications and 1 division</remarks>
        /// <param name="mat">4x4 matrix or bigger</param>
        /// <returns>Matrix of which the first 4x4 block is inverted.</returns>
        public static Matrix Inverse4x4(Matrix mat)
        {
            double det, invDet;

            // 2x2 sub-determinants required to calculate 4x4 determinant
            double det2_01_01 = mat[0, 0] * mat[1, 1] - mat[0, 1] * mat[1, 0];
            double det2_01_02 = mat[0, 0] * mat[1, 2] - mat[0, 2] * mat[1, 0];
            double det2_01_03 = mat[0, 0] * mat[1, 3] - mat[0, 3] * mat[1, 0];
            double det2_01_12 = mat[0, 1] * mat[1, 2] - mat[0, 2] * mat[1, 1];
            double det2_01_13 = mat[0, 1] * mat[1, 3] - mat[0, 3] * mat[1, 1];
            double det2_01_23 = mat[0, 2] * mat[1, 3] - mat[0, 3] * mat[1, 2];

            // 3x3 sub-determinants required to calculate 4x4 determinant
            double det3_201_012 = mat[2, 0] * det2_01_12 - mat[2, 1] * det2_01_02 + mat[2, 2] * det2_01_01;
            double det3_201_013 = mat[2, 0] * det2_01_13 - mat[2, 1] * det2_01_03 + mat[2, 3] * det2_01_01;
            double det3_201_023 = mat[2, 0] * det2_01_23 - mat[2, 2] * det2_01_03 + mat[2, 3] * det2_01_02;
            double det3_201_123 = mat[2, 1] * det2_01_23 - mat[2, 2] * det2_01_13 + mat[2, 3] * det2_01_12;

            det = (-det3_201_123 * mat[3, 0] + det3_201_023 * mat[3, 1] - det3_201_013 * mat[3, 2] + det3_201_012 * mat[3, 3]);

            if (Math.Abs(det) < MATRIX_INVERSE_EPSILON)
            {
                throw new TPCException("determinant is zero");
            }

            invDet = 1.0f / det;

            // remaining 2x2 sub-determinants
            double det2_03_01 = mat[0, 0] * mat[3, 1] - mat[0, 1] * mat[3, 0];
            double det2_03_02 = mat[0, 0] * mat[3, 2] - mat[0, 2] * mat[3, 0];
            double det2_03_03 = mat[0, 0] * mat[3, 3] - mat[0, 3] * mat[3, 0];
            double det2_03_12 = mat[0, 1] * mat[3, 2] - mat[0, 2] * mat[3, 1];
            double det2_03_13 = mat[0, 1] * mat[3, 3] - mat[0, 3] * mat[3, 1];
            double det2_03_23 = mat[0, 2] * mat[3, 3] - mat[0, 3] * mat[3, 2];

            double det2_13_01 = mat[1, 0] * mat[3, 1] - mat[1, 1] * mat[3, 0];
            double det2_13_02 = mat[1, 0] * mat[3, 2] - mat[1, 2] * mat[3, 0];
            double det2_13_03 = mat[1, 0] * mat[3, 3] - mat[1, 3] * mat[3, 0];
            double det2_13_12 = mat[1, 1] * mat[3, 2] - mat[1, 2] * mat[3, 1];
            double det2_13_13 = mat[1, 1] * mat[3, 3] - mat[1, 3] * mat[3, 1];
            double det2_13_23 = mat[1, 2] * mat[3, 3] - mat[1, 3] * mat[3, 2];

            // remaining 3x3 sub-determinants
            double det3_203_012 = mat[2, 0] * det2_03_12 - mat[2, 1] * det2_03_02 + mat[2, 2] * det2_03_01;
            double det3_203_013 = mat[2, 0] * det2_03_13 - mat[2, 1] * det2_03_03 + mat[2, 3] * det2_03_01;
            double det3_203_023 = mat[2, 0] * det2_03_23 - mat[2, 2] * det2_03_03 + mat[2, 3] * det2_03_02;
            double det3_203_123 = mat[2, 1] * det2_03_23 - mat[2, 2] * det2_03_13 + mat[2, 3] * det2_03_12;

            double det3_213_012 = mat[2, 0] * det2_13_12 - mat[2, 1] * det2_13_02 + mat[2, 2] * det2_13_01;
            double det3_213_013 = mat[2, 0] * det2_13_13 - mat[2, 1] * det2_13_03 + mat[2, 3] * det2_13_01;
            double det3_213_023 = mat[2, 0] * det2_13_23 - mat[2, 2] * det2_13_03 + mat[2, 3] * det2_13_02;
            double det3_213_123 = mat[2, 1] * det2_13_23 - mat[2, 2] * det2_13_13 + mat[2, 3] * det2_13_12;

            double det3_301_012 = mat[3, 0] * det2_01_12 - mat[3, 1] * det2_01_02 + mat[3, 2] * det2_01_01;
            double det3_301_013 = mat[3, 0] * det2_01_13 - mat[3, 1] * det2_01_03 + mat[3, 3] * det2_01_01;
            double det3_301_023 = mat[3, 0] * det2_01_23 - mat[3, 2] * det2_01_03 + mat[3, 3] * det2_01_02;
            double det3_301_123 = mat[3, 1] * det2_01_23 - mat[3, 2] * det2_01_13 + mat[3, 3] * det2_01_12;

            mat[0, 0] = -det3_213_123 * invDet;
            mat[1, 0] = +det3_213_023 * invDet;
            mat[2, 0] = -det3_213_013 * invDet;
            mat[3, 0] = +det3_213_012 * invDet;

            mat[0, 1] = +det3_203_123 * invDet;
            mat[1, 1] = -det3_203_023 * invDet;
            mat[2, 1] = +det3_203_013 * invDet;
            mat[3, 1] = -det3_203_012 * invDet;

            mat[0, 2] = +det3_301_123 * invDet;
            mat[1, 2] = -det3_301_023 * invDet;
            mat[2, 2] = +det3_301_013 * invDet;
            mat[3, 2] = -det3_301_012 * invDet;

            mat[0, 3] = -det3_201_123 * invDet;
            mat[1, 3] = +det3_201_023 * invDet;
            mat[2, 3] = -det3_201_013 * invDet;
            mat[3, 3] = +det3_201_012 * invDet;

            return mat;
        }
    }
}
