﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Model
{

    /// <summary>
    /// Input estimated from waterliver
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class WaterLiverInputModel : Model
    {
        private const double kg = 0.5/60.0;
        private double vd = 70.0d;

        private double k2;
        private Vector simulated;

        public WaterLiverInputModel() { Info = GetInfo(); }

        public override void Create(Vector times, params Vector[] Reference)
        {
            model_times = times;
            //this.reference = Reference;
            simulated = new Vector(times.Length);
        }

        /// <summary>
        /// Number of samples simulated
        /// </summary>
        public override int Samples
        {
            get { return model_times.Length; }
        }

        /// <summary>
        /// Return info of Model
        /// </summary>
        /// <returns>Info of model</returns>
        protected override ModelInfo GetInfo()
        {
            ModelInfo info = new ModelInfo("InputFromWaterLiver", "This model is estimating water liver input");

            List<ModelParameter> Parameters = new List<ModelParameter>();
            Parameters.Add(new ModelParameter("A", 1, 1200000, 500000));
            Parameters.Add(new ModelParameter("kA", 0.2, 20, 5.0));
            Parameters.Add(new ModelParameter("t1", -10, 40.0, 0.0));
            Parameters.Add(new ModelParameter("t2", 15, 70, 27));

            Parameters.Add(new ModelParameter("fa", 1, 40, 15));
            Parameters.Add(new ModelParameter("fp", 1, 200, 100.0));
            Parameters.Add(new ModelParameter("v0", 0, 1, 0.05));
            info.Parameters = Parameters.ToArray();
            info.ClassType = this.GetType();

            return info;
        }

        public override Vector Simulate(params double[] p)
        {
            double fa, fp;

            // Get parameters 
            double A = p[0];
            double ka = p[1] / 60;
            double t1 = p[2];
            double t2 = p[3] + t1;
            

            fa = p[4] / 60.0;
            fp = p[5] / 60.0;


            //double v0 = p[6]; OLD
            double v0 = p[6];
            double f = fa + fp;

            double ra = fa / (fa + fp);
            double rp = fp / (fa + fp);
            //double r = fa / fp;
            
            k2 = (fa + fp) / vd;

            int num_times = model_times.Length;
            double x = 0;

            for (int i = 0; i < num_times; i++)
            {
                x = model_times[i];
                double tis_ar = M_tissue_model1_av(x, A, ka, t1, t2, fa, fp); // päämodel
                //tis_ar = M_tissue_model1_ar(x, A, ka, t1, t2, f);
                double inp_ar = M_input_model_av(x, A, ka, t1, t2, ra, rp);
                double val = (1.0 - v0) * tis_ar + v0 * inp_ar;

                simulated[i] = val; 
            }

            return simulated;
        }


        /// <summary>
        /// Liver-perfusion tissue response 
        /// </summary>
        double M_tissue_model1_av(double x, double A, double ka, double t1, double t2, double fa, double fp)
        {
            //double fa, fp;
            double tis_ar, tis_pv;

            //fa = r / (1.0 + r) * f; // Works with ratio fa/fp
            //fp = 1.0 / (1.0 + r) * f; // Works with ratio fa/fp

            tis_ar = M_tissue_model1_ar(x, A, ka, t1, t2, fa, fa+fp);
            tis_pv = M_tissue_model1_pv(x, A, ka, t1, t2, fp, fa+fp);

            return tis_ar + tis_pv;
        }

        protected double M_tissue_model1_ar(double x, double A, double ka, double t1, double t2, double fa, double f)
        {

            double y, t;

            //k2 = f / vd;
            //k2 = kg;

            t = x;

            if (x < t1)
            {
                y = 0.0;
            }
            else if (x <= t2)
            {
                y = (1.0 - Math.Exp(k2 * (t1 - t))) / k2 + (Math.Exp(k2 * (t1 - t)) - Math.Exp(ka * (t1 - t))) / (k2 - ka);
            }
            else
            {
                y = (Math.Exp(k2 * (t2 - t)) - Math.Exp(k2 * (t1 - t)) + Math.Exp(ka * (t1 - t2)) - Math.Exp(k2 * (t2 - t) + ka * (t1 - t2))) / k2
                  + (-2.0 * Math.Exp(ka * (t1 - t)) + 2.0 * Math.Exp(k2 * (t2 - t) + ka * (t1 - t2)) + Math.Exp(ka * (t2 - t)) - Math.Exp(k2 * (t2 - t)) + Math.Exp(k2 * (t1 - t)) - Math.Exp(k2 * (t2 - t) + ka * (t1 - t2))) / (k2 - ka);
            }

            return (y * A * fa);
        }


        double M_tissue_model1_pv(double x, double A, double ka, double t1, double t2, double fp, double f)
        {

            double y1, y2, y3, y4, y5, y6, y7, y8, ya, yb, y, t;

            //k2 = f / vd;
            t = x;

            if (x < t1)
            {
                y = 0.0;
            }
            else if (x <= t2)
            {
                y1 = (1.0 - Math.Exp(k2 * (t1 - t))) / k2 + (Math.Exp(kg * (t1 - t)) - Math.Exp(k2 * (t1 - t))) / (kg - k2);
                y2 = (1.0 - Math.Exp(k2 * (t1 - t))) / k2 - (Math.Exp(ka * (t1 - t)) - Math.Exp(k2 * (t1 - t))) / (k2 - ka);
                y = y1 / kg + y2 / (kg - ka);
            }
            else
            {
                y5 = Math.Exp(ka * (t1 - t2)) * (1.0 - Math.Exp(k2 * (t2 - t))) / k2;

                y6 = (Math.Exp(kg * (t2 - t)) - Math.Exp(kg * (t1 - t)) - Math.Exp(k2 * (t2 - t))
                - Math.Exp(ka * (t1 - t2) + kg * (t2 - t))
                + Math.Exp(kg * (t1 - t2) + k2 * (t2 - t))
                + Math.Exp(ka * (t1 - t2) + k2 * (t2 - t))) / (k2 - kg);

                y7 = (Math.Exp(ka * (t2 - t)) - 2.0 * Math.Exp(ka * (t1 - t))
                - Math.Exp(k2 * (t2 - t)) + 2.0 * Math.Exp(ka * (t1 - t2) + k2 * (t2 - t))) / (k2 - ka);

                y8 = (Math.Exp(ka * (t1 - t2) + kg * (t2 - t))
                + Math.Exp(kg * (t1 - t)) - Math.Exp(kg * (t2 - t))
                + Math.Exp(k2 * (t2 - t))
                - Math.Exp(ka * (t1 - t2) + k2 * (t2 - t))
                - Math.Exp(k2 * (t2 - t) + kg * (t1 - t2))) / (k2 - kg);

                ya = (y5 + y6) / kg + (y7 + y8) / (kg - ka);

                y1 = (Math.Exp(k2 * (t2 - t)) - Math.Exp(k2 * (t1 - t))) / k2;
                y2 = (Math.Exp(kg * (t1 - t2) + k2 * (t2 - t)) - Math.Exp(k2 * (t1 - t))) / (kg - k2);

                y3 = (Math.Exp(ka * (t1 - t2) + k2 * (t2 - t)) - Math.Exp(k2 * (t1 - t))) / (k2 - ka);
                y4 = (Math.Exp(kg * (t1 - t2) + k2 * (t2 - t)) - Math.Exp(k2 * (t1 - t))) / (kg - k2);
                yb = (y1 + y2) / kg + (y3 + y4) / (ka - kg);

                y = yb + ya;
            }

            return (y * A * fp * kg);
        }

        /// <summary>
        /// Liver-perfusion: arterial and portal input funciton
        /// </summary>
        protected double M_input_model_av(double x, double A, double ka, double t1, double t2, double ra, double rp)
        {
            double inp_ar, inp_pv, inp;

            inp_ar = CA_Input(x, A, ka, t1, t2);
            inp_pv = M_input_model_pv(x, A, ka, t1, t2);

            inp = ra * inp_ar + rp * inp_pv;

            //inp = (r * inp_ar + inp_pv) / (1.0 + r); // Works with r = fa/fp
            //if (inp > 0) { int g = 9; }
            //inp = (ra * inp_ar) + (rp * inp_pv);

            return inp;

        }

        /********************************************************************
        Input Model Function for FDG, Water
        *************************************************************/
        protected double CA_Input(double x, double A, double ka, double t1, double t2)
        {
            double y;

            if (x < t1) y = 0.0d;
            else if (x <= t2) y = 1.0d - Math.Exp(ka * (t1 - x));
            else y = Math.Exp(ka * (t1 - t2)) - 2.0d * Math.Exp(ka * (t1 - x)) + Math.Exp(ka * (t2 - x));

            return (y * A );
        }



        protected double M_input_model_pv(double x, double A, double ka, double t1, double t2)
        {
            double y1, y2, y, xx;

            xx = x;

            if (xx <= t1)
            {
                y = 0.0;
            }
            else if (xx <= t2)
            {
                y = (1.0 - Math.Exp(kg * (t1 - xx))) / kg
                  + (Math.Exp(ka * (t1 - xx)) - Math.Exp(kg * (t1 - xx))) / (ka - kg);
            }
            else
            {
                y1 = (Math.Exp(kg * (t2 - xx)) - Math.Exp(kg * (t1 - xx)) + Math.Exp(ka * (t1 - t2))
                - Math.Exp(ka * (t1 - t2)) * Math.Exp(kg * (t2 - xx))) / kg;
                y2 = (-Math.Exp(ka * (t1 - t2)) * Math.Exp(kg * (t2 - xx))
                + Math.Exp(ka * (t2 - xx)) + Math.Exp(kg * (t1 - xx))
                - Math.Exp(kg * (t2 - xx)) - 2.0 * Math.Exp(ka * (t1 - xx))
                + 2.0 * Math.Exp(ka * (t1 - t2)) * Math.Exp(kg * (t2 - xx))) / (kg - ka);
                y = y1 + y2;
            }

            return (y * (A) * kg);
        }
    }
}
