/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib
{
    /// <summary>
    /// Point in 3D space.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public class Point : ConPntCnt
    {
        /// <summary>
        /// x-coordinate, (input/output)
        /// </summary>
        public double x;
        /// <summary>
        /// y-coordinate, (input/output)
        /// </summary>
        public double y;
        /// <summary>
        /// z-coordinate, (input/output)
        /// </summary>
        public double z;
        /// <summary>
        /// Constructs point at specified location.
        /// </summary>
        /// <param name="x">x-coordinate</param>
        /// <param name="y">y-coordinate</param>
        /// <param name="z">z-coordinate</param>
        public Point(double x, double y, double z)
        {
            this.x = x;
            this.y = y;
            this.z = z;
        }
        /// <summary>
        /// Constructs point at specified location.
        /// </summary>
        /// <param name="p">location</param>
        public Point(Point p)
        {
            this.x = p.x;
            this.y = p.y;
            this.z = p.z;
        }
        /// <summary>
        /// Constructs point at specified location.
        /// </summary>
        /// <param name="p">coordinates in 3-length array {x,y,z}</param>
        public Point(double[] p)
        {
            this.x = p[0];
            this.y = p[1];
            this.z = p[2];
        }
        /// <summary>
        /// Constructs point at location (0, 0, 0)
        /// </summary>
        public Point()
        {   
            x = 0.0;
            y = 0.0;
            z = 0.0;
        }
        /// <summary>
        /// Addition operator
        /// </summary>
        /// <param name="p1">1st addition term point</param>
        /// <param name="p2">2nd addition term point</param>
        public static Point operator +(Point p1, Point p2) {
            Point r = new Point();
            r.x = p1.x + p2.x;
            r.y = p1.y + p2.y;
            r.z = p1.z + p2.z;
            return r;
        }
        /// <summary>
        /// Decrement operator
        /// </summary>
        /// <param name="p1">minuend point</param>
        /// <param name="p2">subtrahend point</param>
        public static Point operator -(Point p1, Point p2)
        {
            Point r = new Point();
            r.x = p1.x - p2.x;
            r.y = p1.y - p2.y;
            r.z = p1.z - p2.z;
            return r;
        }
        /// <summary>
        /// Division operator. Divides each coordinate component with divisor.
        /// </summary>
        /// <param name="p1">divident point</param>
        /// <param name="d">divisor</param>
        public static Point operator /(Point p1, double d)
        {
            if (d == 0)
                throw new TPCException("Cannot do division point because divisor has zero value");
            Point r = new Point();
            r.x = p1.x / d;
            r.y = p1.y / d;
            r.z = p1.z / d;
            return r;
        }
        /// <summary>
        /// Division operator
        /// </summary>
        /// <param name="p1">divident point</param>
        /// <param name="p2">divisor point</param>
        public static Point operator /(Point p1, Point p2) {
            if(p2.x == 0 || p2.y == 0 || p2.z == 0)
                throw new TPCException("Cannot do division point because divisor ["+p2.x+","+p2.y+","+p2.z+"]has zero value");
            Point r = new Point();
            r.x = p1.x / p2.x;
            r.y = p1.y / p2.y;
            r.z = p1.z / p2.z;
            return r;
        }
        /// <summary>
        /// Multiplication operator
        /// </summary>
        /// <param name="p1">1st point that is multiplied</param>
        /// <param name="p2">2nd point that is multiplied</param>
        public static Point operator *(Point p1, Point p2)
        {
            Point r = new Point();
            r.x = p1.x * p2.x;
            r.y = p1.y * p2.y;
            r.z = p1.z * p2.z;
            return r;
        }
        /// <summary>
        /// Multiplication operator
        /// </summary>
        /// <param name="p1">1st point that is multiplied</param>
        /// <param name="d">multiplicator</param>
        public static Point operator *(Point p1, double d)
        {
            Point r = new Point();
            r.x = p1.x * d;
            r.y = p1.y * d;
            r.z = p1.z * d;
            return r;
        }
        /// <summary>
        /// Returns string representation of point
        /// </summary>
        /// <returns>point object as string</returns>
        public override string ToString()
        {
            return "Point[" + x.ToString().Replace(',', '.') + "," + y.ToString().Replace(',', '.') + "," + z.ToString().Replace(',', '.') + "]";
        }
        /// <summary>
        /// Equality test.
        /// </summary>
        /// <param name="obj">other object</param>
        /// <returns>true if obj is equal to this object</returns>
        public override bool Equals (object obj)
        {
            if (obj.GetType() != typeof(Point))
                return false;

            Point p = (Point)obj;   
            return ((this.x == p.x) && (this.y == p.y) && (this.z == p.z));
        }
        /// <summary>
        /// Returns hash code.
        /// </summary>
        /// <returns>base class hash code</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        /// <summary>
        /// Calculates euclidean distance
        /// </summary>
        /// <param name="p">reference point</param>
        /// <returns>distance between points in 3D space</returns>
        public double dist(Point p) {
            return Math.Sqrt(Math.Pow(x - p.x, 2.0) + Math.Pow(y - p.y, 2.0) + Math.Pow(z - p.z, 2.0));
        }
        /// <summary>
        /// Calculates cross-product of points
        /// </summary>
        /// <param name="p1">point1</param>
        /// <param name="p2">point2</param>
        /// <returns>cross product point</returns>
        public static Point Cross(Point p1, Point p2)
        {
            Point r = new Point();
	        r.x = p1.y*p2.z-p1.z*p2.y;
            r.y = -p1.x * p2.z + p1.z * p2.x;
            r.z = p1.x * p2.y - p1.y * p2.x;
            return r;
        }
        /// <summary>
        /// Calculates cross-product of points
        /// </summary>
        /// <param name="r">target point</param>
        /// <param name="p1">point1</param>
        /// <param name="p2">point2</param>
        /// <returns>cross product point</returns>
        public static void Cross(ref Point r, Point p1, Point p2)
        {
            r.x = p1.y * p2.z - p1.z * p2.y;
            r.y = -p1.x * p2.z + p1.z * p2.x;
            r.z = p1.x * p2.y - p1.y * p2.x;
        }
        /// <summary>
        /// Calculates dot product
        /// </summary>
        /// <param name="p">reference point</param>
        /// <returns>dot product of points</returns>
        public double Dot(Point p)
        {
	        return x*p.x+y*p.y+z*p.z;
        }
        /// <summary>
        /// Calculates dot product
        /// </summary>
        /// <param name="p1">reference point</param>
        /// <param name="p2">reference point</param>
        /// <returns>dot product of points</returns>
        public static double Dot(Point p1, Point p2)
        {
            return p1.x * p2.x + p1.y * p2.y + p1.z * p2.z;
        }
        /// <summary>
        /// Calculates the norm of point as vector.
        /// </summary>
        /// <returns>norm of coordinate vector</returns>
        public double norm()
        {
            return Math.Sqrt(Math.Pow(x, 2.0) + Math.Pow(y, 2.0) + Math.Pow(z, 2.0));
        }
        /// <summary>
        /// Normalize point. This means that the norm of 
        /// point coordinate becomes 1 (or very close).
        /// </summary>
        public void Normalize()
        {
            double norm = Math.Sqrt(Math.Pow(x, 2.0) + Math.Pow(y, 2.0) + Math.Pow(z, 2.0));
	        if(norm != 0) {
		        x /= norm;
		        y /= norm;
		        z /= norm;
	        } else {
		        x = 0.0;
		        y = 0.0;
		        z = 0.0;
	        }
        }
        /// <summary>
        /// Sets point location
        /// </summary>
        /// <param name="x">x-coordinate</param>
        /// <param name="y">y-coordinate</param>
        /// <param name="z">z-coordinate</param>
        public void Set(int x, int y, int z)
        {
            this.x = x;
            this.y = y;
            this.z = z;
        }
        /// <summary>
        /// Normalize point. This means that the norm of 
        /// point coordinate becomes 1 (or very close).
        /// </summary>
        public static Point Normalize(Point p)
        {
            p.Normalize();
            return p;
        }
        /// <summary>
        /// Division with constant.
        /// </summary>
        /// <param name="d">divisor</param>
        public void Divide(float d) {
            if (d == 0) throw new TPCInvalidArgumentsException("Divisor is zero."); 
            x /= d;
            y /= d;
            z /= d;
        }
        /// <summary>
        /// Sets point location.
        /// </summary>
        /// <param name="p">new location</param>
        public void Set(Point p)
        {
            this.x = p.x;
            this.y = p.y;
            this.z = p.z;
        }
        /// <summary>
        /// Subtraction operation
        /// </summary>
        /// <param name="r">target point</param>
        /// <param name="p1">subtracted</param>
        /// <param name="p2">subtractor</param>
        public static void Sub(ref Point r, Point p1, Point p2) {
            r.x = p1.x - p2.x;
            r.y = p1.y - p2.y;
            r.z = p1.z - p2.z;
        }
        /// <summary>
        /// Access to coordinate indexes
        /// </summary>
        /// <param name="i">index in [0,1,2]</param>
        /// <returns>coordinate value at wanted index</returns>
        /// <exception cref="TPCInvalidArgumentsException">if coordinate index is not in {1,2,3}</exception>
        public double this[int i] {
            get {
                if (i == 0) return x;
                else if (i == 1) return y;
                else if (i == 2) return z;
                else throw new TPCInvalidArgumentsException("Index out of bounds.");
            }
            set {
                if (i == 0) x = value;
                else if (i == 1) y = value;
                else if (i == 2) z = value;
                else throw new TPCInvalidArgumentsException("Index out of bounds.");
            }
        }
        /// <summary>
        /// Returns data of this point as array
        /// </summary>
        /// <returns>array {x,y,z}</returns>
        public double[] ToArray() {
            return new double[] { x, y, z };
        }
    }
}
