﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Cirle-shaped Region Of Interest
    /// </summary>
    public class CircleROI : ROI
    {
        /// <summary>
        /// Gets/Sets the radius of the circle
        /// </summary>
        public Double Radius
        {
            get { return radius; }
            set { if (radius != value) { radius = value; shapewidth = value*2.0d; shapeheight = value*2.0d; ReCalculate(); } }
        }

        /// <summary>
        /// Gets/Sets the number of points drawn to the arc of the circle. If the Imadeus or
        /// Imagetool fill methods are used, the area is considered as traceROI and points are used.
        /// If Shape fill method is used, this will have no affect to result. 
        /// </summary>
        public int PointNum
        {
            get
            {
                return points.Count;
            }
            set
            {
                if (value < 3) value = 3;
                if (value != pointNum)
                {
                    pointNum = value;
                    ReCalculate();
                }
            }
        }

        /// <summary>
        /// Fills the matrix with ones inside the ROI.
        /// </summary>
        protected override void FillMatrix()
        {
            base.FillMatrix();

            if (FillMethod == Fill_Method.Shape)
            {
                CircleFill();
            }            
        }


        /// <summary>
        /// Calculates the transformed(zoomed,rotated) points. This parent function calculates
        /// only the y and x component vectors.
        /// </summary>
        protected override void ReCalculate()
        {
            base.ReCalculate();

            points = new List<ROIPoint>();

            for (int i = 0; i < pointNum; i++)
            {
                Double rad = (Math.PI / 180.0d) * (360.0d * ((Double)i / (Double)pointNum));
                points.Add( new ROIPoint(
                    calcX(Math.Sin(rad) * radius, Math.Cos(rad) * radius ),
                    calcY(Math.Sin(rad) * radius, Math.Cos(rad) * radius ) ) );
            }
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI(Double x, Double y, Double radius)
            : base(".", 1.0d, new Matrix_Number(0), ROI_Type.Circle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = Math.Abs(radius);
            pointNum = 36;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI(Double x, Double y, Double radius, int pointnumber)
            : base(".", 1.0d, new Matrix_Number(0), ROI_Type.Circle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = Math.Abs(radius);
            pointNum = pointnumber;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI(Double x, Double y, Double radius, Double zoomFactor)
            : base(".", zoomFactor, new Matrix_Number(0), ROI_Type.Circle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = Math.Abs(radius);
            pointNum = 36;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI(String name, Double zoomFactor, Matrix_Number MatNum,
                         Double x, Double y, int roiNum, Double width_height )
            : base( name, zoomFactor, MatNum, ROI_Type.Circle, x, y, roiNum )
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = Math.Abs(width_height / 2.0d);
            this.x += this.radius;
            this.y += this.radius;
            pointNum = 36;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI( ROI roi ) : base( roi )
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = 1.0d;
            pointNum = 36;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI(CircleROI roi)
            : base(roi)
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = roi.radius;
            pointNum = roi.pointNum;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI()
            : base()
        {
            this.Header.RoiType = ROI_Type.Circle;
            this.radius = 0;
        }

        /// <summary>
        /// Creates replica of CircleROI
        /// </summary>
        public override ROI Clone()
        {
            CircleROI r = new CircleROI(this);            
            return r;
        }

        private Double radius;
        private int pointNum;

        private void CircleFill()
        {
            // new values, that are scaled and are in same scale as the matrix.
            int new_radius = Convert.ToInt32(radius * factorX);
            int new_y_upper = 0;
            int new_y_lower = 0;
            int new_y = Convert.ToInt32(y * factorY);
            int new_x = Convert.ToInt32(x * factorX);

            int x_length = 0;
            int x_left = 0;
            int x_right = 0;

            // we go through all y pixels and calculate the corresponding x pixels for the circle
            for (int yy = 0; yy < new_radius; yy++)
            {
                // upper and lower half of circle.
                new_y_upper = new_y + yy;
                new_y_lower = new_y - yy;

                // next we calculate the points when we cut the circle along x-axis
                x_length = Convert.ToInt32(Math.Sqrt(new_radius * new_radius - yy * yy));
                x_left = new_x - x_length;
                x_right = new_x + x_length;

                if (x_left < 0) x_left = 0;
                if (x_right > obj_matrix.Width) x_right = obj_matrix.Width;

                // upper half
                if (new_y_upper >= 0 && new_y_upper < obj_matrix.Height)
                    for (int i = x_left; i < x_right; i++)
                    {
                        (obj_matrix as ROI_Matrix)[i, new_y_upper] = 1;
                    }

                // lower half
                if (new_y_lower >= 0 && new_y_lower < obj_matrix.Height) for (int i = x_left; i < x_right; i++)
                    {
                        (obj_matrix as ROI_Matrix)[i, new_y_lower] = 1;
                    }
            }
        }
    }
}
