﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Creates ellipse shaped regions of interests.
    /// </summary>
    public class EllipseROI : ROI
    {
        /// <summary>
        /// Gets/Sets the number of points drawn to the arc of the ellipse. If the Imadeus or
        /// Imagetool fill methods are used, the area is considered as traceROI and points are used.
        /// If EllipseFill -method is used, this does nothing. 
        /// </summary>
        public int PointNum
        {
            get
            {
                return points.Count;
            }
            set
            {
                if (value < 3) value = 3;
                if (value != pointNum)
                {
                    pointNum = value;
                    notReady = true;
                }
            }
        }

        /// <summary>
        /// Fills the matrix with ones inside the ROI.
        /// </summary>
        protected override void FillMatrix() 
        {
            base.FillMatrix();

            if (this.FillMethod == Fill_Method.Shape)
            {
                EllipseFill();
            }            
        }

        /// <summary>
        /// Calculates the transformed(zoomed,rotated) points. This parent function calculates
        /// only the y and x component vectors.
        /// </summary>
        protected override void ReCalculate()
        {
            base.ReCalculate();

            Double half_width = shapewidth / 2.0d;
            Double half_height = shapeheight / 2.0d;

            points = new List<ROIPoint>();

            for (int i = 0; i < pointNum; i++)
            {
                Double rad = (Math.PI / 180.0d) * (360.0d * ((Double)i / (Double)pointNum));

                double newX = Math.Cos(rad) * half_width;
                double newY = Math.Sin(rad) * half_height;

                points.Add(new ROIPoint(
                    calcX(newX, newY),
                    calcY(newX, newY)));
            }
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        /// <param name="x">Location x</param>
        /// <param name="y">Location y</param>
        /// <param name="width">Ellipse width</param>
        /// <param name="height">Ellipse height</param>
        public EllipseROI(Double x, Double y, Double width, Double height )
            : base(".", 1.0d, new Matrix_Number(0), ROI_Type.Circle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Ellipse;
            this.shapewidth = Math.Abs(width);
            this.shapeheight = Math.Abs(height);
            pointNum = 40;

            x = x + width / 2.0d;
            y = y + height / 2.0d;
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI(Double x, Double y, Double width, Double height, int pointnumber)
            : base(".", 1.0d, new Matrix_Number(0), ROI_Type.Circle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Ellipse;
            this.shapewidth = Math.Abs(width);
            this.shapeheight = Math.Abs(height);
            pointNum = pointnumber;

            x = x + width / 2.0d;
            y = y + height / 2.0d;
        }


        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI(Double zoomFactor, Double x, Double y, Double width, Double height)
            : base( ".", zoomFactor, new Matrix_Number(0), ROI_Type.Circle, x, y , 0 )
        {
            this.Header.RoiType = ROI_Type.Ellipse;
            this.shapewidth = Math.Abs(width);
            this.shapeheight = Math.Abs(height);
            pointNum = 40;

            x = x + width / 2.0d;
            y = y + height / 2.0d;
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI(ROI roi)
            : base( roi )
        {
            this.Header.RoiType = ROI_Type.Ellipse;
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI(EllipseROI roi)
            : base(roi)
        {
            this.Header.RoiType = roi.Header.RoiType;
            this.shapeheight = roi.shapeheight;
            this.shapewidth = roi.shapewidth;
            this.pointNum = roi.pointNum;
        }

        /// <summary>
        /// Creates ellipse ROI
        /// </summary>
        public EllipseROI()
            : base()
        {
            this.Header.RoiType = ROI_Type.Ellipse;
            this.width = 0;
            this.height = 0;
            this.pointNum = 40;
        }
        /*
        public EllipseROI(String name, Double zoomFactor, Matrix_Number MatNum,
                         Double x, Double y, int roiNum, Double width, Double height )
            : base( name, zoomFactor, MatNum, ROI_Type.Circle, x, y, roiNum )
        {
            this.Header.RoiType = ROI_Type.Ellipse;
            this.shapewidth = Math.Abs(width);
            this.shapeheight = Math.Abs(height);
            pointNum = 40;
        }
        */

        /// <summary>
        /// Creates replica of EllipseROI
        /// </summary>
        public override ROI Clone()
        {
            EllipseROI r = new EllipseROI(this);
            return r;
        }

        private int pointNum;

        /// <summary>
        /// This method fills the matrix without points when FillMethod is Shape. This ignores the
        /// transforming ( flipping, zooming, rotating )
        /// </summary>
        private void EllipseFill()
        {
            ROI_Matrix matrix = (obj_matrix as ROI_Matrix);
            
            Double new_width = shapewidth * factorX * 0.5d; // a
            Double new_height = shapeheight * factorY * 0.5d; // b
            
            // new values, that are scaled and are in same scale as the matrix.
            //int new_radius = Convert.ToInt32(radius * factor);
            int new_y_upper = 0;
            int new_y_lower = 0;
            int new_y = Convert.ToInt32(y * factorY + new_width);
            int new_x = Convert.ToInt32(x * factorX + new_height);

            int x_length = 0;
            int x_left = 0;
            int x_right = 0;

            // we go through all y pixels and calculate the corresponding x pixels for the circle
            for (int yy = 0; yy < new_height; yy++)
            {
                // upper and lower half of circle.
                new_y_upper = new_y + yy;
                new_y_lower = new_y - yy;

                // next we calculate the points when we cut the circle along x-axis
                // old x1 = temp_roundf(sqrt(r * r - y * y));
                //     x1 = sqrt(1-(y*y)/(float)(b*b))*a;
                //x_length = Convert.ToInt32(Math.Sqrt(new_radius * new_radius - yy * yy));
                x_length = (int)(Math.Sqrt(1.0d - (yy * yy) / (new_height * new_height)) * new_width);

                x_left = new_x - x_length;
                x_right = new_x + x_length;

                if (x_left < 0) x_left = 0;
                if (x_right > matrix.Width) x_right = matrix.Width;

                // upper half
                if (new_y_upper >= 0 && new_y_upper < matrix.Height)
                    for (int i = x_left; i < x_right; i++)
                    {
                        matrix[i, new_y_upper] = 1;
                    }

                // lower half
                if (new_y_lower >= 0 && new_y_lower < matrix.Height) for (int i = x_left; i < x_right; i++)
                    {
                        matrix[i, new_y_lower] = 1;
                    }
            }
        }
    }
}
