﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// This class provides some general helper functions.
    /// </summary>
    static class HelpFunctions
    {
        /// <summary>
        /// Rounds the double value always to lower int
        /// </summary>
        public static int LowRound(double value)
        {
            if (value > 0) return (int)value;
            else return (int)value - 1;
        }

        /// <summary>
        /// Creates a fixed size String from input string and result length.
        /// All extra space is filled with spaces. If the input string is longer than result,
        /// the remainders are cut off
        /// </summary>
        /// <param name="input">The input string</param>
        /// <param name="length">length if result string</param>
        /// <returns></returns>
        public static String FixStr(String input, int length)
        {
            // null input is treated like empty string 
            if (input == null) return new String(' ', length);

            int difference = length - input.Length;

            // the input string is longer than the boundaries given
            if (difference < 0) { return input.Substring(0, length); }
            else if (difference == 0) { return input; }
            else return input.PadRight(input.Length + difference, ' ');
        }

        /// <summary>
        /// Gets one remaining string in line from Header file. For example getString( "Study:", long_text_file )
        /// </summary>
        /// <param name="str">Name of information, for example "Study:"</param>
        /// <param name="text">Reference to string object containing all header text.</param>
        /// <returns></returns>
        public static String GetParameterFromText(String str, ref String text)
        {
            if (text.Contains(str))
            {
                int start_index = text.IndexOf(str) + str.Length;
                int end_index = text.IndexOf("\n", start_index);

                //if (end_index - start_index < 0) return "";

                String result = text.Substring(start_index, end_index - start_index);

                result = result.Trim(new char[] { ' ', '\t', '\r', '\n' });
                return result;
            }
            else
            {
                return ".";
            }
        }

        /// <summary>
        /// Gets one remaining string in line from Header file. For example getString( "Study:", long_text_file )
        /// </summary>
        /// <param name="str">Name of information, for example "Study:"</param>
        /// <param name="text">Reference to string object containing all header text.</param>
        /// <returns></returns>
        public static String GetParameterFromText(String str, String text)
        {
            if (text.Contains(str))
            {
                int start_index = text.IndexOf(str) + str.Length;
                int end_index = text.IndexOf("\n", start_index);
                
                // If the line do not contain the end character
                if (end_index == -1) end_index = text.Length;

                String result = text.Substring(start_index, end_index - start_index);

                result = result.Trim(new char[] { ' ', '\t', '\r', '\n' });
                return result;
            }
            else
            {
                return ".";
            }
        }

        /// <summary>
        /// Converts double value to string. NaN value is returned as "."
        /// </summary>
        /// <param name="number">Double value that is to be converted</param>
        /// <returns>Double value as string</returns>
        public static String DoubleToStr(Double number)
        {
            if (Double.IsNaN(number)) return ".";
            else
            {
                // If the number is very small/big, it is better to print the double value in scientific format
                //if (Math.Abs(number) > 100000 || Math.Abs(number) < 0.00001) 
                //return number.ToString("e8", new System.Globalization.CultureInfo("en-GB"));
                return number.ToString("F8", new System.Globalization.CultureInfo("en-GB"));
            }
        }

        /// <summary>
        /// Converts strings to double in scientific mode with dot
        /// </summary>
        /// <param name="str"></param>
        /// <returns>a double which is converted from string</returns>
        public static Double StrToDouble(String str)
        {
            if (str.Equals(".")) return Double.NaN;
            else return Convert.ToDouble(str, new System.Globalization.CultureInfo("en-GB"));
        }

        /// <summary>
        /// Prints 1 if value is true, 0 if false
        /// </summary>
        /// <param name="value">Boolean value</param>
        /// <returns>0 or 1</returns>
        public static char Print1_0(bool value)
        {
            if (value) return '1';
            else return '0';
        }

        /// <summary>
        /// This function reads next not empty line of the reader and returns it as string.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        public static String ReadNextLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            return line;
        }

        /// <summary>
        /// Rounding function for imagetool ROIs
        /// </summary>
        /// <param name="value">Value to be rounded</param>
        /// <returns>Rounded value</returns>
        public static int temp_roundf(double value)
        {
            if (value < 0)
                return (int)(value - 0.5d);
            else
                return (int)(value + 0.5d);
        }
    }
}
