﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// This class fills the matrix with Imadeus masking algorithm
    /// </summary>
    static class ImadeusMask
    {
        /// <summary>
        /// Fills the matrix with Imadeus algorithm.
        /// </summary>
        /// <param name="matrix">Matrix that will be filled.</param>
        /// <param name="points">Trace points od ROI.</param>
        public static void Fill(ROI_Matrix matrix, List<ROIPoint> points)
        {
            //else if (fill_method == Fill_Method.Imadeus) return Convert.ToInt32((x_axisY * xx + y_axisY * yy ) + y * factorY);

            // Converting points to imadeus:
            List<ROIPoint> p = new List<ROIPoint>();
            for (int i = 0; i < points.Count; i++)
            {
                p.Add( new ROIPoint(HelpFunctions.LowRound(points[i].x), HelpFunctions.LowRound(points[i].y)) );
            }
            
            List<ROIPoint> steps = CalculateSteps(ref p);
            ROI_Matrix mat = new ROI_Matrix(matrix.Width, matrix.Height);
            Mask(ref steps, ref mat);
            
            // Last we must ensure that the area is symmetric.
            MaskTraceLine(ref p, ref steps, ref mat);

            matrix.Union(mat);
        }

        /// <summary>
        /// Calculates all the steps in the trace
        /// </summary>
        /// <param name="points">Trace points</param>
        /// <returns>List of steps (one pixel)</returns>
        public static List<ROIPoint> CalculateSteps(ref List<ROIPoint> points)
        {
            List<ROIPoint> ret = new List<ROIPoint>();
            if (points.Count == 0) return ret;

            int i_n = 0;

            for (int i = 0; i < points.Count; i++)
            {
                if (i == points.Count - 1) i_n = 0;
                else i_n = i + 1;
                brlinexya(points[i], points[i_n], ref ret);
            }
            return ret;
        }


        /// <summary>
        /// Calculates the steps(one pixel length) of line
        /// </summary>
        /// <param name="start">Starting point of the line</param>
        /// <param name="end">Ending point of the line</param>
        /// <param name="ret">list of points that are included in line</param>
        /// <returns>List of steps.</returns>
        private static void brlinexya(ROIPoint start, ROIPoint end, ref List<ROIPoint> ret)
        {
            // If there is no line
            if (start.x == end.x && start.y == end.y) return;

            int x0 = (int)start.x;
            int y0 = (int)start.y;
            int x1 = (int)end.x;
            int y1 = (int)end.y;

            //int pix = color.getRGB();
            int dy = y1 - y0;
            int dx = x1 - x0;
            int stepx, stepy;

            if (dy < 0) { dy = -dy; stepy = -1; } else { stepy = 1; }
            if (dx < 0) { dx = -dx; stepx = -1; } else { stepx = 1; }
            dy <<= 1;                                                  // dy is now 2*dy
            dx <<= 1;                                                  // dx is now 2*dx

            //raster.setPixel(pix, x0, y0);
            ret.Add(new ROIPoint(x0, y0));

            if (dx > dy)
            {
                int fraction = dy - (dx >> 1);                         // same as 2*dy - dx
                while (x0 != x1)
                {
                    if (fraction >= 0)
                    {
                        y0 += stepy;
                        fraction -= dx;                                // same as fraction -= 2*dx
                    }
                    x0 += stepx;
                    fraction += dy;                                    // same as fraction -= 2*dy
                    //raster.setPixel(pix, x0, y0);
                    ret.Add(new ROIPoint(x0,y0));
                }
            }
            else
            {
                int fraction = dx - (dy >> 1);
                while (y0 != y1)
                {
                    if (fraction >= 0)
                    {
                        x0 += stepx;
                        fraction -= dy;
                    }
                    y0 += stepy;
                    fraction += dx;
                    //raster.setPixel(pix, x0, y0);
                    ret.Add(new ROIPoint(x0,y0));
                }
            }
        }

        /// <summary>
        /// Fills the matrix with ones inside the region.
        /// </summary>
        /// <param name="points">List of one step points.</param>
        /// <param name="matrix">Matrix which is to be filled</param>
        private static void Mask( ref List<ROIPoint> points, ref ROI_Matrix matrix )
        {
            int dy = 0;
            int i_n = 1;

            for (int i = 0; i < points.Count; i++)
            {
                if (i == points.Count - 1) { i_n = 0; }
                else i_n = i + 1;

                //if (points[i].y < 0 || points[i].x < 0 /*|| points[i].x >= matrix.Width*/ || points[i].y >= matrix.Height) continue;
                //if (points[i_n].y < 0 || points[i_n].x < 0 /*|| points[i_n].x >= matrix.Width*/ || points[i_n].y >= matrix.Height) continue;

                dy = (int)(points[i_n].y - points[i].y);

                if (dy < 0)
                {
                    if (points[i].y < 0 || points[i].y >= matrix.Height) continue;
                    int Endx = (int)points[i].x;
                    if (Endx >= matrix.Width) Endx = matrix.Width;
                    for (int w = 0; w < Endx; w++) matrix[w, (int)points[i].y]++; 
                }
                else if (dy>0 )
                {
                    if (points[i_n].y < 0 || points[i_n].y >= matrix.Height) continue;
                    int Endx = (int)points[i_n].x;
                    if (Endx >= matrix.Width) Endx = matrix.Width;
                    for (int w = 0; w < Endx; w++) matrix[w, (int)points[i_n].y]++;
                }
            }

            // After this the matrix will have only 0 and 1 values. Divided (mod 2) 
            matrix.Mod();          
        }

        private static void MaskTraceLine(ref List<ROIPoint> points, ref List<ROIPoint> steps,ref ROI_Matrix matrix )
        {
            // Because clockwise steps have been calculated earlier, we dont have to
            // calculate those again
            foreach (ROIPoint rp in steps)
            {
                if (rp.x < 0 || rp.x >= matrix.Width || rp.y < 0 || rp.y >= matrix.Height) continue;
                matrix[(int)rp.x, (int)rp.y] = 1;
            }

            List<ROIPoint> counterclockwise = new List<ROIPoint>();
            // Counter clockwise boundary masking.
            int i_n = points.Count;
            for (int i = points.Count-1; i >= 0; i--)
            {
                if (i == 0) i_n = points.Count-1;
                else i_n = i - 1;
                brlinexya(points[i], points[i_n], ref counterclockwise );
            }

            foreach (ROIPoint rp in counterclockwise)
            {
                if (rp.x < 0 || rp.x >= matrix.Width || rp.y < 0 || rp.y >= matrix.Height) continue;
                matrix[(int)rp.x, (int)rp.y] = 1;
            }
        }
    }
}
