﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace TPClib.ROI 
{
    /// <summary>
    /// Class that reads ROI -files of Imadeus
    /// </summary>
    public class ImadeusROIFile : ROIFile
    {
     
        /// <summary>
        /// Possible combinations of Imadeus ROIs
        /// </summary>
        public class ROICombination
        {
            /// <summary>
            /// Name of combinated roiset
            /// </summary>
            public String Name;

            /// <summary>
            /// Roi names which are part of combination
            /// </summary>
            public List<String> ROIs;
            /// <summary>
            /// Default constructor
            /// </summary>
            public ROICombination()
            {
                Name = ".";
                ROIs = new List<string>();
            }
        }

        /// <summary>
        /// Possible combinations of Imadeus ROIs.
        /// </summary>
        public List<ROICombination> Combinations;

        /// <summary>
        /// The [Creator] part at the end of the Imadeus file
        /// </summary>
        public struct Creator_Field
        {
            /// <summary>
            /// Creator product name
            /// </summary>
            public String ProductName;
            /// <summary>
            /// Creator version
            /// </summary>
            public String Version;
            /// <summary>
            /// Copyright notice if any
            /// </summary>
            public String Copyright;
            /// <summary>
            /// Gets this object as a string
            /// </summary>
            /// <returns>string representation of this object</returns>
            public override String ToString()
            {
                String output = 
                    "[Creator]\r\nProductName="+ProductName+"\r\nVersion="+Version+
                    "\r\nProductName="+ProductName+"\r\nCopyright="+Copyright+"\r\n";
                return output;
            }
        }

        /// <summary>
        /// The [Creator] part at the end of the Imadeus file
        /// </summary>
        public Creator_Field CreatorField;

        private int Regions;

        /// <summary>
        /// Format how doubles are stored in ImadeusROIFile
        /// </summary>
        private static System.Globalization.CultureInfo doubleformat = new System.Globalization.CultureInfo("en-GB");

        /// <summary>
        /// Creates an Imadeus ROI file object
        /// </summary>
        public ImadeusROIFile() : base()
        {            
            Init();
        }
        /// <summary>
        /// Creates an Imadeus ROI file object
        /// </summary>
        public ImadeusROIFile(String filename) : base( filename )
        {
            Init();
        }

        /// <summary>
        /// Checks if the file is valid Imadeus ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
        public override bool CheckFormat()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            try
            {
                FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
                StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
                string text = reader.ReadToEnd();
                filestream.Close();
                reader.Close();

                if (!text.Contains("[Definition]")) return false;
                if (!text.Contains("XVoxelDim=")) return false;
                if (!text.Contains("XOri=")) return false;

                return true;
            }
            catch (Exception)
            {
                return false;
            }
        }

        /// <summary>
        /// Writes the ROI information to ImadeusROIfile
        /// </summary>
        public override void WriteFile()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;

            fileWriter.Write(WriteHeader());
            fileWriter.Write(WriteData());
            fileWriter.Write(WriteCombinations());
            fileWriter.Write(WriteCreator());

            filestream.Close();
        }

        /// <summary>
        /// Reads Imadeus ROI files.
        /// </summary>
        public override void ReadFile()
        {
            if (!CheckFormat())
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct Imadeus ROI format.");

            this.ROIFileFormat = ROIFormat.Imadeus;

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());

            string text = reader.ReadToEnd();
            //StringReader strReader = new StringReader(text);

            filestream.Close();
            String HeaderPart = "";
            String DataPart = "";

            try
            {
                HeaderPart = text.Substring(0, text.IndexOf("\n[ROI1]"));
                DataPart = text.Substring(text.IndexOf("\n[ROI1]"));
            }
            catch (Exception)
            {
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct Imadeus ROI format.");
            }

            Init();

            ReadHeader(HeaderPart);
            ReadData(DataPart);
            ReadCombinations(text);
            ReadCreator(text);

            //Console.WriteLine(Header);

        }

        /// <summary>
        /// Writes the contents of ImadeusROIFile to string
        /// </summary>
        /// <returns>String containing the contents of ImadeusFile</returns>
        public override string ToString()
        {
            return WriteHeader() + WriteData() + WriteCombinations() + WriteCreator();
        }

        #region private members

        private new void Init()
        {
            base.Init();

            Combinations = new List<ROICombination>();

            this.CreatorField.Copyright = "";
            this.CreatorField.ProductName = "";
            this.CreatorField.Version = "";
            Regions = 0;
        }

        private String WriteHeader()
        {
            StringBuilder str = new StringBuilder();

            str.Append("[Definition]\r\n");
            str.Append("Regions=" + VOIs.Count + "\r\n");
            str.Append("Image=" + Header.ImageFile + "\r\n");
            str.Append("RawImage=" + Header.RawImage + "\r\n");
            str.Append("Name=" + Header.Name + "\r\n");
            str.Append("Orientetion=" + Header.Orientation + "\r\n");
            str.Append("Orient=" + Header.Orient + "\r\n");
            str.Append("XResolution=" + Header.Resolution_X + "\r\n");
            str.Append("YResolution=" + Header.Resolution_Y + "\r\n");
            str.Append("ZResolution=" + Header.Resolution_Z + "\r\n");
            str.Append("XVoxelDim=" + Header.VoxelDimension_X + "\r\n");
            str.Append("YVoxelDim=" + Header.VoxelDimension_Y + "\r\n");
            str.Append("ZVoxelDim=" + Header.VoxelDimension_Z + "\r\n");
            str.Append("XFlip=" + HelpFunctions.Print1_0(Header.Flip_X) + "\r\n");
            str.Append("YFlip=" + HelpFunctions.Print1_0(Header.Flip_Y) + "\r\n");
            str.Append("ZFlip=" + HelpFunctions.Print1_0(Header.Flip_Z) + "\r\n");
            str.Append("XOri=" + Header.Ori_X + "\r\n");
            str.Append("YOri=" + Header.Ori_Y + "\r\n");
            str.Append("ZOri=" + Header.Ori_Z + "\r\n");
            str.Append("Comment=" + Header.Comment + "\r\n\r\n");

            return str.ToString();
        }

        private void ReadHeader( String text )
        {
            Regions = Convert.ToInt32(HelpFunctions.GetParameterFromText("Regions=", ref text));
            Header.ImageFile = HelpFunctions.GetParameterFromText("Image=", ref text);
            Header.RawImage = HelpFunctions.GetParameterFromText("RawImage=", ref text);
            Header.Name = HelpFunctions.GetParameterFromText("Name=", ref text);
            Header.Orientation = HelpFunctions.GetParameterFromText("Orientation=", ref text);
            Header.Orient = Convert.ToInt32(HelpFunctions.GetParameterFromText("Orient=", ref text));
            Header.Resolution_X = Convert.ToInt32(HelpFunctions.GetParameterFromText("XResolution=", ref text));
            Header.Resolution_Y = Convert.ToInt32(HelpFunctions.GetParameterFromText("YResolution=", ref text));
            Header.Resolution_Z = Convert.ToInt32(HelpFunctions.GetParameterFromText("ZResolution=", ref text));
            Header.VoxelDimension_X = Convert.ToDouble(HelpFunctions.GetParameterFromText("XVoxelDim=", ref text));
            Header.VoxelDimension_Y = Convert.ToDouble(HelpFunctions.GetParameterFromText("YVoxelDim=", ref text));
            Header.VoxelDimension_Z = Convert.ToDouble(HelpFunctions.GetParameterFromText("ZVoxelDim=", ref text));
            Header.Flip_X = Convert.ToBoolean(HelpFunctions.GetParameterFromText("XFlip=", ref text).Equals("1"));
            Header.Flip_Y = Convert.ToBoolean(HelpFunctions.GetParameterFromText("YFlip=", ref text).Equals("1"));
            Header.Flip_Z = Convert.ToBoolean(HelpFunctions.GetParameterFromText("ZFlip=", ref text).Equals("1"));
            Header.Ori_X = Convert.ToInt32(HelpFunctions.GetParameterFromText("XOri=", ref text));
            Header.Ori_Y = Convert.ToInt32(HelpFunctions.GetParameterFromText("YOri=", ref text));
            Header.Ori_Z = Convert.ToInt32(HelpFunctions.GetParameterFromText("ZOri=", ref text));
            Header.Comment = HelpFunctions.GetParameterFromText("Comment=", ref text);
        }

        private void ReadData(String text)
        {
            if (Regions == 0) return;

            //String s;
            //StringSplitOptions.RemoveEmptyEntries;
            char[] separators = new char[] { ' ', ',' };
            
            StringReader sr = new StringReader(text);
            String[] words;

            for (int i = 0; i < Regions; i++)
            {
                // [ROI..n..]
                words = ReadWordsFromLine( ref sr );
                if (words.Length>0 && !words[0].Contains("[ROI" + (i+1) + "]")) 
                    throw new TPCROIFileException("ReadData: File was not in correct format.");

                VOI voi = new VOI();
                
                // Information for one ROI. (Contains many regions)
                //ROIContainer roi = new ROIContainer();
                voi.Name = HelpFunctions.GetParameterFromText( "Name=", sr.ReadLine() );
                int num = Convert.ToInt32( HelpFunctions.GetParameterFromText("nRegion=", sr.ReadLine() ) );
                voi.Color = Convert.ToInt32(HelpFunctions.GetParameterFromText("Color=", sr.ReadLine()));
                voi.Col = Convert.ToInt32(HelpFunctions.GetParameterFromText("col=", sr.ReadLine()));
                voi.ZoomFactor = 1.0d;// Header.VoxelDimension_X;

                // Lets read all the regions for the ROI
                for (int region = 1; region <= num; region++)
                {
                    //String line = HelpFunctions.GetParameterFromText( "Region"+region+"=", sr.ReadLine() );
                    String line = HelpFunctions.ReadNextLine( ref sr );

                    line = line.Substring( line.IndexOf("=")+1 );

                    words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries);

                    int numPoints = Convert.ToInt32(words[2]);

                    if (words.Length > 0)
                    {
                        //TraceROI t_roi = new TraceROI( Header.Ori_X, Header.Ori_Y, Convert.ToInt32(words[0])+Header.Ori_Z );
                        int Z = Convert.ToInt32(words[0]);
                        if (Header.Flip_Z) Z = Header.Resolution_Z - Z;
                        TraceROI t_roi = new TraceROI( 0, 0, Z );

                        t_roi.Header.ROIName = "Region" + region;
                        t_roi.FillMethod = Fill_Method.Imadeus;
                        //t_roi.FlipX = Header.Flip_X;
                        //t_roi.FlipY = Header.Flip_Y;
                        //t_roi.FlipZ = Header.Flip_Z;
                        t_roi.ZoomFactor = 1.0d; // Header.VoxelDimension_X;                        

                        for (int w = 0; w < numPoints; w++ )
                        {
                            /*if( w==0 )
                            {
                                ekax = (Header.Resolution_X - HelpFunctions.StrToDouble(words[w * 2 + 3])) * -1.0d - 1.0d;
                                ekay = (Header.Resolution_Y - HelpFunctions.StrToDouble(words[w * 2 + 4])) * -1.0d - 1.0d;
                                t_roi.X = Header.Ori_X - ekax;
                                t_roi.Y = Header.Ori_Y - ekay;
                            }*/
                            
                            double xp, yp; // flipped points

                            // point x position
                            if (Header.Flip_X) xp = HelpFunctions.StrToDouble(words[w * 2 + 3]);
                            else xp = (Header.Resolution_X - (HelpFunctions.StrToDouble(words[w * 2 + 3]))-1.0d);

                            // point y position
                            if (!Header.Flip_Y) yp = (Header.Resolution_Y - (HelpFunctions.StrToDouble(words[w * 2 + 4])));
                            else yp = HelpFunctions.StrToDouble(words[w * 2 + 4])-1.0d;

                            t_roi.AddPoint(
                                xp,  // x
                                yp   // y
                                );
                        }
                        
                        voi.Add(t_roi);
                    }
                }

                VOIs.Add(voi);
            }         
        }



        private String WriteData()
        {
            StringBuilder str = new StringBuilder();

            int num = 1;

            foreach (VOI voi in VOIs)
            {
                str.Append("[ROI" + num + "]\r\n");
                str.Append( VOIToString( voi ) );
                num++;
            }

            return str.ToString();
        }

        /// <summary>
        /// Writes the contents of VOI to string
        /// </summary>
        /// <returns>String containing the VOI information.</returns>
        private string VOIToString( VOI voi )
        {
            StringBuilder str = new StringBuilder();
            str.Append("Name=" + voi.Name + "\r\n");
            str.Append("nRegion=" + voi.Count + "\r\n");
            str.Append("Color=" + voi.Color + "\r\n");
            str.Append("col=" + voi.Col + "\r\n\r\n");

            int num = 1;

            foreach (ROI r in voi)
            {
                str.Append("Region" + num + "=" + r.Z + ",2," + r.Points.Count + ", ");
                foreach (ROIPoint p in r.Points )
                {
                    double xp, yp; // flipped points

                    // point x position
                    if (Header.Flip_X) xp = p.x;
                    else xp = (Header.Resolution_X - p.x - 1.0d);

                    // point y position
                    if (!Header.Flip_Y) yp = (Header.Resolution_Y - p.y);
                    else yp = p.y - 1.0d;

                    str.Append( xp.ToString("f2", doubleformat ) + ", " + yp.ToString("f2", doubleformat ) + ", " );
                }
                str.Remove(str.Length - 2, 2); // removes the last ", "
                str.Append("\r\n\r\n");

                num++;
            }

            return str.ToString();
        }

        /// <summary>
        /// Reads all combinations from file if those exist
        /// </summary>
        /// <param name="text">File contents in string</param>
        private void ReadCombinations(String text)
        {
            if (text.Contains("[Combinations]"))
            {
                String Comb = text.Substring( text.IndexOf("[Combinations]") );
                StringReader reader = new StringReader(Comb);

                reader.ReadLine(); // [Combinations]
                bool End=false;
                int num = 0;
                while (!End)
                {
                    String line = reader.ReadLine(); // This should be Comb0  ...etc...

                    if (line == null || line.Length <= 4) break;
                    String Param = HelpFunctions.GetParameterFromText("Comb"+num+"=", line);

                    // If there is no meaningful line, we add empty combination line
                    if (Param.Equals(".")) break;//Combinations.Add( new ROICombination() );

                    ROICombination combinat = new ROICombination();
                    String[] words = Param.Split( new char[] {' ' }, StringSplitOptions.RemoveEmptyEntries );
                    if (words.Length <= 3) continue;

                    combinat.Name = words[0];
                    for (int i = 3; i < words.Length; i++) combinat.ROIs.Add(words[i]);

                    Combinations.Add(combinat);

                    num++;
                }
            }
        }

        /// <summary>
        /// Reads the Creator part from Imadeus file
        /// </summary>
        /// <param name="text">File contents in String</param>
        private void ReadCreator(String text)
        {
            if (text.Contains("[Creator]"))
            {
                String Comb = text.Substring(text.IndexOf("[Creator]"));
                StringReader reader = new StringReader(Comb);
                reader.ReadLine(); // The "[Creator]" line
                CreatorField.ProductName = HelpFunctions.GetParameterFromText( "ProductName=", reader.ReadLine() );
                CreatorField.Version = HelpFunctions.GetParameterFromText("Version=", reader.ReadLine());
                reader.ReadLine(); // The Product name is second time here, so the line is useless
                CreatorField.Copyright = HelpFunctions.GetParameterFromText("Copyright=", reader.ReadLine());
            }
        }

        /// <summary>
        /// Writes the combinations to disk
        /// </summary>
        private String WriteCombinations()
        {
            if (Combinations.Count > 0)
            {
                StringBuilder str = new StringBuilder();
                str.Append("[Combinations]\r\n");

                int num=0;
                foreach (ROICombination rc in Combinations)
                {
                    // Main part of combination line "Comb0=name 0 2 "
                    str.Append("Comb"+num+"="+rc.Name+" 0 "+rc.ROIs.Count+" ");
                    foreach (String name in rc.ROIs)
                    {
                        // Names of rois that are part of combination
                        str.Append(name+" ");
                    }
                    str.Append("\r\n");
                    num++;
                }
                str.Append("\r\n");
                return str.ToString();
            }
            return "";
        }

        /// <summary>
        /// Writes the Creator part of Imadeus file to disk
        /// </summary>
        /// <returns>The Creator part as String</returns>
        private String WriteCreator()
        {
            return CreatorField.ToString() + "\r\n";
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored. Returns null if
        /// file has been read to the end.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] ReadWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            //Console.WriteLine(line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

#endregion
    }
}
