﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace TPClib.ROI
{
    /// <summary>
    /// Class that writes/reads ImageTool ROI files.
    /// </summary>
    public class ImageToolROIFile : ROIFile
    {
        /// <summary>
        /// VOI and ROI data of file.
        /// </summary>
        private VOI Data;

        /// <summary>
        /// Creates an Imadeus ROI file object
        /// </summary>
        public ImageToolROIFile()
        {
            Init();
        }
        /// <summary>
        /// Creates an Imadeus ROI file object
        /// </summary>
        public ImageToolROIFile(String filename)
        {
            this.filename = filename;
            Init();
        }

        /// <summary>
        /// Writes the ROI information to ImageToolROIfile
        /// </summary>
        public override void WriteFile()
        {
            Console.WriteLine("filename");
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;

            fileWriter.Write(WriteData());

            filestream.Close();
        }


        /// <summary>
        /// Reads ImageTool ROI files.
        /// </summary>
        public override void ReadFile()
        {
            if (!CheckFormat()) 
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct ImageTool ROI format.");

            this.ROIFileFormat = ROIFormat.ImageTool;

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadToEnd();
            filestream.Close();

            Init();
            ReadData(text);

            this.VOIs = new List<VOI>();
            VOIs.Add(Data);
        }

        /// <summary>
        /// Checks if the file is valid ImageTool ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
        public override bool CheckFormat()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadToEnd();
            filestream.Close();

            StringReader strReader = new StringReader(text);
            String line = HelpFunctions.ReadNextLine(ref strReader);
            
            // If the file is empty
            if (line == null) return false;
            
            // Imagetool roi file starts always with *
            if (line[0] != '*') return false;

            String[] words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries );
            if (words.Length < 13) return false;
            strReader.Close();

            return true;
        }

        /// <summary>
        /// Writes the contents of ImadeusROIFile to string
        /// </summary>
        /// <returns>String containing the contents of ImadeusFile</returns>
        public override string ToString()
        {
            return WriteData();
        }

        #region private members

        private Double recZoom;

        private new void Init()
        {
            base.Init();
            Data = new VOI();
            recZoom = 0.0d;
        }


        private void ReadData(String text)
        {            
            StringReader sr = new StringReader(text);

            //VOI voi = new VOI();

            bool end = false;
            int numRegions=0;

            while (!end)
            {
                String line = HelpFunctions.ReadNextLine(ref sr);
                if (line != null)
                {
                    if (line[0] != '*') throw new TPCROIFileException("The ImageTool file was not in correct format.");

                    String[] words = line.Split(separators);

                    if (words.Length < 13) throw new TPCROIFileException("The ImageTool file was not in correct format.");

                    String imageFile = words[0].Substring(1); // removes the '*'
                    Double zoomFactor = HelpFunctions.StrToDouble(words[1]);
                    Data.ZoomFactor = zoomFactor;

                    // unused reconstuction zoom.
                    recZoom = HelpFunctions.StrToDouble(words[2]);

                    uint MatrixNumber = Convert.ToUInt32(words[3]);
                    int ROItype = Convert.ToInt32(words[4]);
                    int OriginX = Convert.ToInt32(words[6]);
                    int OriginY = Convert.ToInt32(words[7]);
                    int OriginZ = new Matrix_Number(MatrixNumber).Plane-1;

                    int width = Convert.ToInt32(words[8]);
                    int height = Convert.ToInt32(words[9]);
                    int ROInumber = Convert.ToInt32(words[11]);

                    int numPoints = Convert.ToInt32(words[words.Length - 1]);

                    // ROI name can contain spaces
                    String ROIName = "";
                    for (int i = 12; i < words.Length - 1; i++)
                    {
                        if (i != 12) ROIName += " ";
                        ROIName += words[i];
                    }
                    ROIName = ROIName.Substring(0, ROIName.IndexOf("///0"));

                    ROI roi;

                    if (ROItype == 0) { roi = new RectangleROI(); }
                    else if (ROItype == 1) { roi = new CircleROI(); }
                    else if (ROItype == 2) { roi = new EllipseROI(); }
                    else if (ROItype == 3) { roi = new TraceROI(); }
                    else throw new TPCROIFileException("The ImageTool file was not in correct format.");

                    //roi.X = Convert.ToInt32((double)OriginX / zoomFactor);
                    //roi.Y = Convert.ToInt32((double)OriginY / zoomFactor);
                    roi.X = (double)OriginX / zoomFactor;
                    roi.Y = (double)OriginY / zoomFactor;
                    roi.Z = OriginZ;
                    roi.ShapeWidth = (double)width / zoomFactor;
                    roi.ShapeHeight = (double)height / zoomFactor;
                    roi.ZoomFactor = zoomFactor;
                    Header.ImageFile = imageFile;
                    roi.Header.MatrixNumber = new Matrix_Number(MatrixNumber);
                    roi.Header.ROIName = ROIName;
                    roi.Header.ROINumber = ROInumber;
                    roi.Header.Direction = Slice.Direction.Transaxial;
                    roi.FillMethod = Fill_Method.ImageTool;

                    // If ROI type is Trace, next line will contain the trace point data.
                    if (ROItype == 3)
                    {
                        line = HelpFunctions.ReadNextLine(ref sr);
                        words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries);

                        // ROI points must be same amount as the file claims
                        if ((words.Length / 2) != numPoints)
                            throw new TPCROIFileException("The ImageTool file was not in correct format.");

                        for (int i = 0; i < numPoints; i++)
                        {
                            (roi as TraceROI).AddPoint(
                                //Convert.ToInt32(HelpFunctions.StrToDouble(words[i * 2]) / zoomFactor),
                                //Convert.ToInt32(HelpFunctions.StrToDouble(words[i * 2 + 1]) / zoomFactor)
                                (HelpFunctions.StrToDouble(words[i * 2]) / zoomFactor),
                                (HelpFunctions.StrToDouble(words[i * 2 + 1]) / zoomFactor)
                                );
                        }
                    }

                    Data.Add(roi);

                    numRegions++;
                }
                else
                { 
                    end = true;
                }
            }

            this.VOIs.Add( Data );
        }

        private String WriteData()
        {
            //if (VOIs.Count > 0) Data = VOIs[0];
            StringBuilder str = new StringBuilder();

            int num = 1;

            // normally always only one
            foreach (VOI voi in VOIs)
            {
                Data = voi;
                foreach( ROI roi in Data )
                {
                    str.Append("*"+ Header.ImageFile);
                    str.Append(" " + HelpFunctions.DoubleToStr( roi.ZoomFactor ) );
                    str.Append(" " + HelpFunctions.DoubleToStr( recZoom )); // reconstruction zoom
                    str.Append(" " + roi.Header.MatrixNumber.Value );
                    str.Append(" " + (int)roi.Header.RoiType);
                    str.Append(" 1" ); // status code
                    str.Append(" " + roi.X * roi.ZoomFactor );
                    str.Append(" " + roi.Y * roi.ZoomFactor );
                    str.Append(" " + roi.ShapeWidth * roi.ZoomFactor);
                    str.Append(" " + roi.ShapeHeight * roi.ZoomFactor);
                    str.Append(" 0"); // unused
                    str.Append(" " + roi.Header.ROINumber );
                    str.Append(" " + roi.Header.ROIName+"///0");

                    int npoints = 0;
                    if (roi.Header.RoiType == ROI_Type.Trace) npoints = roi.Points.Count;
                    str.Append(" " + npoints);

                    str.Append( "\r\n" );

                    if (roi.Header.RoiType == ROI_Type.Trace)
                    {
                        foreach (ROIPoint rp in (roi as TraceROI).OriginalPoints)
                        {
                            str.Append( (int)(rp.x*roi.ZoomFactor) + " " + (int)(rp.y * roi.ZoomFactor) + " ");
                        }
                        str.Append("\r\n");
                    }

                }
                num++;
            }
                
            return str.ToString();
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored. Returns null if
        /// file has been read to the end.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] ReadWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            //Console.WriteLine(line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

        private char[] separators = new char[] { ' ', '\r', '\n' };

#endregion

    }
}
