﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Parent for all Matrix classes.
    /// </summary>
    public abstract class MatrixObject
    {
        /// <summary>
        /// Returns the width of matrix
        /// </summary>
        public int Width
        {
            get
            {
                return width;
            }
        }
        /// <summary>
        /// Returns the height of matrix
        /// </summary>
        public int Height
        {
            get
            {
                return height;
            }
        }

        /// <summary>
        /// Width of matrix
        /// </summary>
        protected int width;
        /// <summary>
        /// Height of matrix
        /// </summary>
        protected int height;
        /// <summary>
        /// Clears all cell data.
        /// </summary>
        public abstract void Clear();

        /// <summary>
        /// Calculates every selected pixel inside the ROI with given calculator function
        /// </summary>
        /// <param name="img">Image, which pixel values are given to the calculator</param>
        /// <param name="cells">2D cell array containing the ROI area</param>
        /// <param name="slice">The position of 2D cell array in 3D object</param>
        /// <param name="calculator">Calculator object containing the function</param>
        protected void Calculate(ref Image.Image img, byte[][] cells, Slice slice, ROICalculator calculator )
        {
            // ratio check
            int width = cells.Length;
            int height = cells[0].Length;

            if ((img.width / img.height) != (width / height))
                throw new TPCROIException("Mean: Image and matrix can not have different ratio.");

            // Image is larger than matrix. Matrix will be treated as scaled to the size of image
            if (img.width > width)
            {
                int maxx = width;
                int maxy = height;
                int x_img = 0;
                int y_img = 0;

                // Zoom factor must be integers. Image can be only 2x, 3x, 4x, etc... times larger than matrix
                if ((img.width % width) != 0)
                    throw new TPCROIException("Image and matrix ratio cannot have remainder.");

                int ratio = (img.width / width);
                int ratio_power = ratio * ratio;

                for (int y = 0; y < maxy; y++)
                {
                    x_img = 0;
                    for (int x = 0; x < maxx; x++)
                    {
                        if (cells[x][y] > 0)
                        {
                            for (int zy = 0; zy < ratio; zy++) for (int zx = 0; zx < ratio; zx++)
                                {
                                    // We make the transforming of 2d slice to 
                                    // 3d space (transaxial, sagittal and depeths, etc..)
                                    slice.X2D = x_img + zx;
                                    slice.Y2D = y_img + zy;
                                    calculator.AddValue(slice.GetCurve(ref img)); /*img.getCurve( slice.X, slice.Y, slice.Z )*/
                                }
                        }
                        x_img += ratio;
                    }
                    y_img += ratio;
                }
            }
            else if (img.width < width) // matrix is larger than image. Matrix will be treated as scaled to the size of image
            {
                int maxx = img.width;
                int maxy = img.height;
                int x_mat = 0;
                int y_mat = 0;

                // Zoom factor must be integers. Matrix must be 2x, 3x, 4x, etc... times larger than image
                if ((width % img.width) != 0)
                    throw new TPCROIException("Image and matrix ratio cannot have remainder.");

                int ratio = (width / img.width);
                int ratio_power = ratio * ratio;
                for (int y = 0; y < maxy; y++)
                {
                    x_mat = 0;
                    for (int x = 0; x < maxx; x++)
                    {
                        // We make the transforming of 2d slice to 
                        // 3d space (transaxial, sagittal and depeths, etc..)
                        slice.X2D = x;
                        slice.Y2D = y;
                        float[] imgfloat = slice.GetCurve( ref img );
                        for (int zy = 0; zy < ratio; zy++) for (int zx = 0; zx < ratio; zx++)
                            {
                                if (cells[x_mat + zx][y_mat + zy] > 0)
                                {
                                    calculator.AddValue( imgfloat );
                                }
                            }
                        x_mat += ratio;
                    }
                    y_mat += ratio;
                }
            }
            else // image and matrix are same size.
            {
                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        if (cells[x][y] > 0)
                        {
                            // We make the transforming of 2d slice to 
                            // 3d space (transaxial, sagittal and depeths, etc..)
                            slice.X2D = x;
                            slice.Y2D = y;
                            calculator.AddValue(slice.GetCurve(ref img)); /*img.getCurve(slice.X, slice.Y, slice.Z)*/
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Gets list of pixels(ROIValuePoints) from image. All pixels that have selected in matrix
        /// are returned
        /// </summary>
        /// <param name="img">reference image</param>
        /// <param name="cells"></param>
        /// <param name="slice">sluce in image</param>
        /// <returns></returns>
        protected List<ROIValuePoint> GetPixels(ref Image.Image img, byte[][] cells, Slice slice)
        {
            // ratio check
            int width = cells.Length;
            int height = cells[0].Length;

            // List for result
            List<ROIValuePoint> result = new List<ROIValuePoint>();

            if ((img.width / img.height) != (width / height))
                throw new TPCROIException("Mean: Image and matrix can not have different ratio.");

            // Image is larger than matrix
            if (img.width > width)
            {
                int maxx = width;
                int maxy = height;
                int x_img = 0;
                int y_img = 0;
                if ((img.width % width) != 0)
                    throw new TPCROIException("Image and matrix ratio cannot have remainder.");

                int ratio = (img.width / width);
                int ratio_power = ratio * ratio;

                for (int y = 0; y < maxy; y++)
                {
                    x_img = 0;
                    for (int x = 0; x < maxx; x++)
                    {
                        if (cells[x][y] > 0)
                        {
                            for (int zy = 0; zy < ratio; zy++) for (int zx = 0; zx < ratio; zx++)
                                {
                                    // We make the transforming of 2d slice to 
                                    // 3d space (transaxial, sagittal and depeths, etc..)
                                    slice.X2D = x_img + zx;
                                    slice.Y2D = y_img + zy;
                                    result.AddRange( slice.GetCurvePixels(ref img) );
                                }
                        }
                        x_img += ratio;
                    }
                    y_img += ratio;
                }
            }
            else if (img.width < width) // matrix is larger than image
            {
                int maxx = img.width;
                int maxy = img.height;
                int x_mat = 0;
                int y_mat = 0;

                if ((width % img.width) != 0)
                    throw new TPCROIException("Image and matrix ratio cannot have remainder.");

                int ratio = (width / img.width);
                int ratio_power = ratio * ratio;
                for (int y = 0; y < maxy; y++)
                {
                    x_mat = 0;
                    for (int x = 0; x < maxx; x++)
                    {
                        // We make the transforming of 2d slice to 
                        // 3d space (transaxial, sagittal and depeths, etc..)
                        slice.X2D = x;
                        slice.Y2D = y;
                        List<ROIValuePoint> points = slice.GetCurvePixels(ref img);

                        for (int zy = 0; zy < ratio; zy++) for (int zx = 0; zx < ratio; zx++)
                            {
                                if (cells[x_mat + zx][y_mat + zy] > 0)
                                {
                                    result.AddRange(points);
                                    //result.Add( new ROIValuePoint( slice.X,slice.Y, slice.Z, imgfloat ) );
                                }
                            }
                        x_mat += ratio;
                    }
                    y_mat += ratio;
                }
            }
            else // image and matrix are same size
            {
                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        if (cells[x][y] > 0)
                        {
                            // We make the transforming of 2d slice to 
                            // 3d space (transaxial, sagittal and depeths, etc..)
                            slice.X2D = x;
                            slice.Y2D = y;
                            result.AddRange(slice.GetCurvePixels(ref img));
                        }
                    }
                }
            }

            return result;
        }

    }
}
