﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Matrix number of ROI
    /// </summary>
    public class Matrix_Number
    {
        /// <summary>
        /// Matrix number information in 32 bit uint
        /// </summary>
        public uint Value;

        /// <summary>
        /// Frame part of matrix number
        /// </summary>
        public int Frame
        {
            get
            {
                return (int)(Value & 0x1FF);
            }
            set
            {
                if (value < 0 || value > 0x1FF) throw new TPCROIException("MatrixNumber: Frame index out of bounds (0-1FF).");
                Value &= ~((uint)0x1FF);
                Value |= (uint)value;
            }
        }

        /// <summary>
        /// Plane part of matrix number
        /// </summary>
        public int Plane
        {
            get
            {
                //get { return (int)(((matrix_id >> 16) & 0xFF) + ((matrix_id >> 1) & 0x300)); }
                return (int)(((Value >> 16) & 0xFF) + ((Value >> 1) & 0x300));
            }
            set
            {
                if (value < 0 || value > 0x3FF) throw new TPCROIException("MatrixNumber: Plane index out of bounds (0-3FF).");
                Value &= ~(((uint)0xFF << 16) + ((uint)0x3 << 9));
                Value |= (((uint)value & 0xFF) << 16) + (((uint)value & 0x300) << 1);
            }
        }

        /// <summary>
        /// Bed part of matrix number
        /// </summary>
        public int Bed
        {
            get
            {
                return (int)((Value >> 12) & 0xF);
            }
            set
            {
                if (value < 0 || value > 0xF) throw new TPCROIException("MatrixNumber: Bed index out of bounds (0-F).");
                Value &= ~((uint)0xF << 12);
                Value |= ((uint)value << 12);
            }
        }

        /// <summary>
        /// Gate part of matrix number
        /// </summary>
        public int Gate
        {
            get
            {
                return (int)((Value >> 24) & 0x3F);
            }
            set
            {
                if (value < 0 || value > 0x3F) throw new TPCROIException("MatrixNumber: Gate index out of bounds (0-3F).");
                Value &= ~((uint)0x3F << 24);
                Value |= ((uint)value << 24);
            }
        }

        /// <summary>
        /// Data part of matrix number
        /// </summary>
        public int Data
        {
            get
            {
                //get { return (int)(((matrix_id >> 30) & 0x3) + ((matrix_id >> 9) & 0x4)); }
                return (int)(((Value >> 30) & 0x3) + ((Value >> 9) & 0x4));
            }
            set
            {
                if (value < 0 || value > 0x7) throw new TPCROIException("MatrixNumber: Data index out of bounds (0-7).");
                Value &= ~(((uint)0x3 << 30) + (((uint)0x4) << 9));
                Value |= (((uint)value << 30) + ((uint)(value & (uint)0x4) << 9));
            }
        }

        /// <summary>
        /// Constructs new matrix number with given 32 bit uint
        /// </summary>
        /// <param name="value">32 bit uint value containing the matrix number information</param>
        public Matrix_Number(uint value)
        {
            if (value < 0 || value > 0xFFFFFFFF) throw new TPCROIException("MatrixNumber: Index out of bounds.");
            Value = (uint)value;
        }

        /// <summary>
        /// Creates matrix number from given information
        /// </summary>
        /// <param name="frame">Frame part of matrix number</param>
        /// <param name="plane">Plane part of matrix number</param>
        /// <param name="gate">Gate part of matrix number</param>
        /// <param name="data">Data part of matrix number</param>
        /// <param name="bed">Bed part of matrix number</param>
        public Matrix_Number(int frame, int plane, int gate, int data, int bed)
        {
            Value = 0;

            Frame = frame;
            Plane = plane;
            Gate = gate;
            Data = data;
            Bed = bed;
        }

        /// <summary>
        /// Matrix object as string
        /// </summary>
        /// <returns>MatrixNumber as string</returns>
        public override string ToString()
        {
            return
                "[Value=" + Value + "  " +
                "Frame=" + Frame + "  " +
                "Plane=" + Plane + "  " +
                "Gate=" + Gate + "  " +
                "Data=" + Data + "  " +
                "Bed=" + Bed + "]";
        }
    }
}