﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Mexel inside mesh
    /// </summary>
    public class Mexel
    {
        /// <summary>
        /// center
        /// </summary>
        public Point p = new Point();
        /// <summary>
        /// adjanced faces (unordered)
        /// </summary>
        public int[] triface = new int[TriMeshVOI.Mexel_N];
        /// <summary>
        /// neighbouring mexels
        /// </summary>
        public int[] Ng = new int[TriMeshVOI.Mexel_N];
        /// <summary>
        /// Default constructor. All index values are set to -1.
        /// </summary>
        public Mexel()
        {
            for (int i = 0; i < TriMeshVOI.Mexel_N; i++)
            {
                Ng[i] = -1;
                triface[i] = -1;
            }
        }
        /// <summary>
        /// Flags CHANGE MORE CONVENIENT
        /// </summary>
        public byte flag = 0;
        /// <summary>
        /// Creates mexel with initial location
        /// </summary>
        /// <param name="p">initial location</param>
        public Mexel(Point p)
        {
            this.p.x = p.x;
            this.p.y = p.y;
            this.p.z = p.z;
            for (int i = 0; i < TriMeshVOI.Mexel_N; i++)
            {
                Ng[i] = -1;
                triface[i] = -1;
            }
            this.flag = 0;
        }
        /// <summary>
        /// Returns true if mexel has face as a neighbour.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if face is found</returns>
        public bool HasFace(int j)
        {
            for (int i = 0; i < triface.Length; i++)
            {
                if (triface[i] == -1) break;
                if (triface[i] == j) return true;
            }
            return false;
        }
        /// <summary>
        /// Gets neighbour face index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>face index at j</returns>
        public int GetFace(int j)
        {
            if (triface[j] == -1) throw new TriMeshException("Face index does not exist.");
            return triface[j];
        }
        /// <summary>
        /// Resolves if neighbour face index exists for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if face exsits, or false if face does not exist</returns>
        public bool FaceExists(int j)
        {
            return triface[j] != -1;
        }
        /// <summary>
        /// Sets neighbour face index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <param name="value">new index value that is placed into j'th position</param>
        public void SetFace(int j, int value)
        {
            if (HasFace(value))
                throw new TriMeshException("Face already exists.");
            triface[j] = value;
        }
        /// <summary>
        /// Adds neighbour face index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        public void AddFace(int j)
        {
            for (int i = 0; i < triface.Length; i++)
            {
                if (triface[i] == -1)
                {
                    triface[i] = j;
                    return;
                }
                if (triface[i] == j)
                {
                    throw new TriMeshException("Face already exists.");
                }
            }
            throw new TriMeshException("No space for new face. Constant Mexel.Mexel_N is too small.");
        }
        /// <summary>
        /// Removes neighbour face index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if removal was success</returns>
        public bool RemoveFace(int j)
        {
            for (int i = 0; i < triface.Length; i++)
            {
                if (triface[i] == j)
                {
                    for (; i < triface.Length - 1; i++)
                    {
                        triface[i] = triface[i + 1];
                    }
                    triface[triface.Length - 1] = -1;
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// Removes neighbour mexel index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if removal was success</returns>
        public bool RemoveNg(int j)
        {
            for (int i = 0; i < Ng.Length; i++)
            {
                if (Ng[i] == j)
                {
                    for (; i < Ng.Length - 1; i++)
                    {
                        Ng[i] = Ng[i + 1];
                    }
                    Ng[Ng.Length - 1] = -1;
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// Adds neighbour mexel index for this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if add was success</returns>
        public bool AddNg(int j)
        {
            for (int i = 0; i < Ng.Length; i++)
            {
                if (Ng[i] == -1)
                {
                    Ng[i] = j;
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// Returns mexel index of neighbour of this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>index of neighbour of this mexel</returns>
        public int GetNg(int j)
        {
            for (int i = 0; i < Ng.Length; i++)
            {
                if (Ng[i] == j)
                {
                    return i;
                }
            }
            throw new TriMeshException("Index " + j + " is not neighbour of this mexel.");
        }
        /// <summary>
        /// Returns true if mexel index is neighbour of this mexel.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if index is neighbour of this mexel</returns>
        public bool HasNg(int j)
        {
            for (int i = 0; i < Ng.Length; i++)
            {
                if (Ng[i] == j)
                {
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// String representation of this object
        /// </summary>
        /// <returns>object information as string</returns>
        public override string ToString()
        {
            string r = "Mexel[" + Ng[0];
            for (int i = 1; i < TriMeshVOI.Mexel_N; i++)
            {
                r = r + "," + Ng[i];
            }
            r = r + "]";
            return r + p;
        }
    }
}