﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;

namespace TPClib.ROI
{
    /// <summary>
    /// Class for ROI mask testing
    /// </summary>
    [TestFixture]
    public class NUnitTestBench_ROIMasks
    {
        private String Ppath = "P:\\Data\\ROI\\";

        /// <summary>
        /// Tests regions of ImageToolMask
        /// </summary>
        [Test]
        public void Test1_0_ImageTool_Regions()
        {
            ImageToolROIFile file = new ImageToolROIFile();
            file.ReadFile(Ppath + "ImageTool\\roi_regional.roi");

            for (int w = 0; w < file.VOIs[0].Count; w++)
            {
                ROI roi = file.VOIs[0].GetRoi(w);
                roi.Fill(7, 8);

                Console.WriteLine(file.VOIs[0].Count);
                ROI_Matrix matrix = roi.GetMatrix();
                Console.WriteLine("ROI=" + roi.ToString());

                Console.WriteLine(matrix.ToString());
                Assert.IsTrue(matrix.Equals(transpose(correctRegional_ImageTool[w])));
            }
        }

        /// <summary>
        /// Tests overlaps of ImageToolMask
        /// </summary>
        [Test]
        public void Test1_1_ImageTool_Overlap()
        {
            ROI.Zoom_Method = ZoomMethod.Fixed;
            ImageToolROIFile file = new ImageToolROIFile();
            file.ReadFile(Ppath + "ImageTool\\roi_overlap.roi");

            for (int w = 0; w < file.VOIs[0].Count; w++)
            {
                ROI roi = file.VOIs[0].GetRoi(w);
                roi.Fill(14, 8);

                Console.WriteLine(file.VOIs[0].Count);
                ROI_Matrix matrix = roi.GetMatrix();
                Console.WriteLine("ROI=" + roi.ToString());

                Console.WriteLine(matrix.ToString());
                Assert.IsTrue(matrix.Equals(transpose(correctOverlap_ImageTool[w])));
            }
        }

        /// <summary>
        /// Tests regions of ImadeusMask
        /// </summary>
        [Test]
        public void Test2_0_Imadeus_Regions()
        {
            Console.WriteLine("TESTING IMADEUS MASK USING MEAN:");
            Image.Ecat7File filee = new TPClib.Image.Ecat7File(Ppath +"Imadeus\\image.v");
            //Image.Analyze75File filee = new Image.Analyze75File(Ppath + "Imadeus\\analyze\\image");

            filee.ReadFile();

            Image.Image image = filee.image;

            ImadeusROIFile file = new ImadeusROIFile();
            file.ReadFile(Ppath + "Imadeus\\roi_regional.roi");

            for (int w = 0; w < file.VOIs.Count; w++)
            {
                VOI voi = file.VOIs[w];
                voi.Fill(128, 128, 1);

                ROI_Matrix matrix = voi.GetMatrix().GetRoiMatrix(0);
                Console.WriteLine("ROI=" + voi.Name);

                //Console.WriteLine(matrix.ToString());

                Slice slice = Slice.Create(Slice.Direction.Transaxial, 0, 128, 128, 1);
                double[] mean = matrix.Calculate(ref image, slice, ROICalculator.func.Mean );
                Console.WriteLine("Mean   =" +  mean[0] );
                Console.WriteLine("Correct=" + correctRegional_ImadeusMean[w]);
                Assert.IsTrue( CloseEnough( mean[0], correctRegional_ImadeusMean[w] ) );
            }
        }

        /// <summary>
        /// Tests overlaps of ImadeusMask
        /// </summary>
        [Test]
        public void Test2_1_Imadeus_Overlaps()
        {
            Console.WriteLine("TESTING IMADEUS MASK USING MEAN:");
            Image.Ecat7File filee = new TPClib.Image.Ecat7File(Ppath + "Imadeus\\image.v");
            filee.ReadFile();

            Image.Image image = filee.image;

            ImadeusROIFile file = new ImadeusROIFile();
            file.ReadFile(Ppath + "Imadeus\\roi_overlap.roi");

            for (int w = 0; w < file.VOIs.Count; w++)
            {
                VOI voi = file.VOIs[w];
                voi.Fill(128, 128, 1);

                ROI_Matrix matrix = voi.GetMatrix().GetRoiMatrix(0);
                Console.WriteLine("ROI=" + voi.Name);

                Console.WriteLine(matrix.ToString());

                Slice slice = Slice.Create(Slice.Direction.Transaxial, 0, 128, 128, 1);
                double[] mean = matrix.Calculate(ref image, slice, ROICalculator.func.Mean);
                Console.WriteLine("Mean   =" + mean[0]);
                Console.WriteLine("Correct=" + correctOverlap_ImadeusMean[w]);
                Assert.IsTrue(CloseEnough(mean[0], correctOverlap_ImadeusMean[w]));
            }
        }

        /// <summary>
        /// Testing imagetool fixed zoom method
        /// </summary>
        [Test]
        public void Test3_1_TestImageTool_FixedZoom()
        {
            ROIObject.Zoom_Method = ZoomMethod.Fixed;

            //string imagefile = @"P:\source\input_from_image\InputFromImage\Example_LiverFDG\ep0082dy1_mrp.v";
            string roifile = @"P:\source\input_from_image\InputFromImage\Example_LiverFDG\ep0082dy1_mrp.roi";
            //string roifile = @"P:\data\roi\imagetool\zoom.roi";

            ImageToolROIFile file = new ImageToolROIFile();
            file.ReadFile(roifile);

            //Image.Ecat7File ifile = new TPClib.Image.Ecat7File(imagefile);
            //ifile.ReadFile();

            

            int planes = 15;
            int imageWidth = 128;
            int imageHeight = 128;
            //int frames = 31;
            
            file.VOIs[0].Fill(imageWidth,imageHeight,planes);



            // Getting the correct values calculated with imgpext
            FileStream filestream = new FileStream(@"p:\data\ROI\imagetool\test_leikattu.txt", FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadToEnd();
            StringReader strReader = new StringReader(text);
            filestream.Close();

            VOI_Matrix original = new VOI_Matrix(imageWidth, imageHeight, planes);
            original.Clear();

            string line = "";

            do
            {
                line = strReader.ReadLine();
                if (line != null)
                {
                    // Reading the imgpext output from file to get correct pixel positions
                    string[] words = line.Split(new char[] { '[', ']', ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    original[Convert.ToInt32(words[2]), Convert.ToInt32(words[1]), Convert.ToInt32(words[0])-1] = 1;
                }
            }
            while (line != null);

            strReader.Close();
            int errors = 0;
            StringBuilder str = new StringBuilder();

            Console.WriteLine();
            for (int z = 0; z < 2; z++)
            {
                for (int y = 0; y < imageHeight; y++) 
                {
                    for (int x = 0; x < imageWidth; x++)
                    {
                        byte val1 = original[x, y, z]; // The correct values
                        byte val2 = file.VOIs[0].GetMatrix()[x, y, z]; // Values to be tested
                        
                        // Printing the matrixes to screen. Differences are printed with different character
                        if (val1 == 1 && val2 == 1) str.Append("X");
                        else if (val1 == 0 && val2 == 0) str.Append("O");
                        else if (val1 == 1 && val2 == 0) { str.Append("1"); errors++; }
                        else if (val1 == 0 && val2 == 1) { str.Append("2"); errors++; }
                    }
                    str.Append("\r\n");
                }
                str.Append("\r\n\r\n");
            }

            Console.WriteLine(str.ToString());
            Console.WriteLine("Errors: "+errors);
            Assert.IsTrue(errors == 0);
        }

        #region priavate members

        private bool CloseEnough( double first, double second )
        {
            if ( Double.IsNaN( first ) && (Double.IsNaN(second) || second == 0.0d) ) return true;
            if ( Double.IsNaN(second ) && (Double.IsNaN(first) || first == 0.0d)) return true;
            if (Math.Abs(first - second) < 0.000001d) return true;
            else
            {
                Console.WriteLine("Values not close enough:"+ first + " and " + second);
                return false;
            }
        }

        /// <summary>
        /// Imadeus means calculated by Imadeus. (from roi_regional.roi)
        /// </summary>
        private double[] correctRegional_ImadeusMean = new double[] {
            2.67487487162e+01,
            2.84998626402e+01,
            0.00000000000e+00,
            3.36987517616e+01,
            4.82789696502e+01,
            4.88560572247e+01,
            4.17421546743e+01,
            0.00000000000e+00
        };

        /// <summary>
        /// Imadeus means calculated by Imadeus. (from roi_overlap.roi)
        /// </summary>
        private double[] correctOverlap_ImadeusMean = new Double[] {
            3.31102090888e+01,
            3.39993489068e+01,
            4.57263404829e+01,
            0.00000000000e+00,
            4.59989396851e+01,
            3.24990081182e+01
        };

        /// <summary>
        /// Imagetool matrix claculated by imgpext.exe
        /// </summary>
        private byte[][][] correctRegional_ImageTool = new byte[][][] { 
        new byte[][] { // ROI1
            new byte[] {1, 1, 1, 0, 0, 0, 0},
            new byte[] {1, 1, 1, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI2
            new byte[] {0, 0, 0, 1, 1, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI3
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI4
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI5
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 1, 1, 1},
            new byte[] {0, 0, 0, 0, 1, 1, 1},
            new byte[] {0, 0, 0, 0, 1, 1, 1},
        },
        new byte[][] { // ROI6
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {1, 1, 1, 0, 0, 0, 0},
            new byte[] {1, 1, 1, 0, 0, 0, 0},
            new byte[] {1, 1, 1, 1, 0, 0, 0},
        },
        new byte[][] { // ROI7
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
            new byte[] {1, 1, 1, 1, 1, 1, 1},
        },
        new byte[][] { // ROI8
            new byte[] {1, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0},
        }
        };

        private byte[][][] correctOverlap_ImageTool = new byte[][][] { 
        new byte[][] { // ROI1
            new byte[] {0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI2
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI3
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI4
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        } ,
        new byte[][] { // ROI5
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        },
        new byte[][] { // ROI6
            new byte[] {0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
            new byte[] {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        }
        };

        private byte[][] transpose(byte[][] obj)
        {
            int s_width = obj[0].Length;
            int s_height = obj.Length;
            byte[][] result = new byte[s_width][];

            for (int y = 0; y < s_width; y++)
            {
                result[y] = new byte[s_height];

                for (int x = 0; x < s_height; x++)
                {
                    result[y][x] = obj[x][y];
                }
            }
            return result;
        }
        #endregion
    }
}
