﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Curve;
using TPClib.Image;
using NUnit.Framework;

namespace TPClib.ROI
{
    /// <summary>
    /// Class for ImadeusROIFile testing
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_ImadeusROIFile
    {
        private String Ppath = "P:\\Data\\ROI\\Imadeus\\";
        private String Cpath = "ROITest\\Imadeus\\";

        /// <summary>
        /// Read Imadeus ROI file from disk and Write it by another name. Read the saved file
        /// and calculate means. Compare mean results to means calculated by Imadeus
        /// </summary>
        [Test]
        public void Test1_0_ReadWrite_File()
        {
            System.IO.Directory.CreateDirectory(Cpath);

            Console.WriteLine("TESTING IMADEUS ROI FILE READING AND WRITING USING MEAN:");
            Image.Ecat7File filee = new TPClib.Image.Ecat7File(Ppath + "image.v");
            filee.ReadFile();
            Image.Image image = filee.image;

            ROIFile file = ROIFile.CreateROIFileObject(Ppath + "roi_regional.roi");
            file.ReadFile(Ppath + "roi_regional.roi");

            //ImadeusROIFile file2 = new ImadeusROIFile();
            //file2.Data = file.Data;
            file.WriteFile(Cpath + "Test1_0.roi");

            // Now we have file that is gone through load and save
            file.ReadFile(Cpath + "Test1_0.roi");

            Assert.IsTrue( file.VOIs.Count == 8 );

            for (int w = 0; w < file.VOIs.Count; w++)
            {
                VOI voi = file.VOIs[w];
                voi.Fill(128, 128, 1);

                ROI_Matrix matrix = voi.GetMatrix().GetRoiMatrix(0);
                Console.WriteLine("ROI=" + voi.Name);

                Console.WriteLine(matrix.ToString());

                Slice slice = Slice.Create(Slice.Direction.Transaxial, 0, 128, 128, 1);
                double[] mean = matrix.Calculate(ref image, slice, ROICalculator.func.Mean);
                Console.WriteLine("Mean   =" + mean[0]);
                Console.WriteLine("Correct=" + correctRegional_ImadeusMean[w]);
                Assert.IsTrue(CloseEnough(mean[0], correctRegional_ImadeusMean[w]));
            }

        }

        /// <summary>
        /// Tests complicated ROI file that contains many regions, VOIs, Combinations and Creator 
        /// </summary>
        [Test]
        public void Test2_2_Imadeus_ComplicatedROI()
        {
            ImadeusROIFile file = new ImadeusROIFile();
            file.ReadFile(Ppath + "TestiPotilas.roi");
            file.WriteFile(Cpath + "Test2_2.roi");

            DFTCurveFile dft_file = DFTCurveFile.ReadFile(Ppath + "TestiPotilas.dft");
            TAC tac = (TAC)dft_file.Curves.GetColumn(0);

            Analyze75File analyze_file = new Analyze75File(Ppath + "testipotilasdy1");
            analyze_file.ReadFile();
            DynamicImage image = new DynamicImage( analyze_file.image );
            
            List<Double> results = new List<double>();

            Double[] res = file.VOIs[0].Calculate(ref analyze_file.image, ROICalculator.func.Mean);
            /*
            List<ROIValuePoint> points = file.VOIs[0].GetMatrix().GetPixels(ref analyze_file.image);
            Console.WriteLine("Points:"+points.Count);
            foreach (ROIValuePoint p in points)
            {
                Console.WriteLine(p.ToString());
            }*/

            for (int i = 0; i < image.frames; i++)
            {
                //Image.Image img = image.getFrame(i);
                
                Console.WriteLine("ROI="+file.VOIs[0].Name);
                Console.WriteLine("Calculated=" + res[i]);
                Console.WriteLine("From DFT  =" + tac[i].Value+"\r\n" );
                //results.Add( res );
            }
        }

        /// <summary>
        /// Imadeus means calculated by Imadeus. (from roi_regional.roi)
        /// </summary>
        private double[] correctRegional_ImadeusMean = new double[] {
            2.67487487162e+01,
            2.84998626402e+01,
            0.00000000000e+00,
            3.36987517616e+01,
            4.82789696502e+01,
            4.88560572247e+01,
            4.17421546743e+01,
            0.00000000000e+00
        };

        private bool CloseEnough(double first, double second)
        {
            if (Double.IsNaN(first) && (Double.IsNaN(second) || second == 0.0d)) return true;
            if (Double.IsNaN(second) && (Double.IsNaN(first) || first == 0.0d)) return true;
            if (Math.Abs(first - second) < 0.000001d) return true;
            else return false;
        }

    }
}
