﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// ROI parent abstract class that has common ROI abilities.
    /// </summary>
    public abstract class ROI : ROIObject
    {
        /// <summary>
        /// Every point will be scaled with this factor.
        /// </summary>
        public override Double ZoomFactor
        {
            get { return zoomFactor; }
            set { if (zoomFactor != value) { zoomFactor = value; notReady = true; } }
        }

        /// <summary>
        /// This indicates how coordinates are threated. If percent, all the coordinates must
        /// be given in values between 0-1. Resulting points of fill are always in pixel
        /// coordinates.
        /// </summary>
        public Coordinate_System CoordinateSystem
        {
            get { return coordinatesystem; }
            set { 
                if (coordinatesystem != value) {
                    /*
                    // We must have width and height information if we want to use percents
                    if (obj_matrix == null) throw new TPCROIException("Cannot convert coordinate system. No matrix dimensions found.");

                    if (value == Coordinate_System.Pixel)
                    {
                        // We are changing coordinate system from percent to pixel
                        this.x = this.x * width;
                        this.y = this.y * height;
                        this.shapewidth = this.shapewidth * width;
                        this.shapeheight = this.shapeheight * height;
                    }
                    else
                    {
                        // We are changing coordinate system from pixel to percent
                        this.x = this.x / width;
                        this.y = this.y / height;
                        this.shapewidth = this.shapewidth / width;
                        this.shapeheight = this.shapeheight / height;
                    }
                    */
                    coordinatesystem = value;
                    notReady = true; 
                } 
            }
        }


        /// <summary>
        /// The x coordinate of the ROI.
        /// </summary>
        public double X
        {
            get { return x; }
            set { if (x != value) { x = value; notReady = true; } }
        }

        /// <summary>
        /// The y coordinate of the ROI.
        /// </summary>
        public double Y
        {
            get { return y; }
            set { if (y != value) { y = value; notReady = true; } }
        }

        /// <summary>
        /// The z coordinate of the ROI.
        /// </summary>
        public double Z
        {
            get { return z; }
            set { if (z != value) { z = value; } }
        }

        /// <summary>
        /// Gets the matrix of the ROI/VOI that has been generated in Fill(). If the
        /// geometry of ROI has been changed from last Fill() call, null will be returned
        /// </summary>
        /// <returns>Matrix generated in fill.</returns>
        public new ROI_Matrix GetMatrix()
        {
            if (notReady)
            {
                if (obj_matrix != null) Fill(obj_matrix.Width, obj_matrix.Height);
                else return null;
            }
            return (base.obj_matrix as ROI_Matrix);
        }

        /// <summary>
        /// The rotate angle of the ROI.
        /// </summary>
        public double Angle
        {
            get { return angle; }
            set 
            { 
                if (angle != value)
                {
                    angle = value;
                    notReady = true;
                } 
            }
        }

        /// <summary>
        /// Flips the ROI along x-axis.
        /// </summary>
        public bool FlipX
        {
            get { return flipX; }
            set { if (flipX != value) { flipX = value; notReady = true; } }
        }

        /// <summary>
        /// Flips the ROI along y-axis.
        /// </summary>
        public bool FlipY
        {
            get { return flipY; }
            set { if (flipY != value) { flipY = value; notReady = true; } }
        }

        /// <summary>
        /// Calculates the value of some function inside of ROI
        /// </summary>
        /// <param name="image">Image reference</param>
        /// <param name="function">calculation method</param>
        /// <returns>Value of function inside the ROI</returns>
        public double[] Calculate(ref TPClib.Image.Image image, ROICalculator.func function )
        {
            UpdateMatrix(image.width, image.height);

            // we create the slice, because we are calculating function of 2D roi from 3D image
            Slice slice = Slice.Create(this.Header.Direction, (int)this.z, image.width, image.height, image.planes);
            
            return (obj_matrix as ROI_Matrix).Calculate( ref image, slice, function );
        }

        /// <summary>
        /// Gets all the points(pixels) of image inside the roi. Only the pixels inside the slice
        /// are included based on Header.Dirction and z fields.
        /// </summary>
        /// <param name="image">3D image, where the pixel values are retrieved</param>
        /// <returns>List of ROI points.</returns>
        public List<ROIValuePoint> GetPixels(ref TPClib.Image.Image image)
        {
            UpdateMatrix(image.width, image.height);

            // we create the slice, because we are calculating function of 2D roi from 3D image
            Slice slice = Slice.Create(this.Header.Direction, (int)this.z, image.width, image.height, image.planes);

            return (obj_matrix as ROI_Matrix).GetPixels(ref image, slice);
        }

        /// <summary>
        /// Creates new matrix if necessary. Takes width,height as input
        /// </summary>
        /// <param name="dimensions">X,Y Dimensions are needed</param>
        protected override void CreateMatrix(params int[] dimensions)
        {
            if (dimensions.Length < 2) throw new TPCROIException("CreateMatrix: Too few parameters. X and Y must be defined.");

            obj_matrix = new ROI_Matrix(dimensions[0], dimensions[1] );

            // In percent coordinate system, all coordinates must be calculated to match
            // the new matrix dimensions
            /*
            if (this.coordinatesystem == Coordinate_System.Percent)
            {
                ReCalculate();
            }*/

            if (VOIParent != null) VOIParent.NeedsUpDate = true;
        }

        /// <summary>
        /// Flips the ROI along z-axis.
        /// </summary>
        public bool FlipZ
        {
            get { return flipZ; }
            set { if (flipZ != value) { flipZ = value; notReady = true; } }
        }

        /// <summary>
        /// Pointer to VOI class, where this ROI belongs.
        /// </summary>
        public VOI VOIParent;

        /// <summary>
        /// Gets the transformed(flipped,rotated,scaled) point information of ROI
        /// </summary>
        public List<ROIPoint> Points
        {
            get
            {
                if ( !notReady ) return this.points;
                if (obj_matrix != null)
                {
                    Fill(obj_matrix.Width, obj_matrix.Height);
                    return this.points;
                }
                if (this.coordinatesystem == Coordinate_System.Pixel)
                {
                    ReCalculate();
                    return this.points;
                }

                throw new TPCROIException("Cannot get points");
            }
        }

        /// <summary>
        /// All ROI information that will not affect to the shape is included here.
        /// </summary>
        public ROI_Header Header;

        
        /// <summary>
        /// Fills matrix. All pixels inside ROI area are filled with 1.
        /// </summary>
        protected override void FillMatrix()
        {
            ReCalculate();

            if (this.fill_method == Fill_Method.ImageTool)
            {
                ROI_Matrix rm = obj_matrix as ROI_Matrix;
                ImageToolMask.Fill( rm, points, x, y);
            }
            else if (this.fill_method == Fill_Method.Imadeus)
            {
                ROI_Matrix rm = obj_matrix as ROI_Matrix;
                ImadeusMask.Fill(rm, points);
            }
        }

        /// <summary>
        /// Default constructor
        /// </summary>
        public ROI() : base()
        {
            init();
        }
        /// <summary>
        /// Cosntructor with values
        /// </summary>
        /// <param name="name">ROI name</param>
        /// <param name="zoomFactor">zooming factor</param>
        /// <param name="MatNum">matrix number</param>
        /// <param name="roiType"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="roiNum"></param>
        public ROI(String name, Double zoomFactor, Matrix_Number MatNum, ROI_Type roiType,
                    Double x, Double y, int roiNum) : base()
        {
            init();

            this.Header.ROIName = name;
            this.zoomFactor = zoomFactor;
            this.Header.MatrixNumber = new Matrix_Number(MatNum.Value);
            this.Header.RoiType = roiType;
            this.x = x;
            this.y = y;
            this.Header.ROINumber = roiNum;
        }
        /// <summary>
        /// Copy constructor for ROI
        /// </summary>
        /// <param name="roi">copy data</param>
        public ROI(ROI roi) : base()
        {
            init();
            this.Header = new ROI_Header(roi.Header);

            this.zoomFactor = roi.zoomFactor;
            this.x = roi.x;
            this.y = roi.y;
            this.width = roi.width;
            this.height = roi.height;

            this.angle = roi.angle;
            this.x_axisX = roi.x_axisX;
            this.x_axisY = roi.x_axisY;
            this.y_axisX = roi.y_axisX;
            this.y_axisY = roi.y_axisY;

            this.flipX = roi.flipX;
            this.flipY = roi.flipY;
            this.flipZ = roi.flipZ;
        }
        /// <summary>
        /// Constructor with parameters
        /// </summary>
        /// <param name="x">x-coordinate in plane</param>
        /// <param name="y">y-coordinate in plane</param>
        /// <param name="zoomFactor">zooming factor</param>
        /// <param name="header">ROI header information</param>
        public ROI(Double x, Double y, Double zoomFactor, ROI_Header header) : base()
        {
            init();

            this.Header = new ROI_Header(header);
            this.x = x;
            this.y = y;
            this.zoomFactor = zoomFactor;
        }

        /// <summary>
        /// Calculates the transformed(zoomed,rotated) points. This parent function calculates
        /// only the y and x component vectors.
        /// </summary>
        protected virtual void ReCalculate()
        {
            // The matrix must be filled again
            notReady = true;

            if (ROIObject.Zoom_Method == ZoomMethod.Fixed)
            {
                // In fixed method the matrix will be kept as large as the image
                if (zoomFactor == 0.0d) zoomFactor = 1.0d;
                factorX = 1.0d;// / zoomFactor;
                factorY = 1.0d;// / zoomFactor;
            }
            else
            {
                // In Enlarge method, the result matrix will be enlarged to correspond the
                // zoom factor. 
                factorX = 1.0d * zoomFactor;
                factorY = 1.0d * zoomFactor;
            }

            // x-component vector for transform. All x cordinates are multiplied with this
            x_axisX = Math.Cos(angle) * factorX;
            x_axisY = Math.Sin(angle) * factorY;
            if (FlipX) { x_axisX = x_axisX * -1.0d; x_axisY = x_axisY * -1.0d; }

            // y-component vector for transform. All y cordinates are multiplied with this
            y_axisX = Math.Cos(angle + (Math.PI / 2.0d)) * factorX;
            y_axisY = Math.Sin(angle + (Math.PI / 2.0d)) * factorY;
            if (FlipY) { y_axisX = y_axisX * -1.0d; y_axisY = y_axisY * -1.0d; }

            // If we are using percents, we need the matrix, where to compare the percents
            if (coordinatesystem == Coordinate_System.Percent)
            {
                if (obj_matrix != null)
                {
                    x_axisX *= (double)obj_matrix.Width;
                    x_axisY *= (double)obj_matrix.Width;

                    y_axisX *= (double)obj_matrix.Height;
                    y_axisY *= (double)obj_matrix.Height;

                    factorX *= (double)obj_matrix.Width;
                    factorY *= (double)obj_matrix.Width;
                }
            }
        }
        /// <summary>
        /// List of ROI's edge points
        /// </summary>
        protected List<ROIPoint> points;
        /// <summary>
        /// True if image was flipped against x-axis while drawing the ROI.
        /// </summary>
        protected bool flipX;
        /// <summary>
        /// True if image was flipped against y-axis while drawing the ROI.
        /// </summary>
        protected bool flipY;
        /// <summary>
        /// True if image was flipped against z-axis while drawing the ROI.
        /// </summary>
        protected bool flipZ;
        /// <summary>
        /// Scaling factor for x-axis.
        /// </summary>
        protected double factorX;
        /// <summary>
        /// Scaling factor for y-axis.
        /// </summary>
        protected double factorY;
        /// <summary>
        /// Rotation angle around ROI plane normal.
        /// </summary>
        protected double angle;
        /// <summary>
        /// X component of x-axis factor vector.
        /// </summary>
        protected double x_axisX;
        /// <summary>
        /// Y component of x-axis factor vector.
        /// </summary>
        protected double x_axisY;
        /// <summary>
        /// X component of y-axis factor vector.
        /// </summary>
        protected double y_axisX;
        /// <summary>
        /// Y component of y-axis factor vector.
        /// </summary>
        protected double y_axisY;
        /// <summary>
        /// ROI x-coordinate
        /// </summary>
        protected Double x;
        /// <summary>
        /// ROI y-coordinate
        /// </summary>
        protected Double y;
        /// <summary>
        /// ROI z-coordinate
        /// </summary>
        protected Double z;
        /// <summary>
        /// Largest width of ROI.
        /// </summary>
        protected Double shapewidth;
        /// <summary>
        /// Largest height of ROI.
        /// </summary>
        protected Double shapeheight;
        /// <summary>
        /// Coordinate system type.
        /// </summary>
        private Coordinate_System coordinatesystem;

        /// <summary>
        /// The width of the ROI shape.
        /// </summary>
        public double ShapeWidth
        {
            get { return shapewidth; }
            set { if (shapewidth != value) { shapewidth = value; notReady = true; } }
        }

        /// <summary>
        /// The height of the ROI shape.
        /// </summary>
        public double ShapeHeight
        {
            get { return shapeheight; }
            set { if (shapeheight != value) { shapeheight = value; notReady = true; } }
        }

        /// <summary>
        /// Calculates the rotated, flipped, zoomed and aligned x and x-cordinate
        /// </summary>
        /// <param name="xx">X value</param>
        /// <param name="yy">Y value</param>
        /// <returns>The rotated x-cordinate</returns>
        protected double calcX(double xx, double yy)
        {
            /*if (fill_method == Fill_Method.ImageTool) return (x_axisX * xx + y_axisX * yy) + (double)((int)x) * factorX;
            else if (fill_method == Fill_Method.Imadeus) return Convert.ToInt32((x_axisX * xx + y_axisX * yy) + x * factorX);*/
            return (x_axisX * xx + y_axisX * yy) + x * factorX;
        }

        /// <summary>
        /// Calculates the rotated, flipped, zoomed and aligned x and y-cordinate
        /// </summary>
        /// <param name="xx">X value</param>
        /// <param name="yy">Y value</param>
        /// <returns>The rotated y-cordinate</returns>
        protected double calcY(double xx, double yy)
        {
            /*if (fill_method == Fill_Method.ImageTool) return (x_axisY * xx + y_axisY * yy) + (double)((int)y) * factorY;
            else if (fill_method == Fill_Method.Imadeus) return Convert.ToInt32((x_axisY * xx + y_axisY * yy ) + y * factorY);
            else */return (x_axisY * xx + y_axisY * yy) + y * factorY;
        }

        /// <summary>
        /// Creates replica of ROI object
        /// </summary>
        /// <returns></returns>
        public abstract ROI Clone();

        /// <summary>
        /// Writes the contents of ROI to String. 
        /// </summary>
        /// <returns>Roi information as String</returns>
        public string MatrixToString()
        {
            StringBuilder str = new StringBuilder();

            for (int y = 0; y < obj_matrix.Height; y++)
            {
                for (int x = 0; x < obj_matrix.Width; x++)
                {
                    str.Append((obj_matrix as ROI_Matrix)[x, y]);
                }
                str.Append("\r\n");
            }
            return str.ToString();
        }

        /// <summary>
        /// Writes the contents of region to String
        /// </summary>
        /// <returns>Contents of ROI as String</returns>
        public override string ToString()
        {
            StringBuilder str = new StringBuilder();
            foreach (ROIPoint rp in this.points)
            {
                str.Append(
                    " " +  rp.x.ToString("f2", new System.Globalization.CultureInfo("en-GB")));
                str.Append(
                    ", " + rp.y.ToString("f2", new System.Globalization.CultureInfo("en-GB")) );
            }

            return str.ToString();
        }

        private void init()
        {
            this.points = new List<ROIPoint>();

            this.Header.ROIName = ".";
            //this.Header.fileName = ".";
            this.Header.Direction = Slice.Direction.Transaxial;
            this.Header.MatrixNumber = new Matrix_Number(0);
            this.Header.RoiType = ROI_Type.Rectangle;

            this.coordinatesystem = Coordinate_System.Pixel;
            this.FillMethod = Fill_Method.Shape;
            this.x = 0.0d;
            this.y = 0.0d;
            this.Header.ROINumber = 0;
            this.Visible = true;
            this.width = 0;
            this.height = 0;
            this.angle = 0.0d;
            this.flipX = false;
            this.flipY = false;
            this.flipZ = false;
        }

    }
    /// <summary>
    /// Type of ROI shape. (rectangle, circle, ellipse, trace)
    /// </summary>
    public enum ROI_Type
    {
        /// <summary>
        /// Rectangle ROI
        /// </summary>
        Rectangle = 0,
        /// <summary>
        /// Circle ROI
        /// </summary>
        Circle = 1,
        /// <summary>
        /// Ellipse ROI
        /// </summary>
        Ellipse = 2,
        /// <summary>
        /// Trace ROI, consisting of series of points
        /// </summary>
        Trace = 3
    }
    /// <summary>
    /// This indicates how coordinates are threated. If percent, all the coordinates must
    /// be given in values between 0-1. Resulting points of fill are always in pixel
    /// coordinates.
    /// </summary>
    public enum Coordinate_System
    {
        /// <summary>
        /// Coordinates of ROIs are given as pixels.
        /// </summary>
        Pixel = 0,
        /// <summary>
        /// Coordinates of ROIs are given as values between 0-1. 0 is the first pixel
        /// in the matrix and 1 is last.
        /// </summary>
        Percent = 1
    }
    /// <summary>
    /// One point of trace ROI.
    /// </summary>
    public struct ROIPoint
    {
        /// <summary>
        /// X coordinate of ROIPoint
        /// </summary>
        public double x;
        /// <summary>
        /// Y coordinate of ROIPoint
        /// </summary>
        public double y;
        /// <summary>
        /// Creates ROIPoint from another ROIPoint
        /// </summary>
        public ROIPoint(ROIPoint rp)
        {
            x = rp.x;
            y = rp.y;
        }
        /// <summary>
        /// Creates ROIPoint from x and y
        /// </summary>
        public ROIPoint(int xx, int yy)
        {
            x = xx;
            y = yy;
        }

        /// <summary>
        /// Creates ROIPoint from x and y
        /// </summary>
        public ROIPoint(Double xx, Double yy)
        {
            x = xx;
            y = yy;
        }
        /// <summary>
        /// Returns string representation of this object
        /// </summary>
        /// <returns>ROI point pixel location as string</returns>
        public override string ToString()
        {
            return "[" + x + "  " + y + "]";
        }
    }
    /// <summary>
    /// ROI information that will not affect to ROI shape.
    /// </summary>
    public struct ROI_Header
    {
        /// <summary>
        /// Matrix number containing information about ROI frame, gate etc.
        /// </summary>
        public Matrix_Number MatrixNumber;
        /// <summary>
        /// ROI number 
        /// </summary>
        public int ROINumber;
        /// <summary>
        /// ROI name
        /// </summary>
        public String ROIName;
        /// <summary>
        /// Type of ROI definition
        /// </summary>
        public ROI_Type RoiType;
        /// <summary>
        /// Orientation of ROI related to image
        /// </summary>
        public Slice.Direction Direction;
        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="from">data for copying</param>
        public ROI_Header(ROI_Header from)
        {
            //this.fileName = from.fileName;
            this.MatrixNumber = new Matrix_Number(from.MatrixNumber.Value);
            this.ROINumber = from.ROINumber;
            this.ROIName = from.ROIName;
            this.RoiType = from.RoiType;
            this.Direction = from.Direction;
        }
    }
}
