﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Calculates means, mins maxs and other results from image and ROI/VOI
    /// </summary>
    public abstract class ROICalculator
    {
        /// <summary>
        /// Number of ROIs to calculate
        /// </summary>
        protected int Length;

        /// <summary>
        /// List of possible functions
        /// </summary>
        public enum func
        {
            /// <summary>
            /// The "arithmetic mean" or "standard average" of pixel values.
            /// </summary>
            Mean = 1,
            /// <summary>
            /// Maximum value among the pixels values
            /// </summary>
            Max,
            /// <summary>
            /// Minimum value among the pixels values
            /// </summary>
            Min,
            /// <summary>
            /// Standard deviation of pixel values
            /// </summary>
            Standard_Deviation,
            /// <summary>
            /// Covariance of pixels values
            /// </summary>
            Covariance
        }

        /// <summary>
        /// Adds new value to function object.
        /// </summary>
        /// <param name="value">New value</param>
        public abstract void AddValue( float[] value );
        
        /// <summary>
        /// Gets the final result from calculator
        /// </summary>
        /// <returns></returns>
        public abstract double[] GetResult();

        /// <summary>
        /// Sets the length of the elements in the calculator.
        /// Usually it is the number of frames of image.
        /// </summary>
        /// <param name="length">Length of elements( frames )</param>
        public abstract void SetLength(int length);

        /// <summary>
        /// Creates calculator object of given function with only one frame. All other
        /// frame values are ignored
        /// </summary>
        /// <param name="function">Function of result object</param>
        /// <returns>Calculator object for given function</returns>
        public static ROICalculator Create( func function )
        {
            return Create(function, 1);
        }

        /// <summary>
        /// Creates calculator object of given function.
        /// </summary>
        /// <param name="function">Function of result object</param>
        /// <param name="length">Length of result array ( one result for one frame )</param>
        /// <returns>alculator object for given function</returns>
        public static ROICalculator Create(func function, int length)
        {
            switch (function)
            {
                case func.Mean:
                    return new ROICalculator_Mean(length);
                case func.Max:
                    return new ROICalculator_Max(length);
                case func.Min:
                    return new ROICalculator_Min(length);
                case func.Covariance:
                    return new ROICalculator_Covariance(length);
            }
            return null;
        }
    }
}
