/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using TPClib.ROI;
using TPClib;
using System.IO;

namespace TPClib.ROI
{
    /// <summary>
    /// Class representing ROI file.
    /// </summary>
    public class ROIFile: TPCFile
    {
        /// <summary>
        /// Supported ROI file types
        /// </summary>
        public enum ROIFormat
        {
            /// <summary>
            /// Vinci software ROI format. Max-Planck-Institut fr neurologische Forschung, Kln (Cologne, Germany).
            /// </summary>
            Vinci,
            /// <summary>
            /// Imadeus software ROI format. Forima.
            /// </summary>
            Imadeus,
            /// <summary>
            /// ImageTool ROI format. ECAT.
            /// </summary>
            ImageTool,
            /// <summary>
            /// Indicates that ROI format is unknown.
            /// </summary>
            Unknown
        }

        /// <summary>
        /// Gets list of supported ROI formats
        /// </summary>
        /// <returns>List of supported ROI formats</returns>
        public static List<ROIFormat> GetSupportedFormats()
        {
            List<ROIFormat> result = new List<ROIFormat>();
            result.Add(ROIFormat.Imadeus);
            result.Add(ROIFormat.ImageTool);
            result.Add(ROIFormat.Vinci);
            return result;
        }

        /// <summary>
        /// Header information for ROIs.
        /// </summary>
        public struct ROIFileHeader
        {
            /// <summary>
            /// Name of image file
            /// </summary>
            public String ImageFile;
            /// <summary>
            /// Raw file that was used when creating ROI
            /// </summary>
            public String RawImage;
            /// <summary>
            /// ROI set name.
            /// </summary>
            public String Name;
            /// <summary>
            /// ROI set orientation.
            /// </summary>
            public String Orientation;
            /// <summary>
            /// Orientation code.
            /// </summary>
            public int Orient;
            /// <summary>
            /// Width of the reference image.
            /// </summary>
            public int Resolution_X;
            /// <summary>
            /// Height of the reference image.
            /// </summary>
            public int Resolution_Y;
            /// <summary>
            /// Number of planes of the reference image.
            /// </summary>
            public int Resolution_Z;
            /// <summary>
            /// Voxel x-size of the reference image.
            /// </summary>
            public Double VoxelDimension_X;
            /// <summary>
            /// Voxel y-size of the reference image.
            /// </summary>
            public Double VoxelDimension_Y;
            /// <summary>
            /// Voxel z-size of the reference image.
            /// </summary>
            public Double VoxelDimension_Z;
            /// <summary>
            /// Flag for flipping around x-axis while drawing the ROI
            /// </summary>
            public bool Flip_X;
            /// <summary>
            /// Flag for flipping around y-axis while drawing the ROI
            /// </summary>
            public bool Flip_Y;
            /// <summary>
            /// Flag for flipping around z-axis while drawing the ROI
            /// </summary>
            public bool Flip_Z;
            /// <summary>
            /// ROI set x-origin.
            /// </summary>
            public int Ori_X;
            /// <summary>
            /// ROI set y-origin.
            /// </summary>
            public int Ori_Y;
            /// <summary>
            /// ROI set z-origin.
            /// </summary>
            public int Ori_Z;
            /// <summary>
            /// Free comment field.
            /// </summary>
            public String Comment;
        }

        /// <summary>
        /// Common information for all ROI's
        /// </summary>
        public ROIFileHeader Header;

        /// <summary>
        /// Format of the ROI file.
        /// </summary>
        public ROIFormat ROIFileFormat;

        /// <summary>
        /// VOI and ROI data of file.
        /// </summary>
        public List<VOI> VOIs;

        /// <summary>
        /// Constructs ROI file.
        /// </summary>
        public ROIFile() 
        {
            this.filename = null;
            Init();
        }

        /// <summary>
        /// Constructs ROI file.
        /// </summary>
        /// <param name="filename">full path to file.</param>
        public ROIFile(string filename)
        {
            Init();
            this.filename = filename;
        }
        /// <summary>
        /// Initializes ROI file data. 
        /// </summary>
        protected void Init()
        {
            ROIFileFormat = ROIFormat.Unknown;
            VOIs = new List<VOI>();

            this.Header.Resolution_X = 0;
            this.Header.Resolution_Y = 0;
            this.Header.Resolution_Z = 0;
            //this.Header.Regions = 0;
            this.Header.RawImage = ".";
            this.Header.Orientation = ".";
            this.Header.Orient = 0;
            this.Header.Ori_X = 0;
            this.Header.Ori_Y = 0;
            this.Header.Ori_Z = 0;
            this.Header.VoxelDimension_X = 0.0d;
            this.Header.VoxelDimension_Y = 0.0d;
            this.Header.VoxelDimension_Z = 0.0d;
            this.Header.Name = ".";
            this.Header.ImageFile = ".";
            this.Header.Flip_X = true;
            this.Header.Flip_Y = false;
            this.Header.Flip_Z = false;
            this.Header.Comment = ".";
        }

        /// <summary>
        /// Writes the ROI information to ImageToolROIfile
        /// </summary>
        /// <param name="fileName">Name of file</param>
        public void ReadFile(String fileName)
        {
            this.filename = fileName;
            ReadFile();
        }

        /// <summary>
        /// Writes the ROI information to ImageToolROIfile
        /// </summary>
        /// <param name="fileName">Name of file</param>
        public void WriteFile(String fileName)
        {
            this.filename = fileName;
            WriteFile();
        }
        /// <summary>
        /// Writes ROI file into defined file
        /// <seealso cref="TPCFile.filename"/>
        /// </summary>
        public override void WriteFile()
        {
            if (ROIFileFormat == ROIFormat.Unknown) throw new TPCROIFileException("Cannot write file. File type is unknown.");

            ROIFile file;
            if      (ROIFileFormat == ROIFormat.ImageTool) { file = new ImageToolROIFile(filename); }
            else if (ROIFileFormat == ROIFormat.Imadeus) { file = new ImadeusROIFile(filename); }
            else if (ROIFileFormat == ROIFormat.Vinci) { file = new VinciROIFile(filename); }
            else return;
            
            file.VOIs = this.VOIs;
            file.WriteFile();
        }

        /// <summary>
        /// Reads ROI file from defined file
        /// <seealso cref="TPCFile.filename"/>
        /// </summary>
        public override void ReadFile()
        {
            ROIFile file = ROIFile.CreateROIFileObject( filename );
            if (file == null) throw new TPCROIFileException("Cannot recognize file type: " + filename );            
            file.ReadFile();
            this.VOIs = file.VOIs;
            this.ROIFileFormat = file.ROIFileFormat;
        }

        /// <summary>
        /// Checks the given file format. If the file is in some supported format, true will
        /// be returned
        /// </summary>
        /// <returns>True, if the given file is in some supported format.</returns>
        public virtual bool CheckFormat()
        {
            if (ROIFile.ResolveFormat( filename ) == ROIFormat.Unknown) return false;
            else return true;
        }

        /// <summary>
        /// Checks the given file format and returns the format if it is supported.
        /// </summary>
        /// <returns>Format of the given file</returns>
        public static ROIFormat ResolveFormat( String filename )
        {
            Console.WriteLine("Detecting ROI file...");

            if (filename == null) throw new TPCROIFileException("You have not given filename");
            if (!File.Exists(filename)) return ROIFormat.Unknown;

            ROIFormat result = ROIFormat.Unknown;

            // Next we have to find out the file type 
            ROIFile file;
            
            if (result == ROIFormat.Unknown)
            {
                file = new ImageToolROIFile(filename);
                if (file.CheckFormat()) { result = ROIFormat.ImageTool; }
            }

            if (result == ROIFormat.Unknown)
            {
                file = new ImadeusROIFile(filename);
                if (file.CheckFormat()) { result = ROIFormat.Imadeus; }
            }

            if (result == ROIFormat.Unknown)
            {
                file = new VinciROIFile(filename);
                if (file.CheckFormat()) { result = ROIFormat.Vinci; }
            }

            if (result != ROIFormat.Unknown)
            {
                // If we found correct format:
                Console.WriteLine("ROI file is recognized to " + result);
                return result;
            }

            return ROIFormat.Unknown;
        }

        /// <summary>
        /// Creates ROI file object which format is detected from given file
        /// </summary>
        /// <param name="filename">Name of file</param>
        /// <returns>ROIFile object to the file</returns>
        public static ROIFile CreateROIFileObject(String filename)
        {
            ROIFormat format = ROIFile.ResolveFormat(filename);
            if (format == ROIFormat.ImageTool) return new ImageToolROIFile(filename);
            if (format == ROIFormat.Imadeus) return new ImadeusROIFile(filename);
            if (format == ROIFormat.Vinci) return new VinciROIFile(filename);

            return null;
        }
    }
}
