﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// ROIObject is base class for all classes which have regions of interests. All ROI objects have
    /// own matrix object, which can be retrieved with GetMatrix() method. Every child class must
    /// implement its own CreateMatrix and FillMatrix methods which are automaticly called from
    /// ROIObject when needed.
    /// </summary>
    public abstract class ROIObject
    {
        /// <summary>
        /// Protected FillMatrix method implemented in child ROI classes
        /// </summary>
        protected abstract void FillMatrix();

        /// <summary>
        /// Protected CreateMatrix method implemented in child ROI classes
        /// </summary>
        protected abstract void CreateMatrix(params int[] dimensions);

        /// <summary>
        /// This defines how the zoom operation is handled. If zoom method is ImageEnlarge, the size of matrix
        /// will be enlarged. For example if the ROI:s zoom factor is 3 the result Matrix, where rois are drawn
        /// will be three times larger than image itself. If the method is fixed, the matrix will always be as
        /// big as image itself. The cordinates of ROIs are then divided by zoomfactor. 
        /// </summary>
        public static ZoomMethod Zoom_Method = ZoomMethod.Enlarge;

        /// <summary>
        /// Fills the matrix inside of ROI/VOI object with given dimensions
        /// </summary>
        /// <param name="dimensions">Dimensions of the result matrix.</param>
        public void Fill(params int[] dimensions)
        {
            UpdateMatrix(dimensions);
        }

        /// <summary>
        /// Fills the matrix inside of ROI/VOI object with given images dimensions
        /// </summary>
        /// <param name="image">Image containing the dimensions of the result matrix.</param>
        public void Fill(TPClib.Image.Image image)
        {
            UpdateMatrix(image.width, image.height, image.planes);
        }

        /// <summary>
        /// Updates the matrix object. Matrix will be created and filled if necessary. Unchanged
        /// matrix will be kept unchanged.
        /// </summary>
        /// <param name="dimensions"></param>
        protected virtual void UpdateMatrix(params int[] dimensions)
        {
            if (Zoom_Method == ZoomMethod.Fixed)
            {
                CreateMatrix(dimensions);
                width = dimensions[0];
                height = dimensions[1];
            }
            else
            {
                width = dimensions[0];
                height = dimensions[1];

                // Zoom must be Enlarge
                dimensions[0] *= (int)zoomFactor; // width
                dimensions[1] *= (int)zoomFactor; // height
                CreateMatrix(dimensions);
            }

            if (!this.visible) return;

            FillMatrix();

            notReady = false;
        }

        /// <summary>
        /// The matrix will be generated again when next statement(fill,mean,getpixels) is called.
        /// This cannot be set to false. Only true
        /// </summary>
        public virtual bool NeedsUpDate
        {
            get { return notReady;  }
            set {
                if( value ) this.notReady = true;
            }
        }

        /// <summary>
        /// The width of the ROI. (matrix)
        /// </summary>
        public virtual int Width
        {
            get { return width; }
            set { if (width != value) { width = value; notReady = true; } }
        }

        /// <summary>
        /// The height of the ROI. (matrix)
        /// </summary>
        public virtual int Height
        {
            get { return height; }
            set { if (height != value) { height = value; notReady = true; } }
        }

        /// <summary>
        /// Gets the matrix of the ROI/VOI that has been generated in Fill(). If the
        /// geometry of ROI has been changed from last Fill() call, null will be returned
        /// </summary>
        /// <returns>Matrix generated in fill.</returns>
        public virtual MatrixObject GetMatrix()
        {
            if (notReady) return null;
            else return obj_matrix;
        }

        /// <summary>
        /// True if ROI is visible
        /// </summary>
        protected bool visible;
        /// <summary>
        /// Filling method of ROI.
        /// </summary>
        protected Fill_Method fill_method;
        
        /// <summary>
        /// Width and height of ROI (matrix)
        /// </summary>
        protected int width, height;
        /// <summary>
        /// Internal ROI zooming factor value
        /// </summary>
        protected Double zoomFactor;
        /// <summary>
        /// ROI zooming factor
        /// </summary>
        public virtual Double ZoomFactor
        {
            get { return zoomFactor; }
            set { zoomFactor = value; }
        }

        /// <summary>
        /// Objects own matrix. If the geometry ob object is not changed from last
        /// fill call, this matrix will be straightly returned and masking calculations.
        /// are not performed in vain.
        /// </summary>
        protected MatrixObject obj_matrix;
        /// <summary>
        /// If this is true, the obj_matrix need to be recalculated
        /// </summary>
        protected bool notReady;

        /// <summary>
        /// If True, the ROI will be drawn when the fill is called. If False it will not.
        /// </summary>
        public virtual bool Visible
        {
            get { return visible; }
            set { visible = value; }
        }

        /// <summary>
        /// Default constructor
        /// </summary>
        public ROIObject()
        {
            notReady = true;
            this.zoomFactor = 1;
            this.visible = true;            
        }

        /// <summary>
        /// The method which is used when the mask is calculated. If the method is Imadeus or ImageTool,
        /// points are used as in the trace roi. If Shape, the mask area is calculated straight from
        /// shape of current class, for example ellipses or circles arc or rectangles dimensions.
        /// In that case points are not used.
        /// </summary>
        public virtual Fill_Method FillMethod
        {
            get { return fill_method; }
            set { fill_method = value; }
        }
    }
    /// <summary>
    /// The method which is used when the mask is calculated. If the method is Imadeus or ImageTool,
    /// points are used as in the trace roi. If Shape, the mask area is calculated straight from
    /// shape of current class, for example ellipses or circles arc or rectangles dimensions.
    /// In that case points are not used.
    /// </summary>
    public enum Fill_Method
    {
        /// <summary>
        /// Most ROI classes have possibility to calculate the ROI area straight from shape. For example
        /// CircleROI can calculate the ROI matemathically from x,y and radius. If method is Shape, the
        /// ROI class will use its own method to calculate ROI. This usually ignores the
        /// transforming ( flipping, zooming, rotating )
        /// </summary>
        Shape,
        /// <summary>
        /// The ROI will be calculated with Imadeus mask algorithm. Every shape will be converted to points
        /// when this method is used.
        /// </summary>
        Imadeus,
        /// <summary>
        /// The ROI will be calculated with ImageTool mask algorithm. Every shape will be converted to points
        /// when this method is used.
        /// </summary>
        ImageTool
    }

    /// <summary>
    /// This defines how the zoom operation is handled. If zoom method is ImageEnlarge, the size of matrix
    /// will be enlarged. For example if the ROI:s zoom factor is 3 the result Matrix, where rois are drawn
    /// will be three times larger than image itself. If the method is fixed, the matrix will always be as
    /// big as image itself. The cordinates of ROIs are then divided by zoomfactor. 
    /// </summary>
    public enum ZoomMethod
    {
        /// <summary>
        /// Matrix is enlargened
        /// </summary>
        Enlarge,
        /// <summary>
        /// Matrix size is fixed to image size
        /// </summary>
        Fixed
    }

}
