﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// 2D Matrix object which is used in all 2D ROI shapes.
    /// </summary>
    public class ROI_Matrix : MatrixObject
    {
        /// <summary>
        /// Two dimensional array containing the ROI information. If the cell is
        /// inside the ROI, it's value is 1. If not it is 0.
        /// </summary>
        public byte[][] cells;

        /// <summary>
        /// Constructs ROI matrix
        /// </summary>
        /// <param name="width">Width of matrix</param>
        /// <param name="height">Height of matrix</param>
        public ROI_Matrix(int width, int height)
        {
            this.width = width;
            this.height = height;

            cells = new byte[width][];
            for( int i=0; i< width; i++ ) cells[i] = new byte[height];

            Clear();
        }

        /// <summary>
        /// Clears all cells to zero
        /// </summary>
        public override void Clear()
        {
            for (int y = 0; y < height; y++)
            {
                for (int x = 0; x < width; x++)
                {
                    cells[x][y] = 0;
                }
            }
        }
        /// <summary>
        /// Value at location (x,y)
        /// </summary>
        /// <param name="x">x-index [0..width-1]</param>
        /// <param name="y">y-index [0..height-1]</param>
        /// <returns></returns>
        public byte this[ int x, int y ]
        {
            get 
            {
                if( x < 0 || x >= width ) throw new TPCROIException("ROIMatrix: Index out of bounds.");
                if( y < 0 || y >= height ) throw new TPCROIException("ROIMatrix: Index out of bounds.");
                return cells[x][y];
            }
            set
            {
                if (x < 0 || x >= width) throw new TPCROIException("ROIMatrix: Index out of bounds.");
                if (y < 0 || y >= height) throw new TPCROIException("ROIMatrix: Index out of bounds.");
                cells[x][y] = value;
            }
        }

        /// <summary>
        /// Divides the matrix elements with (mod 2)
        /// </summary>
        public void Mod()
        {
            for (int y = 0; y < height; y++)
            {
                for (int x = 0; x < width; x++)
                {
                    cells[x][y] %= 2;
                }
            }
        }

        /// <summary>
        /// Claculates some function inside the ROI and returns the result
        /// </summary>
        /// <param name="img">Image reference</param>
        /// <param name="slice">Slice in 3D image where this 2D matrix is getting pixel values from</param>
        /// <param name="function">Function which is used for calculation. Can be for example Mean,Min,Max...</param>
        /// <returns>Result of function inside the ROI. Array of resuts for every frame</returns>
        public Double[] Calculate(ref Image.Image img, Slice slice, ROICalculator.func function )
        {
            // Frame length must be given
            int frames;
            if (img.dim.Length > 3) frames = img.dim.GetDimension(3);
            else frames = 1;

            ROICalculator rc = ROICalculator.Create(function, frames);            

            Calculate(ref img, cells, slice, rc );
            return rc.GetResult();
        }

        /// <summary>
        /// Claculates some function inside the ROI, but doesn't return the result. The
        /// Values are only added to the calculator and result must be got by caller. This
        /// is good when function result of many different objects is needed, for example
        /// (mean of union of many VOIs/ROIs inside the image).
        /// </summary>
        /// <param name="img">Image reference</param>
        /// <param name="slice">Slice in 3D image where this 2D matrix is getting pixel values</param>
        /// <param name="calculator">Calculator reference where the result is added to</param>
        public void Calculate(ref Image.Image img, Slice slice, ROICalculator calculator)
        {
            Calculate(ref img, cells, slice, calculator);
        }

        
        /// <summary>
        /// Gets list of pixels(ROIValuePoints) from image. All pixels that have selected in matrix
        /// are returned
        /// </summary>
        /// <param name="img">Image reference</param>
        /// <param name="slice">Slice in 3D image where this 2D matrix is getting pixel values from</param>
        /// <returns>List of points(pixels) which are inside the region of interest.</returns>
        public List<ROIValuePoint> GetPixels(ref Image.Image img, Slice slice)
        {
            return GetPixels(ref img, cells, slice);
        }

        /// <summary>
        /// Combines two ROI_Matrix together with binary OR.
        /// </summary>
        /// <param name="source">Another ROI matrix to combine</param>
        public void Union( ROI_Matrix source )
        {
            if (this.height != source.height || this.width != source.width)
                throw new TPCROIException("Union: Dimensions differ.");

            for (int y = 0; y < this.height; y++)
            {
                for (int x = 0; x < this.width; x++)
                {
                    this.cells[x][y] |= source.cells[x][y]; 
                }
            }
        }
        

        /// <summary>
        /// Writes contents of ROIMatrix to string.
        /// </summary>
        /// <returns>String containing the ROI information</returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.Append("Matrix:\r\n");

            for (int y = 0; y < height; y++)
            {
                for (int x = 0; x < width; x++)
                {
                    if (
                        x == 45 && y == 80                 
                        /*x == 42 && y == 82 ||
                        x == 41 && y == 83 ||
                        x == 36 && y == 87 ||
                        x == 36 && y == 88 ||
                        x == 35 && y == 91 ||
                        x == 35 && y == 92 ||
                        x == 34 && y == 94 ||
                        x == 36 && y == 95 ||
                        x == 39 && y == 97*/
                        ) 
                            sb.Append("X");
                    else if (
                        x == 44 && y == 80 ||
                        x == 43 && y == 81
                        /*y == 81 && x == 49 ||
                        y == 91 && x == 53 ||
                        y == 92 && x == 53 ||
                        y == 93 && x == 53 ||
                        y == 94 && x == 53 ||
                        y == 95 && x == 53 ||
                        y == 96 && x == 53 ||
                        y == 97 && x == 47*/
                        )
                        sb.Append("Å");

                    else sb.Append(cells[x][y].ToString());
                }
                sb.Append("\r\n");
            }
            return sb.ToString();
        }

        /// <summary>
        /// Iterate through matrix elements
        /// </summary>
        /// <returns></returns>
        public System.Collections.Generic.IEnumerator<Byte> GetEnumerator()
        {
            for (int y = 0; y < height; y++)
            {
                for (int x = 0; x < width; x++)
                {
                    yield return cells[x][y];
                }
            }
        }

        /// <summary>
        /// Checks if this matrix is identical with obj matrix. Object can be byte[][] or another
        /// ROI_Matrix
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(Object obj)
        {
            byte[][] bytes;
            if (obj is ROI_Matrix) bytes = (obj as ROI_Matrix).cells;
            else if (obj is byte[][]) bytes = (obj as byte[][]);
            else return false;

            if (bytes.Length != width) return false;
            if (bytes.Length == 0) return true;
            if (bytes[0].Length != height) return false;
            for (int y = 0; y < height; y++)
            {
                for (int x = 0; x < width; x++)
                {
                    if (bytes[x][y] != cells[x][y]) return false;
                }
            }
            return true;
        }
        /// <summary>
        /// Gets hash code of this object. Default bahaviour.
        /// </summary>
        /// <returns>returns hash code with default implementation</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
