﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Rectangle-shaped Region Of Interest
    /// </summary>
    public class RectangleROI : ROI
    {
        /// <summary>
        /// Fills the matrix with ones inside the ROI.
        /// </summary>
        protected override void FillMatrix()
        {
            base.FillMatrix();

            if (FillMethod == Fill_Method.Shape)
            {
                RectangleFill();
            }
        }

        /// <summary>
        /// Calculates the transformed(zoomed,rotated) points. This parent function calculates
        /// only the y and x component vectors.
        /// </summary>
        protected override void ReCalculate()
        {
            base.ReCalculate();

            double halfw = shapewidth / 2.0d;
            double halfh = shapeheight / 2.0d;

            points = new List<ROIPoint>();

            points.Add(new ROIPoint( calcX(-halfw, -halfh), calcY(-halfw, -halfh) ) );
            points.Add(new ROIPoint(calcX(halfw, -halfh), calcY(halfw, -halfh) ));
            points.Add(new ROIPoint(calcX(halfw, halfh), calcY(halfw, halfh) ));
            points.Add(new ROIPoint(calcX(-halfw, halfh), calcY(-halfw, halfh) ));
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI(Double x, Double y, Double width, Double height)
            : base(".", 1.0d, new Matrix_Number(0), ROI_Type.Rectangle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Rectangle;
            this.shapewidth = width;
            this.shapeheight = height;
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI(Double x, Double y, Double width, Double height, Double zoomFactor)
            : base(".", zoomFactor, new Matrix_Number(0), ROI_Type.Rectangle, x, y, 0)
        {
            this.Header.RoiType = ROI_Type.Rectangle;
            this.shapewidth = width;
            this.shapeheight = height;
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI(String name, Double zoomFactor, Matrix_Number MatNum,
                                Double x, Double y, int roiNum, Double width, Double height)
            : base(name, zoomFactor, MatNum, ROI_Type.Rectangle, x, y, roiNum)
        {
            this.Header.RoiType = ROI_Type.Rectangle;
            this.shapewidth = width;
            this.shapeheight = height;
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI(ROI roi)
            : base(roi)
        {
            this.Header.RoiType = ROI_Type.Rectangle;
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI(RectangleROI roi)
            : base(roi)
        {
            this.Header.RoiType = roi.Header.RoiType;
            this.shapewidth = roi.width;
            this.shapeheight = roi.height;
        }

        /// <summary>
        /// Creates Rectangnle ROI
        /// </summary>
        public RectangleROI()
            : base()
        {
            this.Header.RoiType = ROI_Type.Rectangle;
            this.width = 0;
            this.height = 0;
        }

        /// <summary>
        /// Creates replica of RectangleROI
        /// </summary>
        public override ROI Clone()
        {
            RectangleROI r = new RectangleROI(this);

            r.points = new List<ROIPoint>();
            foreach (ROIPoint ro in this.points)
            {
                r.points.Add(new ROIPoint(ro.x, ro.y));
            }

            return r;
        }

        /// <summary>
        /// This method fills the matrix without points when FillMethod is Shape. This ignores the
        /// transforming ( flipping, zooming, rotating )
        /// </summary>
        private void RectangleFill()
        {
            int x1 = (int)points[0].x;
            int x2 = (int)points[2].x;
            int y1 = (int)points[0].y;
            int y2 = (int)points[2].y;

            if (x1 < 0) x1 = 0;
            if (x2 >= obj_matrix.Width) x2 = obj_matrix.Width;
            if (y1 < 0) y1 = 0;
            if (y1 >= obj_matrix.Height) y2 = obj_matrix.Height;

            for (int y = y1; y < y2; y++) for (int x = x1; x < x2; x++)
                {
                    if (0 <= y && y<obj_matrix.Width && 0 <= x && x < obj_matrix.Height) (obj_matrix as ROI_Matrix)[x,y] = 1;
                }
        }
    }
}
