﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Slice class takes one 2D slice from 3d volume and returns always the 3D space
    /// coordinates of the 2D point on the slice.
    /// </summary>
    public abstract class Slice
    {
        /// <summary>
        /// Direction of the 2D slice
        /// </summary>
        public enum Direction
        {
            /// <summary>
            /// The direction is transaxial (x,y) depeth=z
            /// </summary>
            Transaxial,
            /// <summary>
            /// The direction is coronal (x,z) depeth=y
            /// </summary>
            Coronal,
            /// <summary>
            /// The direction is sagittal (y,z) depeth=x
            /// </summary>
            Sagittal,
            /// <summary>
            /// The direction is free. Depeth starts from center
            /// </summary>
            Free
        }
        /// <summary>
        /// z position in volume
        /// </summary>
        protected int depeth;
        /// <summary>
        /// Slice upper left low x-coordinate
        /// </summary>
        protected int x;
        /// <summary>
        /// Slice upper left low y-coordinate
        /// </summary>
        protected int y;
        /// <summary>
        /// Slice upper left low z-coordinate
        /// </summary>
        protected int z;
        /// <summary>
        /// Slice width in voxels.
        /// </summary>
        protected int width;
        /// <summary>
        /// Slice height in voxels.
        /// </summary>
        protected int height;
        /// <summary>
        /// Thickness of slice.
        /// </summary>
        protected int maxDepeth;
        /// <summary>
        /// Slice normal vector's x-component.
        /// </summary>
        protected double normalx;
        /// <summary>
        /// Slice normal vector's y-component.
        /// </summary>
        protected double normaly;
        /// <summary>
        /// Slice normal vector's z-component.
        /// </summary>
        protected double normalz;
        /// <summary>
        /// Center of slice x-coordinate
        /// </summary>
        protected int origox;
        /// <summary>
        /// Center of slice x-coordinate
        /// </summary>
        protected int origoy;
        /// <summary>
        /// Number of voxels in slice
        /// </summary>
        private int FrameSize;
        /// <summary>
        /// Number of pixel locations in slice. Here a pixel location may contain number of voxels.
        /// </summary>
        private int widthheight;
        /// <summary>
        /// Gets the x coordinate of slice in the 3d space 
        /// </summary>
        public int X { get { return x; } }
        /// <summary>
        /// Gets the y coordinate of slice in the 3d space 
        /// </summary>
        public int Y { get { return y; } }
        /// <summary>
        /// Gets the z coordinate of slice in the 3d space 
        /// </summary>
        public int Z { get { return z; } }
        /// <summary>
        /// Gets the coordinates of slice in the 3d space 
        /// </summary>
        public int[] Pos {
            get { return new int[] { x, y, z }; }
            set {
                if (value == null || value.Length < 3) throw new TPCROIException("Slice Pos: Position must have three dimensions.");
                x = value[0]; y = value[1]; z = value[2];
            }
        }
        /// <summary>
        /// Gets the location of point in the 3d space 
        /// </summary>
        public int Location { get { return z*(widthheight) + y*width + x; } }
        /// <summary>
        /// Returns the 3D normal vector of slice
        /// </summary>
        public double[] Normal
        {
            get
            {
                return new double[] {normalx,normaly,normalz};
            }
        }

        /// <summary>
        /// Gets the 2D width of slice 
        /// </summary>
        public int Width { get { return width; } }
        /// <summary>
        /// Gets the 2D height of slice 
        /// </summary>
        public int Height { get { return height; } }
        /// <summary>
        /// Gets the 2D height of slice 
        /// </summary>
        public int Depeth { get { return maxDepeth; } }

        /// <summary>
        /// Gets/Sets the x-coordinate in slices 2D space
        /// </summary>
        public virtual int X2D { get { return 0; } set { } }
        /// <summary>
        /// Gets/Sets the y-coordinate in slices 2D space
        /// </summary>
        public virtual int Y2D { get { return 0; } set { } }
        /// <summary>
        /// Gets/Sets the depeth in slices 2D space
        /// </summary>
        public virtual int Depeth2D { get { return 0; } set { } }
        /// <summary>
        /// Checks if the 3D dimensions match the dimensions of the slice
        /// </summary>
        public abstract bool CheckDimensions(int mat_width, int mat_height, int mat_depeth);

        /// <summary>
        /// Creates slice object by given information
        /// </summary>
        /// <param name="direction">Direction of slice</param>
        /// <param name="dep">Depeth in 3D matrix where the 2D slice resides</param>
        /// <param name="mat_width">width of 3D matrix</param>
        /// <param name="mat_height">height of 3D matrix</param>
        /// <param name="mat_depeth">depeth of 3d matrix</param>
        /// <returns>Slice object created by given information</returns>
        public static Slice Create(Direction direction, int dep, int mat_width, int mat_height, int mat_depeth)
        {
            switch (direction)
            {
                case Direction.Transaxial:
                    return new Slice_Transaxial(dep, mat_width, mat_height, mat_depeth);
                case Direction.Sagittal:
                    return new Slice_Sagittal(dep, mat_width, mat_height, mat_depeth);
                case Direction.Coronal:
                    return new Slice_Coronal(dep, mat_width, mat_height, mat_depeth);
            }
            
            return null;
        }

        /// <summary>
        /// Gets curve data from Image.
        /// </summary>
        /// <param name="img">Reference to image</param>
        /// <returns>Array of floats containing the curve data</returns>
        public float[] GetCurve(ref Image.Image img)
        {
            // Frame length must be given
            int lengt;
            if (img.dim.Length > 3) lengt = img.dim.GetDimension(3);
            else lengt = 1;

            float[] result = new float[lengt];

            int posit = Location;

            for( int i=0; i<lengt; i++ )
            {
                result[i] = img[posit];
                posit += FrameSize;
            }

            return result;
        }

        /// <summary>
        /// Gets all the curve pixels at the current location of the slice
        /// </summary>
        /// <param name="img">reference to the image</param>
        /// <returns>List of ROIValuePoints containing th pixel values</returns>
        public List<ROIValuePoint> GetCurvePixels(ref Image.Image img)
        {
            // Frame length must be given
            int lengt;
            if (img.dim.Length > 3) lengt = img.dim.GetDimension(3);
            else lengt = 1;
            List<ROIValuePoint> result = new List<ROIValuePoint>();

            int posit = Location;

            for (int i = 0; i < lengt; i++)
            {
                result.Add( new ROIValuePoint( x, y, z, i, img[posit] ) );
                posit += FrameSize;
            }
            return result;
        }
        /// <summary>
        /// Initializes number of values in slice
        /// </summary>
        /// <param name="mat_width">matrix width</param>
        /// <param name="mat_height">matrix height</param>
        /// <param name="mat_depeth">matrix thickness</param>
        protected void init( int mat_width, int mat_height, int mat_depeth )
        {
            FrameSize = mat_width * mat_height * mat_depeth;
            widthheight = mat_width * mat_height;
        }
}
}
