﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Region Of Interest consisting of series of points that are connected in order.
    /// </summary>
    public class TraceROI : ROI
    {
        private List<ROIPoint> pointList;

        /// <summary>
        /// Fills the matrix's area of traceroi with ones.
        /// </summary>
        protected override void FillMatrix()
        {
            // Too few points to draw anything
            if (pointList.Count < 2) return;

            base.FillMatrix();

            if (FillMethod == Fill_Method.Shape)
            {
                ROI_Matrix rm = (obj_matrix as ROI_Matrix);
                ImadeusMask.Fill(rm, points);
            }            
        }

        /// <summary>
        /// Calculates the transformed(zoomed,rotated) points. This parent function calculates
        /// only the y and x component vectors.
        /// </summary>
        protected override void ReCalculate()
        {
            base.ReCalculate();

            points = new List<ROIPoint>();

            for (int i = 0; i < pointList.Count; i++)
            {
                double newX = pointList[i].x;
                double newY = pointList[i].y;

                points.Add(new ROIPoint(    calcX( newX, newY ),
                                            calcY( newX, newY )));
            }
        }

        /// <summary>
        /// Creates Trace ROI
        /// </summary>
        public TraceROI()
            : base()
        {
            this.Header.RoiType = ROI_Type.Trace;
            pointList = new List<ROIPoint>();
        }
        /// <summary>
        /// Creates Trace ROI
        /// </summary>
        public TraceROI(ROI roi)
            : base(roi)
        {
            this.Header.RoiType = ROI_Type.Trace;
            pointList = new List<ROIPoint>();
        }

        /// <summary>
        /// Creates Trace ROI
        /// </summary>
        public TraceROI(int x, int y, int z)
            : base()
        {
            this.Header.RoiType = ROI_Type.Trace;
            pointList = new List<ROIPoint>();
            this.x = x;
            this.y = y;
            this.z = z;
        }

        /// <summary>
        /// Adds one point to TraceROI
        /// </summary>
        /// <param name="x">x cordinate of new point.</param>
        /// <param name="y">y cordinate of new point.</param>
        public void AddPoint(Double x, Double y)
        {
            pointList.Add( new ROIPoint(x,y) );
            notReady = true;
        }

        /// <summary>
        /// Gets the original points that are not transformed( zoomed, flipped, rotated )
        /// </summary>
        public List<ROIPoint> OriginalPoints
        {
            get
            {
                return this.pointList;
            }
        }

        /// <summary>
        /// Adds points to TraceROI
        /// </summary>
        /// <param name="ROIpoints">ROIPoints to add.</param>
        public void AddPoint(params ROIPoint[] ROIpoints )
        {
            foreach (ROIPoint rp in ROIpoints)
            {
                pointList.Add(new ROIPoint(rp.x, rp.y));
            }
            notReady = true;
        }

        /// <summary>
        /// Gives new points to TraceROI object.
        /// </summary>
        /// <param name="point_list">List of new points</param>
        public void SetPoints(List<ROIPoint> point_list )
        {
            pointList = new List<ROIPoint>();
            foreach (ROIPoint rp in point_list)
            {
                pointList.Add( new ROIPoint(rp.x, rp.y) );
            }
            notReady = true;
        }

        /// <summary>
        /// Removes one point from TraceROI
        /// </summary>
        /// <param name="index">Index of point to remove.</param>
        public void RemovePoint(int index)
        {
            if (index < 0 || index >= points.Count) throw new TPCROIException("RemovePoint: Index out of bounds.");
            pointList.RemoveAt(index);

            notReady = true;
        }

        /// <summary>
        /// Removes all the points from list.
        /// </summary>
        public void ClearPoints()
        {
            pointList.Clear();
            notReady = true;
        }

        /// <summary>
        /// Creates replica of TraceROI
        /// </summary>
        public override ROI Clone()
        {
            TraceROI r = new TraceROI(this);

            r.points = new List<ROIPoint>();
            foreach (ROIPoint ro in this.points)
            {
                r.points.Add(new ROIPoint(ro.x, ro.y));
            }

            return r;
        }
    }
}
