﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Triangle face in 3D mesh structure.
    /// </summary>
    public class TriFace
    {
        /// <summary>
        /// Normal (arbitrary direction, should be updated before use)
        /// </summary>
        public Point n;
        /// <summary>
        /// Corners (ordered, face outside)
        /// </summary>
        public int[] mexel = new int[3];
        /// <summary>
        /// Neighbouring face indexes
        /// </summary>
        protected int[] Ng_internal = new int[3];
        /// <summary>
        /// Neighbouring face indexes
        /// </summary>
        public int[] Ng
        {
            get
            {
                return Ng_internal;
            }
        }
        /// <summary>
        /// Flag for deleted mexel.
        /// </summary>
        public bool MESH_DELETED_ITEM = false;
        /// <summary>
        /// Designated value for feature mesh item.
        /// So-called feature item in mesh should not be deformed with mesh.
        /// </summary>
        public bool MESH_FEATURE_ITEM = false;
        /// <summary>
        /// Designated value for feature mesh item.
        /// Static item was not changed in the previous round.
        /// </summary>
        public bool MESH_STATIC_ITEM = false;
        /// <summary>
        /// Temporary marker for item, should be cleared after use
        /// </summary>
        public bool MESH_TEMP_ITEM = false;
        /// <summary>
        /// Marker for item that intersects with other item
        /// </summary>
        public bool MESH_INTERSECTING_ITEM = false;
        /// <summary>
        /// Marker for item that is partially valid (intersecting with the mesh)
        /// </summary>
        public bool MESH_PARTIALLY_VALID_ITEM = false;
        /// <summary>
        /// Marker for item that is valid (does not intersect with the mesh)
        /// </summary>
        public bool MESH_VALID_ITEM = false;
        /// <summary>
        /// Marker for item that is unvisited
        /// </summary>
        public bool MESH_UNVISITED_ITEM = false;
        /// <summary>
        /// Default constructor. All indexes are set to -1.
        /// </summary>
        public TriFace()
        {
            mexel[0] = -1;
            mexel[1] = -1;
            mexel[2] = -1;
            Ng_internal[0] = -1;
            Ng_internal[1] = -1;
            Ng_internal[2] = -1;
        }
        /// <summary>
        /// Initializes face with three index numbers in mesh.
        /// </summary>
        /// <param name="p1">corner1</param>
        /// <param name="p2">corner2</param>
        /// <param name="p3">corner3</param>
        public TriFace(int p1, int p2, int p3)
        {
            mexel[0] = p1;
            mexel[1] = p2;
            mexel[2] = p3;
            Ng_internal[0] = -1;
            Ng_internal[1] = -1;
            Ng_internal[2] = -1;
        }
        /// <summary>
        /// Adds neighbour face index for this triface.
        /// </summary>
        /// <param name="i">0-based index in mesh</param>
        /// <returns>true if add was success</returns>
        public bool AddNg(int i)
        {
            if (Ng_internal[0] == i || Ng_internal[1] == i || Ng_internal[0] == i)
                throw new TriMeshException("Face neighbour index is already included.");
            if (Ng_internal[0] == -1) Ng_internal[0] = i;
            else if (Ng_internal[1] == -1) Ng_internal[1] = i;
            else if (Ng_internal[2] == -1) Ng_internal[2] = i;
            else return false;
            return true;
        }
        /// <summary>
        /// Resets all neighbour face information.
        /// </summary>
        public void ClearNg()
        {
            Ng_internal[0] = -1;
            Ng_internal[1] = -1;
            Ng_internal[2] = -1;
        }
        /// <summary>
        /// Removes neighbour face index from this triface.
        /// </summary>
        /// <param name="i">0-based index in mesh</param>
        /// <returns>true if removal was success</returns>
        public bool RemoveNg(int i)
        {
            if (Ng_internal[0] == i) Ng_internal[0] = -1;
            else if (Ng_internal[1] == i) Ng_internal[1] = -1;
            else if (Ng_internal[2] == i) Ng_internal[2] = -1;
            else return false;
            return true;
        }
        /// <summary>
        /// sets triface flag down
        /// </summary>
        /// <param name="f">flag mask</param>
        public void resetFaceFlag(int f)
        {
            switch (f)
            {
                case TriMeshVOI.MESH_DELETED_ITEM: MESH_DELETED_ITEM = false; break;
                case TriMeshVOI.MESH_FEATURE_ITEM: MESH_FEATURE_ITEM = false; break;
                case TriMeshVOI.MESH_STATIC_ITEM: MESH_STATIC_ITEM = false; break;
                case TriMeshVOI.MESH_TEMP_ITEM: MESH_TEMP_ITEM = false; break;
                case TriMeshVOI.MESH_INTERSECTING_ITEM: MESH_INTERSECTING_ITEM = false; break;
                case TriMeshVOI.MESH_PARTIALLY_VALID_ITEM: MESH_PARTIALLY_VALID_ITEM = false; break;
                case TriMeshVOI.MESH_VALID_ITEM: MESH_VALID_ITEM = false; break;
                case TriMeshVOI.MESH_UNVISITED_ITEM: MESH_UNVISITED_ITEM = false; break;
                default: throw new TriMeshException("Unrecognized flag");
            }
        }
        /// <summary>
        /// sets triface flag up
        /// </summary>
        /// <param name="f">flag mask</param>
        public void setFaceFlag(int f)
        {
            switch (f)
            {
                case TriMeshVOI.MESH_DELETED_ITEM: MESH_DELETED_ITEM = true; break;
                case TriMeshVOI.MESH_FEATURE_ITEM: MESH_FEATURE_ITEM = true; break;
                case TriMeshVOI.MESH_STATIC_ITEM: MESH_STATIC_ITEM = true; break;
                case TriMeshVOI.MESH_TEMP_ITEM: MESH_TEMP_ITEM = true; break;
                case TriMeshVOI.MESH_INTERSECTING_ITEM: MESH_INTERSECTING_ITEM = true; break;
                case TriMeshVOI.MESH_PARTIALLY_VALID_ITEM: MESH_PARTIALLY_VALID_ITEM = true; break;
                case TriMeshVOI.MESH_VALID_ITEM: MESH_VALID_ITEM = true; break;
                case TriMeshVOI.MESH_UNVISITED_ITEM: MESH_UNVISITED_ITEM = true; break;
            }
        }
        /// <summary>
        /// returns value of triface flag
        /// </summary>
        /// <param name="flag">integer having 1 at wanted flag location and all other 0's</param>
        /// <returns>true if flag is set</returns>
        public bool getFaceFlag(int flag)
        {
            switch (flag)
            {
                case TriMeshVOI.MESH_DELETED_ITEM: return MESH_DELETED_ITEM;
                case TriMeshVOI.MESH_FEATURE_ITEM: return MESH_FEATURE_ITEM;
                case TriMeshVOI.MESH_STATIC_ITEM: return MESH_STATIC_ITEM;
                case TriMeshVOI.MESH_TEMP_ITEM: return MESH_TEMP_ITEM;
                case TriMeshVOI.MESH_INTERSECTING_ITEM: return MESH_INTERSECTING_ITEM;
                case TriMeshVOI.MESH_PARTIALLY_VALID_ITEM: return MESH_PARTIALLY_VALID_ITEM;
                case TriMeshVOI.MESH_VALID_ITEM: return MESH_VALID_ITEM;
                case TriMeshVOI.MESH_UNVISITED_ITEM: return MESH_UNVISITED_ITEM;
                default: throw new TriMeshException("Unrecognized flag.");
            }
        }
        /// <summary>
        /// Resolves whether two faces are neighbouring each other in the mesh so 
        /// that they have a common side. 
        /// </summary>
        /// <param name="face">face</param>
        /// <returns>true if faces are neighbours</returns>
        public bool AreFacesNeighbours(TriFace face)
        {
            int p = 0;
            for (int i = 0; i < 3; i++)
            {
                for (int j = 0; j < 3; j++)
                {
                    if (mexel[i] == face.mexel[j]) p++;
                }
            }
            if (p == 3) throw new InvalidProgramException("Two identical faces");
            return (p == 2);
        }
        /// <summary>
        /// Returns true if index is neighbour of this face.
        /// </summary>
        /// <param name="j">0-based index in mesh</param>
        /// <returns>true if index is neighbour of this face</returns>
        public bool HasNg(int j)
        {
            for (int i = 0; i < Ng.Length; i++)
            {
                if (Ng[i] == j)
                {
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// Returns true if index is neighbour of this face.
        /// </summary>
        /// <param name="t">tested triface</param>
        /// <returns>true if index is neighbour of this face</returns>
        public bool IsNg(TriFace t)
        {
            int common_vertexes = 0;
            for (int i = 0; i < Ng.Length; i++)
            {
                if (t.HasMexel(Ng[i]))
                {
                    common_vertexes++;
                }
            }
            if (common_vertexes == 3) throw new TriMeshException("Overlapping faces.");
            if (common_vertexes == 2) return true;
            return false;
        }
        /// <summary>
        /// Returns true if index is mexel of this face.
        /// </summary>
        /// <param name="j">0-based mexel index in mesh</param>
        /// <returns>true if index is mexel of this face</returns>
        public bool HasMexel(int j)
        {
            for (int i = 0; i < 3; i++)
            {
                if (mexel[i] == j)
                {
                    return true;
                }
            }
            return false;
        }
        /// <summary>
        /// Removes neighbour mexel index from this triface.
        /// </summary>
        /// <param name="i">0-based index in mesh</param>
        /// <returns>true if removal was success</returns>
        public bool RemoveMexel(int i)
        {
            if (mexel[0] == i) mexel[0] = -1;
            else if (mexel[1] == i) mexel[1] = -1;
            else if (mexel[2] == i) mexel[2] = -1;
            else return false;
            return true;
        }
        /// <summary>
        /// Flips face's normal according to its corner locations
        /// </summary>
        public void Flip()
        {
            //flip normal by changing corner indexes
            int tmp = mexel[0];
            mexel[0] = mexel[1];
            mexel[1] = tmp;
        }
        /// <summary>
        /// Gets string representation of this object.
        /// </summary>
        /// <returns>object information as string</returns>
        public override string ToString()
        {
            return "TriFace [" + mexel[0] + "," + mexel[1] + "," + mexel[2] + "]";
        }
    }
}