﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// Triangular mesh implementation. Implementation is here because 
    /// ManagedITK does not currently have easy trimesh implementation.
    /// </summary>
    public class TriMeshVOI : ROIObject
    {
        /// <summary>
        /// false for skipping all consistency tests
        /// </summary>
        public static readonly bool TESTS_ENABLED = true;
        /// <summary>
        /// Maximum number of neighbours for each mexel
        /// </summary>
        public static readonly int Mexel_N = 10;
        /// <summary>
        /// Designated value for deleted mesh item. 
        /// Deleted item is going to be deleted and should not be used in any way.
        /// </summary>
        public const byte MESH_DELETED_ITEM = 1;
        /// <summary>
        /// Designated value for feature mesh item.
        /// So-called feature item in mesh should not be deformed with mesh.
        /// </summary>
        public const byte MESH_FEATURE_ITEM = 2;
        /// <summary>
        /// Designated value for feature mesh item.
        /// Static item was not changed in the previous round.
        /// </summary>
        public const byte MESH_STATIC_ITEM = 8;
        /// <summary>
        /// Temporary marker for item, should be cleared after use
        /// </summary>
        public const byte MESH_TEMP_ITEM = 4;
        /// <summary>
        /// Marker for item that intersects with other item
        /// </summary>
        public const byte MESH_INTERSECTING_ITEM = 16;
        /// <summary>
        /// Marker for item that is partially valid (intersecting with the mesh)
        /// </summary>
        public const byte MESH_PARTIALLY_VALID_ITEM = 32;
        /// <summary>
        /// Marker for item that is valid (does not intersect with the mesh)
        /// </summary>
        public const byte MESH_VALID_ITEM = 64;
        /// <summary>
        /// Marker for item that is unvisited
        /// </summary>
        public const byte MESH_UNVISITED_ITEM = 128;
        /// <summary>
        /// Center of the mesh.
        /// </summary>
        public Point center = new Point();
        /// <summary>
        /// mesh rigidness (max curvature angle)
        /// </summary>
        public double max_angle = 0.0;
        /// <summary>
        /// mesh bounds [x min, x max, y min, y max, z min, z max]
        /// </summary>
        public Limits bounds = new Limits(0, 0, 0, 0, 0, 0);
        /// <summary>
        /// (desired/expected) mean edge length
        /// </summary>
        public double mean_edge_length;
        /// <summary>
        /// (desired/expected) maximum number of vertexes
        /// </summary>
        public int max_size;
        /// <summary>
        /// array of vertex
        /// </summary>
        public List<Mexel> mexels = new List<Mexel>();
        /// <summary>
        /// array of faces
        /// </summary>
        public List<TriFace> faces = new List<TriFace>();
        /// <summary>
        /// constructs mesh with no faces or mexels at [0,0,0]
        /// </summary>
        public TriMeshVOI()
        {
            max_angle = Math.PI / 5.0;
            mean_edge_length = 1.0;
            max_size = 100000;
            bounds.AddLimit(float.MinValue, float.MaxValue);
            bounds.AddLimit(float.MinValue, float.MaxValue);
            bounds.AddLimit(float.MinValue, float.MaxValue);
        }

        /// <summary>
        /// Protected FillMatrix method implemented in child ROI classes
        /// </summary>
        protected override void FillMatrix() {
            
        }

        /// <summary>
        /// Protected CreateMatrix method implemented in child ROI classes
        /// </summary>
        protected override void CreateMatrix(params int[] dimensions) {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Initializes 3D-mesh with sphere shape starting 
        /// from docecaedron. Old data in mesh will be cleared.
        /// </summary>
        public void Initialize()
        {
            int i = 0;
            int j = 0;
            int k = 0;
            int l = 0;
            float fii = (float)((1.0 + Math.Sqrt(5)) / 2.0);
            Mexel mexel = new Mexel();
            Point[] dodecahedron = new Point[20];
            Point p = new Point();
            Point p1 = new Point();
            Point p2 = new Point();
            double dist;
            //center of mesh
            Point z = new Point(0, 0, 0);

            Clear();

            //create 12 pentagons (center of dodecaedron at 0,0,0)
            //starting from lowest and going up to highest point
            dodecahedron[0] = new Point(-1.0f, -1.0f, -1.0f);
            dodecahedron[1] = new Point(-1.0f, -1.0f, 1.0f);
            dodecahedron[2] = new Point(-1.0f, 1.0f, -1.0f);
            dodecahedron[3] = new Point(-1.0f, 1.0f, 1.0f);
            dodecahedron[4] = new Point(1.0f, -1.0f, -1.0f);
            dodecahedron[5] = new Point(1.0f, -1.0f, 1.0f);
            dodecahedron[6] = new Point(1.0f, 1.0f, -1.0f);
            dodecahedron[7] = new Point(1.0f, 1.0f, 1.0f);
            dodecahedron[8] = new Point(0.0f, -1 / fii, -fii);
            dodecahedron[9] = new Point(0.0f, -1 / fii, fii);
            dodecahedron[10] = new Point(0.0f, 1 / fii, -fii);
            dodecahedron[11] = new Point(0.0f, 1 / fii, fii);
            dodecahedron[12] = new Point(-1 / fii, -fii, 0.0f);
            dodecahedron[13] = new Point(-1 / fii, fii, 0.0f);
            dodecahedron[14] = new Point(1 / fii, -fii, 0.0f);
            dodecahedron[15] = new Point(1 / fii, fii, 0.0f);
            dodecahedron[16] = new Point(-fii, 0.0f, -1 / fii);
            dodecahedron[17] = new Point(-fii, 0.0f, 1 / fii);
            dodecahedron[18] = new Point(fii, 0.0f, -1 / fii);
            dodecahedron[19] = new Point(fii, 0.0f, 1 / fii);

            //create dodecahedron into mesh
            for (i = 0; i < 20; i++)
            {
                AddMexel(new Mexel(dodecahedron[i]));
            }

            Spherize(-1);
            int adds = 0;
            //add connection indexes to neighbouring mexels
            for (i = 0; i < SizeMexels; i++)
            {
                for (j = 0; j < SizeMexels; j++)
                {
                    if (i == j) continue;
                    //add connection if it is close enough
                    dist = mexels[i].p.dist(mexels[j].p);
                    if (dist <= 2 / fii + 0.01)
                    {
                        k = 0;
                        while (mexels[i].Ng[k] != -1) k++;
                        mexels[i].Ng[k] = j;
                        adds++;
                    }
                }
            }
            //add centroids for all pentagons in order to create pentakis dodecahedron
            AddPentagonCentroid(1, 17, 18, 2, 13);
            AddPentagonCentroid(18, 4, 12, 10, 2);
            AddPentagonCentroid(13, 2, 10, 6, 15);
            AddPentagonCentroid(10, 12, 8, 20, 6);
            AddPentagonCentroid(3, 14, 4, 18, 17);
            AddPentagonCentroid(19, 5, 15, 6, 20);
            AddPentagonCentroid(7, 19, 20, 8, 16);
            AddPentagonCentroid(8, 12, 4, 14, 16);
            AddPentagonCentroid(9, 1, 13, 15, 5);
            AddPentagonCentroid(11, 9, 5, 19, 7);
            AddPentagonCentroid(11, 7, 16, 14, 3);
            AddPentagonCentroid(17, 1, 9, 11, 3);

            for (i = 0; i < 0; i++)
            {
                //spherize model
                Divide();
                Spherize(-1);
                Console.WriteLine(SizeMexels + " vertices");
            }

            //set face normals
            l = 0;
            for (i = 0; i < SizeFaces; i++)
            {
                p1.x = mexels[faces[i].mexel[0]].p.x - mexels[faces[i].mexel[1]].p.x;
                p1.y = mexels[faces[i].mexel[0]].p.y - mexels[faces[i].mexel[1]].p.y;
                p1.z = mexels[faces[i].mexel[0]].p.z - mexels[faces[i].mexel[1]].p.z;
                p2.x = mexels[faces[i].mexel[2]].p.x - mexels[faces[i].mexel[1]].p.x;
                p2.y = mexels[faces[i].mexel[2]].p.y - mexels[faces[i].mexel[1]].p.y;
                p2.z = mexels[faces[i].mexel[2]].p.z - mexels[faces[i].mexel[1]].p.z;
                p = Point.Cross(p1, p2);
                //set normal (pointing outside)
                faces[i].n = p;
                faces[i].n.Normalize();
                //assert that normal is pointing outside
                p.x = (mexels[faces[i].mexel[0]].p.x + mexels[faces[i].mexel[1]].p.x + mexels[faces[i].mexel[2]].p.x) / 3.0;
                p.y = (mexels[faces[i].mexel[0]].p.y + mexels[faces[i].mexel[1]].p.y + mexels[faces[i].mexel[2]].p.y) / 3.0;
                p.z = (mexels[faces[i].mexel[0]].p.z + mexels[faces[i].mexel[1]].p.z + mexels[faces[i].mexel[2]].p.z) / 3.0;
                p *= 10.0;
                dist = z.dist(p);
                p.x += faces[i].n.x;
                p.y += faces[i].n.y;
                p.z += faces[i].n.z;
                if (dist > z.dist(p))
                {
                    l++;
                    j = faces[i].mexel[0];
                    faces[i].mexel[0] = faces[i].mexel[2];
                    faces[i].mexel[2] = j;
                }
            }
            for (i = 0; i < SizeFaces; i++)
            {
                p1.x = mexels[faces[i].mexel[0]].p.x - mexels[faces[i].mexel[1]].p.x;
                p1.y = mexels[faces[i].mexel[0]].p.y - mexels[faces[i].mexel[1]].p.y;
                p1.z = mexels[faces[i].mexel[0]].p.z - mexels[faces[i].mexel[1]].p.z;
                p2.x = mexels[faces[i].mexel[2]].p.x - mexels[faces[i].mexel[1]].p.x;
                p2.y = mexels[faces[i].mexel[2]].p.y - mexels[faces[i].mexel[1]].p.y;
                p2.z = mexels[faces[i].mexel[2]].p.z - mexels[faces[i].mexel[1]].p.z;
                p = Point.Cross(p1, p2);
                //set normal (pointing outside)
                faces[i].n = p;
                faces[i].n.Normalize();
                //assert that normal is pointing outside
                p.x = (mexels[faces[i].mexel[0]].p.x + mexels[faces[i].mexel[1]].p.x + mexels[faces[i].mexel[2]].p.x) / 3.0;
                p.y = (mexels[faces[i].mexel[0]].p.y + mexels[faces[i].mexel[1]].p.y + mexels[faces[i].mexel[2]].p.y) / 3.0;
                p.z = (mexels[faces[i].mexel[0]].p.z + mexels[faces[i].mexel[1]].p.z + mexels[faces[i].mexel[2]].p.z) / 3.0;
                p *= 10.0;
                dist = z.dist(p);
                p.x += faces[i].n.x;
                p.y += faces[i].n.y;
                p.z += faces[i].n.z;
                if (dist > z.dist(p))
                {
                    throw new TriMeshException("Invalid normal");
                }
            }
            //update face neighbours information
            UpdateFaceNgs();
            TestConsistency_FaceNormals();
        }
        /// <summary>
        /// Adds new face into mesh at the end of faces list. Duplicate face indexes are checked before addition.
        /// </summary>
        /// <param name="f">added face</param>
        public void AddFace(TriFace f)
        {
            if (f.mexel[0] == f.mexel[1] || f.mexel[1] == f.mexel[2] || f.mexel[2] == f.mexel[0])
                if (f.mexel[0] != -1 || f.mexel[1] != -1 || f.mexel[2] != -1)
                    throw new TriMeshException("Duplicate face mexel index");
            faces.Add(f);
        }
        /// <summary>
        /// Gets mesh face
        /// </summary>
        /// <param name="i">0-based index</param>
        /// <returns>face in mesh</returns>
        public TriFace GetFace(int i)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            return faces[i];
        }
        /// <summary>
        /// Adds new mexel into mesh at the end of mexels list.
        /// </summary>
        /// <param name="m">added mexel</param>
        public void AddMexel(Mexel m)
        {
            mexels.Add(m);
        }
        /// <summary>
        /// Removes (marks as removed) face i from mesh
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        public void RemoveFace(int i)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            faces[i].MESH_DELETED_ITEM = true;
        }
        /// <summary>
        /// Removes (marks as removed) mexel i from mesh
        /// </summary>
        /// <param name="i">mexel index [0.. No mexels]</param>
        public void RemoveMexel(int i)
        {
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count)
            {
                throw new TriMeshException("too big mexel index:" + i + ">=" + (int)mexels.Count);
            }
            mexels[i].flag = (byte)(mexels[i].flag | MESH_DELETED_ITEM);
        }
        /// <summary>
        /// Return true if face i is deleted (marked as deleted)
        /// </summary>
        /// <param name="i">tested face index [0..No faces]</param>
        /// <returns>true for deleted face</returns>
        public bool IsFaceDeleted(int i)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            return faces[i].MESH_DELETED_ITEM;
        }
        /// <summary>
        /// Return true if mexel is deleted (marked as deleted)
        /// </summary>
        /// <param name="i">tested mexel index [0..No mexels]</param>
        /// <returns>true for deleted mexel</returns>
        public bool IsMexelDeleted(int i)
        {
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count)
                throw new TriMeshException("too big mexel index:" + i + ">=" + (int)mexels.Count);
            return (mexels[i].flag & MESH_DELETED_ITEM) != 0;
        }
        /// <summary>
        /// Calculates normal.
        /// </summary>
        /// <param name="tri">face in the mesh</param>
        /// <param name="mesh">related mesh</param>
        /// <returns>normal vector of the face in the mesh</returns>
        public static Point Normal(TriFace tri, TriMeshVOI mesh)
        {
            if (tri.mexel[0] < 0 || tri.mexel[0] >= mesh.mexels.Count)
                throw new TPCException("Face mexel index 0 is out of bounds. Cannot calculate normal.");
            if (tri.mexel[1] < 0 || tri.mexel[1] >= mesh.mexels.Count)
                throw new TPCException("Face mexel index is out of bounds. Cannot calculate normal.");
            if (tri.mexel[2] < 0 || tri.mexel[2] >= mesh.mexels.Count)
                throw new TPCException("Face mexel index is out of bounds. Cannot calculate normal.");
            Point p1 = new Point();
            Point p2 = new Point();
            p1.x = mesh.mexels[tri.mexel[0]].p.x - mesh.mexels[tri.mexel[1]].p.x;
            p1.y = mesh.mexels[tri.mexel[0]].p.y - mesh.mexels[tri.mexel[1]].p.y;
            p1.z = mesh.mexels[tri.mexel[0]].p.z - mesh.mexels[tri.mexel[1]].p.z;
            p2.x = mesh.mexels[tri.mexel[2]].p.x - mesh.mexels[tri.mexel[1]].p.x;
            p2.y = mesh.mexels[tri.mexel[2]].p.y - mesh.mexels[tri.mexel[1]].p.y;
            p2.z = mesh.mexels[tri.mexel[2]].p.z - mesh.mexels[tri.mexel[1]].p.z;
            //set normal (originally pointing outside)
            return Point.Cross(p1, p2);
        }
        /// <summary>
        /// Updates face's normal according to its corner locations. 
        /// Face normal vector might not be up-to-date unless it is specificly 
        /// updated.
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        public void UpdateNormal(int i)
        {
            if (i < 0 || i >= faces.Count)
                throw new TPCException("Face index is out of bounds. Cannot update normal.");
            faces[i].n = Normal(faces[i], this);
        }
        /// <summary>
        /// Flips face's normal according to its corner locations
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        public void FlipNormal(int i)
        {
            if (i < 0 || i >= faces.Count)
                throw new TPCException("Face index is out of bounds. Cannot update normal.");
            //flip normal by changing corner indexes
            faces[i].Flip();
            //update normal
            UpdateNormal(i);
        }
        /// <summary>
        /// Calculates area of face
        /// </summary>
        /// <returns>area of triangle</returns>
        public static double GetFaceArea(ref Point a, ref Point b, ref Point c)
        {
            Point v = new Point();
            Point w = new Point();
            Point.Sub(ref v, b, a);
            Point.Sub(ref w, c, a);
            return Point.Cross(v, w).norm() / 2.0;
        }
        /// <summary>
        /// Calculates area of face
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <returns>face area in units</returns>
        public double GetFaceArea(int i)
        {
            if (i < 0 || i >= faces.Count)
                throw new TPCException("Face index is out of bounds. Cannot update normal.");
            return GetFaceArea(ref mexels[faces[i].mexel[0]].p, ref mexels[faces[i].mexel[1]].p, ref mexels[faces[i].mexel[2]].p);
        }
        /// <summary>
        /// Deletes all mexels and faces that are marked as deleted. 
        /// This means that the mesh no longer contains mexels or faces marked as deleted.
        /// </summary>
        public void PerformDeletions()
        {
            int i = 0;
            int j = 0;
            int k = 0;
            //indexes of deleted mexels in old vectors
            List<int> deleted_mexels = new List<int>();
            List<int> deleted_faces = new List<int>();
            //new mexel indexes
            List<int> shift_mexels = new List<int>();
            List<int> shift_faces = new List<int>();
            List<Mexel> tmp_mexels = new List<Mexel>();
            List<TriFace> tmp_faces = new List<TriFace>();

            //gather indexes of deleted items
            for (i = 0; i < SizeMexels; i++)
            {
                if (IsMexelDeleted(i)) deleted_mexels.Add(i);
                shift_mexels.Add(i);
            }
            for (i = 0; i < SizeFaces; i++)
            {
                if (IsFaceDeleted(i)) deleted_faces.Add(i);
                shift_faces.Add(i);
            }
            if (deleted_faces.Count == 0 && deleted_mexels.Count == 0)
                return;

            //sort deleted indexes
            deleted_mexels.Sort();
            deleted_faces.Sort();

            //set shift tables for mexels and faces
            j = 0;
            k = 0;
            for (i = 0; i < SizeMexels; i++)
            {
                //shift index to the left
                shift_mexels[i] -= j;
                //if current index is deleted, shift next ones one more 
                //further to the left and start to test next one
                if (k < deleted_mexels.Count && i == deleted_mexels[k])
                {
                    j++;
                    k++;
                }
            }
            j = 0;
            k = 0;
            for (i = 0; i < SizeFaces; i++)
            {
                //shift index to the left
                shift_faces[i] -= j;
                //if current index is deleted, shift next ones one more 
                //further to the left and start to test next one
                if (k < deleted_faces.Count && i == deleted_faces[k])
                {
                    j++;
                    k++;
                }
            }
            //shift indexes to temporary vectors (no erasing is done)
            for (i = 0; i < SizeMexels; i++)
            {
                //do not copy deleted item
                if (IsMexelDeleted(i)) continue;
                //copy item and shift its indexes
                tmp_mexels.Add(mexels[i]);
                //shift neighbour connection indexes
                for (j = 0; j < Mexel_N; j++)
                {
                    if (tmp_mexels[(int)tmp_mexels.Count - 1].Ng[j] == -1) break;
                    tmp_mexels[(int)tmp_mexels.Count - 1].Ng[j] = shift_mexels[mexels[i].Ng[j]];
                }
                //shift face connection indexes
                for (j = 0; j < Mexel_N; j++)
                {
                    if (tmp_mexels[(int)tmp_mexels.Count - 1].triface[j] == -1) break;
                    tmp_mexels[(int)tmp_mexels.Count - 1].triface[j] = shift_faces[mexels[i].triface[j]];
                }
            }
            TestConsistency_FaceMexels();
            for (i = 0; i < SizeFaces; i++)
            {
                //do not copy deleted item
                if (IsFaceDeleted(i)) continue;
                //copy item and shift its indexes
                tmp_faces.Add(faces[i]);
                //shift face mexel connection indexes
                for (j = 0; j < 3; j++)
                {
                    tmp_faces[(int)tmp_faces.Count - 1].mexel[j] = shift_mexels[faces[i].mexel[j]];
                }
            }

            //update mexels and faces
            mexels = tmp_mexels;
            faces = tmp_faces;
        }
        /// <summary>
        /// deletes all faces and mexels from mesh
        /// </summary>
        public void Clear()
        {
            mexels.Clear();
            faces.Clear();
        }
        /// <summary>
        /// resets all item flag
        /// </summary>
        /// <param name="reset_items">mask for reseting all items</param>
        public void ResetItemFlags(byte reset_items)
        {
            int i = 0;
            for (i = 0; i < SizeMexels; i++)
                mexels[i].flag = (byte)(mexels[i].flag & (~reset_items));
            for (i = 0; i < SizeFaces; i++)
            {
                faces[i].MESH_DELETED_ITEM = false;
                faces[i].MESH_FEATURE_ITEM = false;
                faces[i].MESH_INTERSECTING_ITEM = false;
                faces[i].MESH_PARTIALLY_VALID_ITEM = false;
                faces[i].MESH_STATIC_ITEM = false;
                faces[i].MESH_TEMP_ITEM = false;
                faces[i].MESH_UNVISITED_ITEM = false;
                faces[i].MESH_VALID_ITEM = false;
            }
        }
        /// <summary>
        /// sets mexel flag down
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <param name="flag">integer having 1 at flag value and all other 0's</param>
        public void ResetMexelFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count)
                throw new TriMeshException("too big mexel index:" + i + ">=" + (int)mexels.Count);
            mexels[i].flag = (byte)(mexels[i].flag & (~flag));
        }
        /// <summary>
        /// sets mexel flag up
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <param name="flag">integer having 1 at flag value and all other 0's</param>
        public void SetMexelFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count)
                throw new TriMeshException("too big mexel index:" + i + ">=" + (int)mexels.Count);
            mexels[i].flag = (byte)(mexels[i].flag | flag);
        }
        /// <summary>
        /// returns value of mexel flag
        /// </summary>
        /// <param name="i">mexel index (0-based)</param>
        /// <param name="flag">flag mask</param>
        /// <returns>flag value, true or false</returns>
        public bool GetMexelFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count)
                throw new TriMeshException("too big mexel index:" + i + ">=" + (int)mexels.Count);
            return (mexels[i].flag & flag) != 0;
        }
        /// <summary>
        /// sets triface flag down
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <param name="flag">integer having 1 at flag value and all other 0's</param>
        public void ResetFaceFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            faces[i].resetFaceFlag(flag);
        }
        /// <summary>
        /// sets triface flag up
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <param name="flag">integer having 1 at flag value and all other 0's</param>
        public void SetFaceFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            faces[i].setFaceFlag(flag);
        }
        /// <summary>
        /// returns value of triface flag
        /// </summary>
        /// <param name="i">face index [0..No faces]</param>
        /// <param name="flag">integer having 1 at flag value and all other 0's</param>
        /// <returns>true if flag is up, false otherwise</returns>
        public bool GetFaceFlag(int i, int flag)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count)
                throw new TriMeshException("too big face index:" + i + ">=" + (int)faces.Count);
            return faces[i].getFaceFlag(flag);
        }
        /// <summary>
        /// Returns number of mexels (underlaying vector size)
        /// </summary>
        /// <returns>number of mexels using memory</returns>
        public int SizeMexels
        {
            get
            {
                return mexels.Count;
            }
        }
        /// <summary>
        /// Finds face from mesh
        /// </summary>
        /// <param name="face">face foor lookup</param>
        /// <returns>face index, or -1 if not found</returns>
        public int FindFace(TriFace face)
        {
            for (int i = 0; i < faces.Count; i++)
            {
                if (faces[i].HasMexel(face.mexel[0]) && faces[i].HasMexel(face.mexel[1]) && faces[i].HasMexel(face.mexel[2]))
                    if (!IsFaceDeleted(i))
                        return i;
            }
            return -1;
        }
        /// <summary>
        /// Finds mexel from mesh
        /// </summary>
        /// <param name="p">location for mexel</param>
        /// <returns>mexel index, or -1 if not found</returns>
        public int FindMexel(Point p)
        {
            for (int i = 0; i < mexels.Count; i++)
            {
                if (p.Equals(mexels[i].p))
                    return i;
            }
            return -1;
        }
        /// <summary>
        /// Returns number of faces (underlaying vector size)
        /// </summary>
        /// <returns>number of faces in memory</returns>
        public int SizeFaces
        {
            get
            {
                return faces.Count;
            }
        }
        /// <summary>
        /// Writes this mesh into file in VRML 2.0 format. Use VRML viewer to display written data.
        /// </summary>
        /// <param name="filename">filename (*.wrl)</param>
        public void SaveVRML2(string filename)
        {

            System.IO.StreamWriter wrlFile = new System.IO.StreamWriter(filename, false);
            int i = 0;
            Point p = new Point();
            float R = 0.0f;
            float G = 0.0f;
            float B = 0.0f;

            //write header
            wrlFile.Write("#VRML V2.0 utf8\r\n");
            wrlFile.Write("\r\n");
            wrlFile.Write("Viewpoint {\r\n");
            wrlFile.Write("\torientation 0 0 1 0.75\r\n");
            wrlFile.Write("\tposition -10 0 100\r\n");
            wrlFile.Write("}\r\n");
            wrlFile.Write("\r\n");
            wrlFile.Write("Transform {\r\n");
            wrlFile.Write("\ttranslation\t");
            wrlFile.Write((-center.x).ToString("G").Replace(',', '.') + " " + (-center.y).ToString("G").Replace(',', '.') + " " + (-center.z).ToString("G").Replace(',', '.'));
            wrlFile.Write("\r\n");
            wrlFile.Write("\tchildren [");
            wrlFile.Write("\t\t\r\n");
            wrlFile.Write("\t\tShape {\r\n");
            wrlFile.Write("\t\t\tappearance Appearance {\r\n");
            wrlFile.Write("\t\t\t\tmaterial Material { diffuseColor\t.8 .8 .8 emissiveColor\t.8 .8 .8 }\r\n");
            wrlFile.Write("\t\t\t}\r\n");
            wrlFile.Write("\t\t\tgeometry IndexedFaceSet {\r\n");
            //write colors
            wrlFile.Write("\t\t\t\tcolor Color {");
            wrlFile.Write("\r\n");
            wrlFile.Write("\t\t\t\tcolor [\r\n");
            for (i = 0; i < SizeMexels; i++)
            {
                wrlFile.Write("\t\t\t\t\t");
                R = 0;
                G = 0;
                B = 0;
                //if (getMexelFlag(i, TriMesh.MESH_FEATURE_ITEM)) R = 1.0f;
                if (GetMexelFlag(i, TriMeshVOI.MESH_INTERSECTING_ITEM)) G = 1.0f;
                if (GetMexelFlag(i, TriMeshVOI.MESH_TEMP_ITEM)) G = R = B = 1.0f;
                wrlFile.Write(R + " " + G + " " + B + "\n");
                wrlFile.Flush();
            }
            wrlFile.Write("\t\t\t\t]\r\n");
            wrlFile.Write("\t\t\t\t}\r\n");
            //write coordinates
            wrlFile.Write("\t\t\t\tcoord Coordinate {\r\n");
            wrlFile.Write("\t\t\t\t\tpoint [\r\n");
            //write mexel coordinates
            for (i = 0; i < SizeMexels; i++)
            {
                wrlFile.Write("\t\t\t\t\t\t" + mexels[i].p.x.ToString("G").Replace(',', '.') +
                                         " " + mexels[i].p.y.ToString("G").Replace(',', '.') +
                                         " " + mexels[i].p.z.ToString("G").Replace(',', '.') + ",\n");
                wrlFile.Flush();
            }
            wrlFile.Write("\t\t\t\t\t]\r\n");
            wrlFile.Write("\t\t\t\t}\r\n");

            //write faces
            wrlFile.Write("\r\n");
            wrlFile.Write("\t\t\t\tcoordIndex [\r\n");
            for (i = 0; i < SizeFaces; i++)
            {
                wrlFile.Write("\t\t\t\t\t");
                wrlFile.Write(faces[i].mexel[1] + ", ");
                wrlFile.Write(faces[i].mexel[0] + ", ");
                wrlFile.Write(faces[i].mexel[2] + ", ");
                wrlFile.Write("-1,\n");
                wrlFile.Flush();
            }
            wrlFile.Write("\t\t\t\t]\r\n");
            wrlFile.Write("\t\t\t}\r\n");
            wrlFile.Write("\t\t}\r\n");
            wrlFile.Write("\t]\r\n");
            wrlFile.Write("}\r\n");

            //write feature mexel coordinates with blue
            wrlFile.Flush();
            wrlFile.Close();
        }

        /// <summary>
        /// Writes this mesh into file in VRML 2.0 format. Use VRML viewer to display writen data.
        /// </summary>
        /// <param name="filename">filename (*.wrl)</param>
        /// <param name="im_i">reference image</param>
        public static void SaveVRML2(string filename, TPClib.Image.Image im_i)
        {
            System.IO.StreamWriter wrlFile = new System.IO.StreamWriter(filename, false);
            int x = 0;
            int y = 0;
            int z = 0;
            int k = 0;
            int x_length = 0;
            int x_start = 0;

            //write header
            wrlFile.Write("#VRML V2.0 utf8\r\n");
            wrlFile.Write("\r\n");
            wrlFile.Write("Viewpoint {\r\n");
            wrlFile.Write("\torientation 0 0 1 0.75\r\n");
            wrlFile.Write("\tposition -10 0 100\r\n");
            wrlFile.Write("}\r\n");
            wrlFile.Write("\r\n");

            wrlFile.Write("DEF BoxShape Shape {\r\n");
            wrlFile.Write("\tappearance Appearance {\tmaterial Material { diffuseColor .8 .8 .8 }\t}\r\n");
            wrlFile.Write("\tgeometry Box { size " + 0.78 + " " + 0.78 + " " + 4.4 + " }\r\n");
            wrlFile.Write("}\r\n");

            x_start = -1;
            x_length = 0;
            for (x = 0; x < im_i.width; x++)
            {
                for (y = 0; y < im_i.height; y++)
                {
                    for (z = 0; z < im_i.planes; z++)
                    {
                        if (im_i[x, y, z] != 0)
                        {
                            if (x_start == -1) x_start = x;
                            x_length++;
                        }
                        if (x_length > 0 && (x == im_i.width - 1 || im_i[x + 1, y, z] == 0))
                        {
                            wrlFile.Write("Transform {\r\n");
                            wrlFile.Write("\ttranslation\t" + (x_start + x_length / 2 - (im_i.width / 2)) * 0.78 + " ");
                            wrlFile.Write((y - (im_i.height / 2)) * 0.78 + " ");
                            wrlFile.Write((z - (im_i.planes / 2)) * 4.4 + "\r\n");

                            wrlFile.Write("\tchildren Shape {\r\n");
                            wrlFile.Write("\t\tappearance Appearance {\r\n");
                            wrlFile.Write("\t\t\tmaterial Material { \r\n");
                            wrlFile.Write("\t\t\t\tdiffuseColor .6 .6 .6\r\n");
                            wrlFile.Write("\t\t\t\tambientIntensity .5\r\n");
                            wrlFile.Write("\t\t\t}\r\n");
                            wrlFile.Write("\t\t}\r\n");
                            wrlFile.Write("\t\tgeometry Box { size " + 0.78 * x_length + " " + 0.78 + " " + 4.4 + " }\r\n");
                            wrlFile.Write("\t}\r\n");
                            wrlFile.Write("}\r\n");
                            k++;
                            x_start = -1;
                            x_length = 0;
                        }
                    }
                }
            }
            wrlFile.Flush();
            wrlFile.Close();
        }

        //functions for deformations
        /// <summary>
        /// Calculates normal at mexel location. Normal is calculated as the 
        /// mean direction of neighbouring face normals. Therefore the face 
        /// normals should be updated before calling this method.
        /// </summary>
        /// <param name="i">mexel index (0-based)</param>
        /// <returns>normal at mexel location</returns>
        public Point GetLocalNormal(int i)
        {
            int j = 0;
            if (i < 0) throw new TriMeshException("negative mexel index");
            if (i >= (int)mexels.Count) throw new TriMeshException("too big mexel index");
            //calculate local surface normal
            Point n = new Point();
            //for all adjanced triangles
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[i].triface[j] == -1) break;
                //add normal to mean normal
                n.x += faces[mexels[i].triface[j]].n.x;
                n.y += faces[mexels[i].triface[j]].n.y;
                n.z += faces[mexels[i].triface[j]].n.z;
            }
            //normalize normal
            n.Normalize();
            return n;
        }
        /// <summary>
        /// Generates integer list of face indexes (0, 1, 2, .. faces-1)
        /// </summary>
        /// <returns>list containing index values in order</returns>
        public List<int> GenerateFaceIndexes()
        {
            List<int> r = new List<int>(faces.Count);
            for (int i = 0; i < faces.Count; i++)
                r.Add(i);
            return r;
        }
        /// <summary>
        /// Calculates barycenter at face. 
        /// </summary>
        /// <param name="i">mexel index (0-based)</param>
        /// <returns>barycenter location</returns>
        public Point GetFaceBaryCenter(int i)
        {
            if (i < 0) throw new TriMeshException("negative face index");
            if (i >= (int)faces.Count) throw new TriMeshException("too big face index");
            Point q = new Point();
            for (int j = 0; j < 3; j++)
            {
                q.x += mexels[faces[i].mexel[j]].p.x;
                q.y += mexels[faces[i].mexel[j]].p.y;
                q.z += mexels[faces[i].mexel[j]].p.z;
            }
            q.x /= 3.0;
            q.y /= 3.0;
            q.z /= 3.0;
            return q;
        }
        /// <summary>
        /// Calculates barycenter at mexel neighbourhood. 
        /// </summary>
        /// <param name="i">mexel index (0-based)</param>
        /// <returns>barycenter location</returns>
        public Point GetMexelBaryCenter(int i)
        {
            Point q = new Point();
            int j = 0;
            for (; j < Mexel_N; j++)
            {
                if (mexels[i].Ng[j] == -1) break;
                q.x += mexels[mexels[i].Ng[j]].p.x;
                q.y += mexels[mexels[i].Ng[j]].p.y;
                q.z += mexels[mexels[i].Ng[j]].p.z;
            }
            if (j == 0) throw new TriMeshException("cannot calcualte barycenter because vertex has no neighbours");
            q.x /= (double)j;
            q.y /= (double)j;
            q.z /= (double)j;
            return q;
        }
        /// <summary>
        /// Resolves two common neighbours for two mexels that are adjanced according 
        /// to triangle faces.
        /// </summary>
        /// <param name="i">mexel 1 index (0-based)</param>
        /// <param name="j">mexel 2 index (0-based)</param>
        /// <returns>two neighbour and face indexes (0-based) in 4-length array</returns>
        public int[] GetTwoCommonNeighbours(int i, int j)
        {
            int l = 0;
            int Ng1 = -1;
            int Ng2 = -1;
            int Face1 = -1;
            int Face2 = -1;
            int t = 0;

            if (i == j) throw new TriMeshException("invalid parameters");
            if (mexels.Count == 0) throw new TriMeshException("mesh has no mexels");
            if (faces.Count == 0) throw new TriMeshException("mesh has no faces");
            if (i < 0) throw new TriMeshException("negative mexel index for 1st parameter");
            if (i >= (int)mexels.Count) throw new TriMeshException("too big mexel index for 1st parameter");
            if (IsMexelDeleted(i)) throw new TriMeshException("1st parameter mexel is deleted");
            if (j < 0) throw new TriMeshException("negative mexel index for 2nd parameter");
            if (j >= (int)mexels.Count) throw new TriMeshException("too big mexel index for 2nd parameter");
            if (IsMexelDeleted(j)) throw new TriMeshException("2nd parameter mexel is deleted");

            //assert that the mexels are neighbours
            l = 0;
            for (int k = 0; k < Mexel_N; k++)
            {
                if (mexels[i].Ng[k] == -1) break;
                if (mexels[i].Ng[k] == j) l++;
            }
            if (l != 1) throw new TriMeshException("Invalid number of neighbour");
            l = 0;
            for (int k = 0; k < Mexel_N; k++)
            {
                if (mexels[j].Ng[k] == -1) break;
                if (mexels[j].Ng[k] == i) l++;
            }
            if (l != 1) throw new TriMeshException("Invalid number of neighbour");

            //go through all faces of mexel
            for (int k = 0; k < Mexel_N; k++)
            {
                t = mexels[i].triface[k];
                if (t == -1) break;
                if (IsFaceDeleted(t)) continue;
                //go around the mexels of the face
                for (l = 0; l < 3; l++)
                {
                    if (IsMexelDeleted(faces[t].mexel[l])) throw new TriMeshException("face has deleted mexel");
                    //if one mexel is the other neighbour point (at other end of split vertex)
                    if (faces[t].mexel[l] == j)
                    {
                        if (Ng1 == -1)
                        {
                            Face1 = t;
                            if (l == 2) l = 0;
                            else l = l + 1;
                            if (faces[t].mexel[l] == i)
                            {
                                if (l == 2) l = 0;
                                else l = l + 1;
                                Ng1 = faces[t].mexel[l];
                            }
                            else
                            {
                                Ng1 = faces[t].mexel[l];
                            }
                            break;
                        }
                        else if (Ng2 == -1)
                        {
                            Face2 = t;
                            if (l == 2) l = 0;
                            else l = l + 1;
                            if (faces[t].mexel[l] == i)
                            {
                                if (l == 2) l = 0;
                                else l = l + 1;
                                Ng2 = faces[t].mexel[l];
                            }
                            else
                            {
                                Ng2 = faces[t].mexel[l];
                            }
                            return new int[] { Ng1, Ng2, Face1, Face2 };
                        }
                    }
                }
            }
            Console.WriteLine("Less than 2 common neighbours found in Mesh::getTwoCommonNeighbours");
            Console.WriteLine("Ng1:" + Ng1 + " Ng2:" + Ng2 + " Face1:" + Face1 + " Face2:" + Face2);
            Console.Write(" mexel i " + i + ":");
            Console.WriteLine("neighbours:");
            for (int k = 0; k < Mexel_N; k++)
            {
                //go around the mexels of the face
                Console.Write("	" + mexels[i].Ng[k]);
            }
            Console.WriteLine();

            Console.Write(" mexel j " + j + ":");
            Console.WriteLine("neighbours:");
            for (int k = 0; k < Mexel_N; k++)
            {
                //go around the mexels of the face
                Console.Write("	" + mexels[j].Ng[k]);
            }
            Console.WriteLine();

            Console.WriteLine("mexel i trifaces:");
            for (int k = 0; k < Mexel_N; k++)
            {
                //go around the mexels of the face
                Console.WriteLine("triface " + mexels[i].triface[k] + ":");
                if (mexels[i].triface[k] == -1) continue;
                for (l = 0; l < 3; l++)
                {
                    //if one mexel is the other neighbour point (at other end of split vertex)
                    Console.Write("	" + faces[mexels[i].triface[k]].mexel[l]);
                }
                Console.WriteLine();
            }
            Console.WriteLine("mexel j trifaces:");
            for (int k = 0; k < Mexel_N; k++)
            {
                //go around the mexels of the face           
                Console.WriteLine("triface " + mexels[j].triface[k] + ":");
                if (mexels[j].triface[k] == -1) continue;
                for (l = 0; l < 3; l++)
                {
                    //if one mexel is the other neighbour point (at other end of split vertex)
                    Console.Write("	" + faces[mexels[j].triface[k]].mexel[l]);
                }
                Console.WriteLine();
            }
            for (int k = 0; k < Mexel_N; k++)
            {
                if (mexels[i].triface[k] == -1) break;
                PrintFace_Matlab(mexels[i].triface[k], "i_" + mexels[i].triface[k], "g");
            }
            for (int k = 0; k < Mexel_N; k++)
            {
                if (mexels[j].triface[k] == -1) break;
                PrintFace_Matlab(mexels[j].triface[k], "j_" + mexels[j].triface[k], "b");
            }
            SaveVRML2("assertion.wrl");
            Print_Matlab(mexels[i].triface[0], 7);
            throw new TriMeshException("failed to find two common neighbours");
        }

        /// <summary>
        /// Removes face adjanced to mexel
        /// </summary>
        /// <param name="mexel_i">mexel index in mesh mexels array (0-based)</param>
        /// <param name="face_i">face index in mesh faces array (0-based)</param>
        public void RemoveFace(int mexel_i, int face_i)
        {
            //remove face_i from mexel_i
            int i = 0;
            int j = 0;
            //look for last triface from mexel's array
            for (; j < Mexel_N; j++) if (mexels[mexel_i].triface[j] == -1) break;
            j--;
            for (; i < Mexel_N; i++)
            {
                if (mexels[mexel_i].triface[i] == -1) break;
                if (mexels[mexel_i].triface[i] == face_i)
                {
                    if (j >= 0)
                    {
                        //move last item onto removed one
                        mexels[mexel_i].triface[i] = mexels[mexel_i].triface[j];
                        mexels[mexel_i].triface[j] = -1;
                    }
                }
            }
        }

        /// <summary>
        /// Adds face into mesh mexel.
        /// </summary>
        /// <param name="mexel_i">related mexel</param>
        /// <param name="face_i">face index to be added</param>
        public void AddFace(int mexel_i, int face_i)
        {
            //add face_i to mexel_i
            int j = 0;
            for (; j < Mexel_N; j++) if (mexels[mexel_i].triface[j] == -1) break;
            mexels[mexel_i].triface[j] = face_i;
        }

        /// <summary>
        /// Adds centroid point into pentagon shape thus creating triangle 
        /// </summary>
        /// <param name="p1">vertex of pentagon</param>
        /// <param name="p2">vertex of pentagon</param>
        /// <param name="p3">vertex of pentagon</param>
        /// <param name="p4">vertex of pentagon</param>
        /// <param name="p5">vertex of pentagon</param>
        void AddPentagonCentroid(int p1, int p2, int p3, int p4, int p5)
        {
            Point p1p2 = new Point();
            Point centroid = new Point();
            int i = 0;
            int j = 0;
            int k = 0;
            int l = 0;
            p1 -= 1;
            p2 -= 1;
            p3 -= 1;
            p4 -= 1;
            p5 -= 1;
            int neighbour_match = 0;

            //calculate centroid
            p1p2.x = mexels[p1].p.x + (mexels[p2].p.x - mexels[p1].p.x) / 2.0;
            p1p2.y = mexels[p1].p.y + (mexels[p2].p.y - mexels[p1].p.y) / 2.0;
            p1p2.z = mexels[p1].p.z + (mexels[p2].p.z - mexels[p1].p.z) / 2.0;
            double sidelength = mexels[p1].p.dist(mexels[p2].p);
            double h = p1p2.dist(mexels[p4].p);
            //pentagon height compared to size
            double h_ratio = Math.Sqrt(25.0 + 10.0 * Math.Sqrt(5.0)) / 10.0;
            centroid.x = p1p2.x + ((mexels[p4].p.x - p1p2.x) / h) * sidelength * h_ratio;
            centroid.y = p1p2.y + ((mexels[p4].p.y - p1p2.y) / h) * sidelength * h_ratio;
            centroid.z = p1p2.z + ((mexels[p4].p.z - p1p2.z) / h) * sidelength * h_ratio;
            //add centroid into mesh
            AddMexel(new Mexel(centroid));

            //add connections to and from centroid
            mexels[SizeMexels - 1].Ng[0] = p1;
            mexels[SizeMexels - 1].Ng[1] = p2;
            mexels[SizeMexels - 1].Ng[2] = p3;
            mexels[SizeMexels - 1].Ng[3] = p4;
            mexels[SizeMexels - 1].Ng[4] = p5;
            i = 0;
            while (mexels[p1].Ng[i] != -1) { i++; }
            mexels[p1].Ng[i] = SizeMexels - 1;
            i = 0;
            while (mexels[p2].Ng[i] != -1) { i++; }
            mexels[p2].Ng[i] = SizeMexels - 1;
            i = 0;
            while (mexels[p3].Ng[i] != -1) { i++; }
            mexels[p3].Ng[i] = SizeMexels - 1;
            i = 0;
            while (mexels[p4].Ng[i] != -1) { i++; }
            mexels[p4].Ng[i] = SizeMexels - 1;
            i = 0;
            while (mexels[p5].Ng[i] != -1) { i++; }
            mexels[p5].Ng[i] = SizeMexels - 1;

            //add adjanced faces for centroid (faces out)
            AddFace(new TriFace(p2, p1, SizeMexels - 1));
            AddFace(new TriFace(p3, p2, SizeMexels - 1));
            AddFace(new TriFace(p4, p3, SizeMexels - 1));
            AddFace(new TriFace(p5, p4, SizeMexels - 1));
            AddFace(new TriFace(p1, p5, SizeMexels - 1));
            mexels[SizeMexels - 1].triface[0] = SizeFaces - 5;
            mexels[SizeMexels - 1].triface[1] = SizeFaces - 4;
            mexels[SizeMexels - 1].triface[2] = SizeFaces - 3;
            mexels[SizeMexels - 1].triface[3] = SizeFaces - 2;
            mexels[SizeMexels - 1].triface[4] = SizeFaces - 1;

            //add two adjanced faces for each pentagon vertice
            i = 0;
            while (mexels[p1].triface[i] != -1) { i++; }
            mexels[p1].triface[i] = SizeFaces - 1;
            mexels[p1].triface[i + 1] = SizeFaces - 5;
            i = 0;
            while (mexels[p2].triface[i] != -1) { i++; }
            mexels[p2].triface[i] = SizeFaces - 5;
            mexels[p2].triface[i + 1] = SizeFaces - 4;
            i = 0;
            while (mexels[p3].triface[i] != -1) { i++; }
            mexels[p3].triface[i] = SizeFaces - 4;
            mexels[p3].triface[i + 1] = SizeFaces - 3;
            i = 0;
            while (mexels[p4].triface[i] != -1) { i++; }
            mexels[p4].triface[i] = SizeFaces - 3;
            mexels[p4].triface[i + 1] = SizeFaces - 2;
            i = 0;
            while (mexels[p5].triface[i] != -1) { i++; }
            mexels[p5].triface[i] = SizeFaces - 2;
            mexels[p5].triface[i + 1] = SizeFaces - 1;

            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[p1].triface[j] == -1) break;
                i = 0;
                for (k = 0; k < 3; k++) if (faces[mexels[p1].triface[j]].mexel[k] == p1) i++;
            }
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[p2].triface[j] == -1) break;
                i = 0;
                for (k = 0; k < 3; k++) if (faces[mexels[p2].triface[j]].mexel[k] == p2) i++;
            }
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[p3].triface[j] == -1) break;
                i = 0;
                for (k = 0; k < 3; k++) if (faces[mexels[p3].triface[j]].mexel[k] == p3) i++;
            }
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[p4].triface[j] == -1) break;
                i = 0;
                for (k = 0; k < 3; k++) if (faces[mexels[p4].triface[j]].mexel[k] == p4) i++;
            }
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[p5].triface[j] == -1) break;
                i = 0;
                for (k = 0; k < 3; k++) if (faces[mexels[p5].triface[j]].mexel[k] == p5) i++;
            }

            i = SizeMexels - 1;
            //go trough all neighbours of the mexel
            for (j = 0; j < Mexel_N; j++)
            {
                if (mexels[i].Ng[j] == -1) break;
                neighbour_match = 0;
                //go through all trifaces' mexels
                for (k = 0; k < Mexel_N; k++)
                {
                    if (mexels[i].triface[k] == -1) break;
                    for (l = 0; l < 3; l++)
                    {
                        if (faces[mexels[i].triface[k]].mexel[l] == mexels[i].Ng[j])
                        {
                            neighbour_match++;
                        }
                    }
                }
                //one neighbour point must be found exactly two times from all
                //connected trifaces
                if (neighbour_match != 2)
                {
                    Console.WriteLine(" mexel " + i + ":");
                    Console.Write("neighbours:");
                    for (k = 0; k < Mexel_N; k++)
                    {
                        if (mexels[i].Ng[k] == -1) break;
                        //go around the mexels of the face
                        Console.Write("       " + mexels[i].Ng[k]);
                    }
                    Console.WriteLine();
                    for (k = 0; k < Mexel_N; k++)
                    {
                        if (mexels[i].triface[k] == -1) break;
                        //go around the mexels of the face
                        Console.Write("triface " + mexels[i].triface[k] + ":");
                        for (l = 0; l < 3; l++)
                        {
                            //if one mexel is the other neighbour point (at other end of split vertex)
                            Console.Write("       " + faces[mexels[i].triface[k]].mexel[l]);
                        }
                        Console.WriteLine();
                    }
                    Console.WriteLine("neighbour match:" + neighbour_match);
                    throw new TriMeshException("Failed to add pentagon centroid");
                }
            }
        }
        /// <summary>
        /// Calculates face area in coordinate units.
        /// </summary>
        /// <param name="i">face index [0..(no faces-1)]</param>
        /// <param name="a">temporary edge length (output)</param>
        /// <param name="b">temporary edge length (output)</param>
        /// <param name="c">temporary edge length (output)</param>
        /// <returns>face are in coordinate units</returns>
        public double FaceArea(int i, ref double a, ref double b, ref double c)
        {
            double p = 0.0;
            if (a == -1) a = mexels[faces[i].mexel[0]].p.dist(mexels[faces[i].mexel[1]].p);
            if (b == -1) b = mexels[faces[i].mexel[1]].p.dist(mexels[faces[i].mexel[2]].p);
            if (c == -1) c = mexels[faces[i].mexel[2]].p.dist(mexels[faces[i].mexel[0]].p);
            //swap so that formula becomes stable for 'needle'-like triangles (a >= b >= c)
            if (c > b)
            {
                p = c;
                c = b;
                b = p;
            }
            if (b > a)
            {
                p = a;
                a = b;
                b = p;
                if (c > a)
                {
                    p = a;
                    a = c;
                    c = p;
                }
            }
            return 0.25 * Math.Sqrt(((a) + ((b) + (c))) * ((c) - ((a) - (b))) * ((c) + ((a) - (b))) * ((a) + ((b) - (c))));
        }
        /// <summary>
        /// Moves all mexels inside current bounds
        /// </summary>
        /// <param name="locking">true for locking out-of-bounds objects from further location changes</param>
        public void ApplyBounds(bool locking)
        {
            int i = 0;
            int j = 0;
            for (i = 0; i < SizeMexels; i++)
            {
                if (GetMexelFlag(i, MESH_FEATURE_ITEM)) continue;
                if (mexels[i].p.x < bounds.GetLimit(0, Limits.Limit.LOW))
                {
                    mexels[i].p.x = bounds.GetLimit(0, Limits.Limit.LOW);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                else if (mexels[i].p.x > bounds.GetLimit(0, Limits.Limit.HIGH))
                {
                    mexels[i].p.x = bounds.GetLimit(0, Limits.Limit.HIGH);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                if (mexels[i].p.y < bounds.GetLimit(1, Limits.Limit.LOW))
                {
                    mexels[i].p.y = bounds.GetLimit(1, Limits.Limit.LOW);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                else if (mexels[i].p.y > bounds.GetLimit(1, Limits.Limit.HIGH))
                {
                    mexels[i].p.y = bounds.GetLimit(1, Limits.Limit.HIGH);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                if (mexels[i].p.z < bounds.GetLimit(2, Limits.Limit.LOW))
                {
                    mexels[i].p.z = bounds.GetLimit(2, Limits.Limit.LOW);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                else if (mexels[i].p.z > bounds.GetLimit(2, Limits.Limit.HIGH))
                {
                    mexels[i].p.z = bounds.GetLimit(2, Limits.Limit.HIGH);
                    if (locking)
                    {
                        SetMexelFlag(i, MESH_FEATURE_ITEM);
                        SetMexelFlag(i, MESH_STATIC_ITEM);
                    }
                }
                //remove single non-features
                if (GetMexelFlag(i, MESH_FEATURE_ITEM)) continue;
                if (locking)
                {
                    SetMexelFlag(i, MESH_FEATURE_ITEM);
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[i].Ng[j] == -1) break;
                        if (!GetMexelFlag(mexels[i].Ng[j], MESH_FEATURE_ITEM))
                        {
                            ResetMexelFlag(i, MESH_FEATURE_ITEM);
                            break;
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Spherizes mesh with radius.
        /// </summary>
        /// <param name="d">radius of the mesh in units</param>
        public void Spherize(double d)
        {
            int i = 0;
            double dist = 0.0f;
            Point n = new Point();
            Point z = new Point();

            //calculate mean distance to center
            if (d == -1)
            {
                d = 0.0;
                for (i = 0; i < SizeMexels; i++)
                {
                    d += z.dist(mexels[i].p) / SizeMexels;
                }
            }
            // set all vertices distance to the mean value
            for (i = 0; i < SizeMexels; i++)
            {
                //calculate normal
                dist = z.dist(mexels[i].p);
                n.x = mexels[i].p.x / dist;
                n.y = mexels[i].p.y / dist;
                n.z = mexels[i].p.z / dist;
                //move vertice so much that it is at mean
                mexels[i].p.x = n.x * d;
                mexels[i].p.y = n.y * d;
                mexels[i].p.z = n.z * d;
            }
            d = 0.0;
            n.x = 0;
            n.y = 0;
            n.z = 0;
            for (i = 0; i < SizeMexels; i++)
            {
                d += z.dist(mexels[i].p) / SizeMexels;
            }
        }
        /// <summary>
        /// Returns mesh quality term [0..100]. The term is at 100 if all triangles in the mesh 
        /// have equal edge lengths (all triangles are exactly triangular).
        /// </summary>
        /// <returns>mesh quality value</returns>
        public double Quality()
        {
            double quality = 0.0;
            double a = 0.0;
            double b = 0.0;
            double c = 0.0;
            double min_edge = 0;
            double max_edge = 0;
            int i = 0;
            if (SizeFaces == 0) throw new TriMeshException("tried to calculate mesh quality while there is no faces");
            for (i = 0; i < SizeFaces; i++)
            {
                a = mexels[faces[i].mexel[0]].p.dist(mexels[faces[i].mexel[1]].p);
                b = mexels[faces[i].mexel[1]].p.dist(mexels[faces[i].mexel[2]].p);
                c = mexels[faces[i].mexel[2]].p.dist(mexels[faces[i].mexel[0]].p);
                min_edge = (a < b ? a : b);
                min_edge = (min_edge < c ? min_edge : c);
                max_edge = (a > b ? a : b);
                max_edge = (max_edge > c ? max_edge : c);
                quality += min_edge / max_edge;
            }
            return 100.0 * quality / SizeFaces;
        }
        /// <summary>
        /// Moves all points of mesh according to 
        /// distance to center point of mesh
        /// </summary>
        /// <param name="location">new location for center of mesh</param>
        public void MoveTo(TPClib.Point location)
        {
            Point diff = location - center;
            for (int i = 0; i < mexels.Count; i++)
            {
                mexels[i].p += diff;
            }
        }
        /// <summary>
        /// Divides every face into three
        /// </summary>
        protected void Divide()
        {
            int i = 0;
            int j = 0;
            int k = 0;
            int l = 0;
            int n = 0;
            Point p1 = new Point();
            Point p2 = new Point();
            Point p3 = new Point();
            int p1_i;
            int p2_i;
            int p3_i;
            //true if points are new
            bool p1_new = false;
            bool p2_new = false;
            bool p3_new = false;
            int first_new_mexel_i = SizeMexels;
            int faces_size_old = SizeFaces;
            Mexel m = new Mexel();
            TriFace f = new TriFace();

            //divide all faces of mesh
            //	assert(dm_verify_mesh_face_back_reference(mesh) == 0);
            //	assert(dm_verify_mesh_back_reference(mesh) == 0);
            //	assert(dm_verify_mesh_self_reference(mesh) == 0);
            //	assert(dm_verify_mesh_face_mexels_and_neighbours(mesh) == 0);
            //	assert(dm_verify_mesh_arrays(mesh) == 0);
            //	assert(dm_verify_mesh_array_duplicates(mesh) == 0);
            for (i = 0; i < SizeMexels; i++)
            {
                for (j = 0; j < SizeMexels; j++)
                {
                    if (i == j) continue;
                }
            }
            for (i = 0; i < faces_size_old; i++)
            {

                p1_new = false;
                p2_new = false;
                p3_new = false;
                //resolve coordinates of middle triangle
                p1.x = (mexels[faces[i].mexel[1]].p.x + mexels[faces[i].mexel[0]].p.x) / 2.0;
                p1.y = (mexels[faces[i].mexel[1]].p.y + mexels[faces[i].mexel[0]].p.y) / 2.0;
                p1.z = (mexels[faces[i].mexel[1]].p.z + mexels[faces[i].mexel[0]].p.z) / 2.0;
                p2.x = (mexels[faces[i].mexel[2]].p.x + mexels[faces[i].mexel[1]].p.x) / 2.0;
                p2.y = (mexels[faces[i].mexel[2]].p.y + mexels[faces[i].mexel[1]].p.y) / 2.0;
                p2.z = (mexels[faces[i].mexel[2]].p.z + mexels[faces[i].mexel[1]].p.z) / 2.0;
                p3.x = (mexels[faces[i].mexel[0]].p.x + mexels[faces[i].mexel[2]].p.x) / 2.0;
                p3.y = (mexels[faces[i].mexel[0]].p.y + mexels[faces[i].mexel[2]].p.y) / 2.0;
                p3.z = (mexels[faces[i].mexel[0]].p.z + mexels[faces[i].mexel[2]].p.z) / 2.0;

                //find existing mexel indexes or add new ones if necessary
                p1_i = -1;
                p2_i = -1;
                p3_i = -1;
                for (j = first_new_mexel_i; j < SizeMexels; j++)
                {
                    if (mexels[j].p.dist(p1) < 0.0001) p1_i = j;
                    if (mexels[j].p.dist(p2) < 0.0001) p2_i = j;
                    if (mexels[j].p.dist(p3) < 0.0001) p3_i = j;
                }

                if (p1_i == -1)
                {
                    l = SizeMexels;
                    AddMexel(new Mexel(p1));
                    p1_i = SizeMexels - 1;
                    p1_new = true;
                }
                if (p2_i == -1)
                {
                    l = SizeMexels;
                    AddMexel(new Mexel(p2));
                    p2_i = SizeMexels - 1;
                    p2_new = true;
                }
                if (p3_i == -1)
                {
                    l = SizeMexels;
                    AddMexel(new Mexel(p3));
                    p3_i = SizeMexels - 1;
                    p3_new = true;
                }

                //add neighbour connections for new mexels
                for (j = 0; j < Mexel_N; j++) if (mexels[p1_i].Ng[j] == -1) break;
                mexels[p1_i].Ng[j++] = p2_i;
                mexels[p1_i].Ng[j++] = p3_i;
                for (j = 0; j < Mexel_N; j++) if (mexels[p2_i].Ng[j] == -1) break;
                mexels[p2_i].Ng[j++] = p3_i;
                mexels[p2_i].Ng[j++] = p1_i;
                for (j = 0; j < Mexel_N; j++) if (mexels[p3_i].Ng[j] == -1) break;
                mexels[p3_i].Ng[j++] = p1_i;
                mexels[p3_i].Ng[j++] = p2_i;

                for (j = 0; j < Mexel_N; j++) if (mexels[p1_i].Ng[j] == -1) break;
                if (!p1_new)
                {
                    mexels[p1_i].Ng[j++] = faces[i].mexel[0];
                    mexels[p1_i].Ng[j++] = faces[i].mexel[1];
                    //cut connection at p1
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[0]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[0]].Ng[j] == faces[i].mexel[1]) mexels[faces[i].mexel[0]].Ng[j] = p1_i;
                    }
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[1]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[1]].Ng[j] == faces[i].mexel[0]) mexels[faces[i].mexel[1]].Ng[j] = p1_i;
                    }
                }
                for (j = 0; j < Mexel_N; j++) if (mexels[p2_i].Ng[j] == -1) break;
                if (!p2_new)
                {
                    //cut connection at p2
                    mexels[p2_i].Ng[j++] = faces[i].mexel[1];
                    mexels[p2_i].Ng[j++] = faces[i].mexel[2];
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[1]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[1]].Ng[j] == faces[i].mexel[2]) mexels[faces[i].mexel[1]].Ng[j] = p2_i;
                    }
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[2]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[2]].Ng[j] == faces[i].mexel[1]) mexels[faces[i].mexel[2]].Ng[j] = p2_i;
                    }
                }

                for (j = 0; j < Mexel_N; j++) if (mexels[p3_i].Ng[j] == -1) break;
                if (!p3_new)
                {
                    mexels[p3_i].Ng[j++] = faces[i].mexel[0];
                    mexels[p3_i].Ng[j++] = faces[i].mexel[2];
                    //cut connection at p3
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[0]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[0]].Ng[j] == faces[i].mexel[2]) mexels[faces[i].mexel[0]].Ng[j] = p3_i;
                    }
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[faces[i].mexel[2]].Ng[j] == -1) break;
                        if (mexels[faces[i].mexel[2]].Ng[j] == faces[i].mexel[0]) mexels[faces[i].mexel[2]].Ng[j] = p3_i;
                    }
                }
                for (n = 0; n < SizeMexels; n++)
                {
                    for (j = 0; j < Mexel_N; j++)
                    {
                        if (mexels[n].triface[j] == -1) break;
                        l = 0;
                        for (k = 0; k < 3; k++)
                        {
                            if (faces[mexels[n].triface[j]].mexel[k] == n) l++;
                        }
                    }
                }

                /**********************
                 * FACE MODIFICATIONS *
                 **********************/

                //add 3 new faces (same order as with original triangle)
                AddFace(new TriFace(p3_i, faces[i].mexel[0], p1_i));
                AddFace(new TriFace(p1_i, faces[i].mexel[1], p2_i));
                AddFace(new TriFace(p2_i, faces[i].mexel[2], p3_i));

                //update adjanced face for each corner
                for (j = 0; j < Mexel_N; j++)
                {
                    if (mexels[faces[i].mexel[0]].triface[j] == i)
                    {
                        mexels[faces[i].mexel[0]].triface[j] = SizeFaces - 3;
                        break;
                    }
                }
                for (j = 0; j < Mexel_N; j++)
                {
                    if (mexels[faces[i].mexel[1]].triface[j] == i)
                    {
                        mexels[faces[i].mexel[1]].triface[j] = SizeFaces - 2;
                        break;
                    }
                }
                for (j = 0; j < Mexel_N; j++)
                {
                    if (mexels[faces[i].mexel[2]].triface[j] == i)
                    {
                        mexels[faces[i].mexel[2]].triface[j] = SizeFaces - 1;
                        break;
                    }
                }

                //modify old face (into new 'middle' face)
                faces[i].mexel[0] = p1_i;
                faces[i].mexel[1] = p2_i;
                faces[i].mexel[2] = p3_i;

                //update adjanced faces for new mexels
                j = 0; while (mexels[p1_i].triface[j] != -1) j++;
                mexels[p1_i].triface[j++] = SizeFaces - 3;
                mexels[p1_i].triface[j++] = i;
                mexels[p1_i].triface[j++] = SizeFaces - 2;
                j = 0; while (mexels[p2_i].triface[j] != -1) j++;
                mexels[p2_i].triface[j++] = SizeFaces - 2;
                mexels[p2_i].triface[j++] = i;
                mexels[p2_i].triface[j++] = SizeFaces - 1;
                j = 0; while (mexels[p3_i].triface[j] != -1) j++;
                mexels[p3_i].triface[j++] = SizeFaces - 1;
                mexels[p3_i].triface[j++] = i;
                mexels[p3_i].triface[j++] = SizeFaces - 3;
            }
        }
        /// <summary>
        /// Resolves neighbouring faces' common mexels. 
        /// </summary>
        /// <param name="f1">face 1</param>
        /// <param name="f2">face 2</param>
        /// <param name="m1">common neighbour index 1</param>
        /// <param name="m2">common neighbour index 2</param>
        /// <returns>true if faces are neighbours</returns>
        public static bool GetFacesNeighbours(TriFace f1, TriFace f2, out int m1, out int m2)
        {
            m1 = -1;
            m2 = -1;
            for (int i = 0; i < 3; i++)
            {
                if (i == 2 && m1 == -1) return false;
                for (int j = 0; j < 3; j++)
                {
                    if (f1.mexel[i] == f2.mexel[j])
                    {
                        if (m1 == -1)
                            m1 = f1.mexel[i];
                        else if (m2 == -1)
                            m2 = f1.mexel[i];
                        else
                            throw new TriMeshException("More than two common vertexes");
                    }
                }
            }
            if (m2 == -1) return false;
            return m1 != -1;
        }
        /// <summary>
        /// Resolves neighbouring faces' common mexels. 
        /// </summary>
        /// <param name="f1">face 1</param>
        /// <param name="f2">face 2</param>
        /// <param name="m1">common neighbour index 1</param>
        /// <param name="m2">common neighbour index 2</param>
        /// <returns>true if faces are neighbours</returns>
        public bool GetFacesNeighbours(int f1, int f2, out int m1, out int m2)
        {
            return GetFacesNeighbours(faces[f1], faces[f2], out m1, out m2);
        }
        /// <summary>
        /// Resolves whether two faces are neighbouring each other in the mesh so 
        /// that they have a common side. 
        /// </summary>
        /// <param name="f1">face 1</param>
        /// <param name="f2">face 2</param>
        /// <returns>true if faces are neighbours</returns>
        public bool AreFacesNeighbours(int f1, int f2)
        {
            int p = 0;
            for (int i = 0; i < 3; i++)
            {
                for (int j = 0; j < 3; j++)
                {
                    if (faces[f1].mexel[i] == faces[f2].mexel[j]) p++;
                }
            }
            if (p == 3) throw new InvalidProgramException("Two identical faces");
            return (p == 2);
        }
        /// <summary>
        /// Resolves whether two faces are neighbouring each other in the mesh so 
        /// that they have a common single vertex. 
        /// </summary>
        /// <param name="f1">face 1</param>
        /// <param name="f2">face 2</param>
        /// <returns>true if faces are neighbours</returns>
        public bool AreFacesNeighboursWithVertex(int f1, int f2)
        {
            int p = 0;
            for (int i = 0; i < 3; i++)
            {
                for (int j = 0; j < 3; j++)
                {
                    if (faces[f1].mexel[i] == faces[f2].mexel[j]) p++;
                }
            }
            if (p == 3) throw new InvalidProgramException("Two identical faces");
            return (p == 1);
        }
        /// <summary>
        /// Updates face neighbour index values.
        /// </summary>
        public void UpdateFaceNgs()
        {
            int i = 0;
            int j = 0;
            int k = 0;
            for (i = 0; i < faces.Count; i++)
            {
                faces[i].ClearNg();
            }
            for (i = 0; i < mexels.Count; i++)
            {
                for (j = 0; j < TriMeshVOI.Mexel_N; j++)
                {
                    if (mexels[i].triface[j] == -1) break;
                    if (GetFaceFlag(mexels[i].triface[j], TriMeshVOI.MESH_DELETED_ITEM)) continue;
                    for (k = 0; k < TriMeshVOI.Mexel_N; k++)
                    {
                        if (mexels[i].triface[k] == -1) break;
                        if (GetFaceFlag(mexels[i].triface[k], TriMeshVOI.MESH_DELETED_ITEM)) continue;
                        if (j == k) continue;
                        //add connections if faces are neighbours
                        if (AreFacesNeighbours(mexels[i].triface[j], mexels[i].triface[k]))
                        {
                            if (!faces[mexels[i].triface[j]].HasNg(mexels[i].triface[k]))
                                faces[mexels[i].triface[j]].AddNg(mexels[i].triface[k]);
                            if (!faces[mexels[i].triface[k]].HasNg(mexels[i].triface[j]))
                                faces[mexels[i].triface[k]].AddNg(mexels[i].triface[j]);
                            int common = 0;
                            if (faces[mexels[i].triface[j]].HasMexel(faces[mexels[i].triface[k]].mexel[0])) common++;
                            if (faces[mexels[i].triface[j]].HasMexel(faces[mexels[i].triface[k]].mexel[1])) common++;
                            if (faces[mexels[i].triface[j]].HasMexel(faces[mexels[i].triface[k]].mexel[2])) common++;
                            if (common != 2) throw new TriMeshException("Invalid neighbouring faces.");
                        }
                    }
                }
            }
            for (i = 0; i < faces.Count; i++)
            {
                if (faces[faces[i].Ng[0]].Ng[0] != i &&
                   faces[faces[i].Ng[0]].Ng[1] != i &&
                   faces[faces[i].Ng[0]].Ng[2] != i)
                {
                    throw new TriMeshException("Invalid face neighbours.");
                }
                if (faces[faces[i].Ng[1]].Ng[0] != i &&
                    faces[faces[i].Ng[1]].Ng[1] != i &&
                    faces[faces[i].Ng[1]].Ng[2] != i)
                {
                    throw new TriMeshException("Invalid face neighbours.");
                }
                if (faces[faces[i].Ng[2]].Ng[0] != i &&
                    faces[faces[i].Ng[2]].Ng[1] != i &&
                    faces[faces[i].Ng[2]].Ng[2] != i)
                {
                    throw new TriMeshException("Invalid face neighbours.");
                }
            }
        }
        /// <summary>
        /// Prints i'th face in Matlab form
        /// </summary>
        /// <param name="i">face index [0..no faces-1]</param>
        /// <param name="str">display string</param>
        /// <param name="color">display color</param>
        public void PrintFace_Matlab(int i, string str, string color)
        {
            PrintFace_Matlab(Console.Out, i, str, color);
        }
        /// <summary>
        /// Prints i'th face in Matlab form
        /// </summary>
        /// <param name="file">file for writing</param>
        /// <param name="i">face index [0..no faces-1]</param>
        /// <param name="str">display string</param>
        /// <param name="color">display color</param>
        public void PrintFace_Matlab(System.IO.TextWriter file, int i, string str, string color)
        {
            file.WriteLine((str + " = [" + mexels[faces[i].mexel[0]].p.x +
                                  " " + mexels[faces[i].mexel[0]].p.y +
                                  " " + mexels[faces[i].mexel[0]].p.z + ";" +
                                        mexels[faces[i].mexel[1]].p.x +
                                  " " + mexels[faces[i].mexel[1]].p.y +
                                  " " + mexels[faces[i].mexel[1]].p.z + ";" +
                                        mexels[faces[i].mexel[2]].p.x +
                                  " " + mexels[faces[i].mexel[2]].p.y +
                                  " " + mexels[faces[i].mexel[2]].p.z + "];").Replace(',', '.'));
            file.WriteLine(str + "(4,:) = " + str + "(1,:);");
            file.WriteLine("fill3(" + str + "(:,1)," + str + "(:,2)," + str + "(:,3)," + color + ");");
            file.WriteLine("hold on;");
            file.WriteLine("text(" + (mexels[faces[i].mexel[0]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.z.ToString()).Replace(',', '.') + ",\'0:" + faces[i].mexel[0] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[1]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.z.ToString()).Replace(',', '.') + ",\'1:" + faces[i].mexel[1] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[2]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.z.ToString()).Replace(',', '.') + ",\'2:" + faces[i].mexel[2] + "\');");
        }
        /// <summary>
        /// Function for printing face so that it can be displayed in Matlab.
        /// </summary>
        /// <param name="file">file for writing</param>
        /// <param name="i">face index [0..no faces-1]</param>
        /// <param name="str">display string</param>
        /// <param name="color">display color</param>
        public void PrintFace_Matlab2(System.IO.TextWriter file, int i, string str, string color)
        {
            file.WriteLine((str + " = [" + mexels[faces[i].mexel[0]].p.x +
                                  " " + mexels[faces[i].mexel[0]].p.y +
                                  " " + mexels[faces[i].mexel[0]].p.z + ";" +
                                        mexels[faces[i].mexel[1]].p.x +
                                  " " + mexels[faces[i].mexel[1]].p.y +
                                  " " + mexels[faces[i].mexel[1]].p.z + ";" +
                                        mexels[faces[i].mexel[2]].p.x +
                                  " " + mexels[faces[i].mexel[2]].p.y +
                                  " " + mexels[faces[i].mexel[2]].p.z + "];").Replace(',', '.'));
            file.WriteLine(str + "(4,:) = " + str + "(1,:);");
            file.WriteLine("fill3(" + str + "(:,1)," + str + "(:,2)," + str + "(:,3)," + color + ");");
            file.WriteLine("hold on;");
            Point center = GetFaceBaryCenter(i);
            file.WriteLine("text(" + (mexels[faces[i].mexel[0]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.z.ToString()).Replace(',', '.') + ",\'0:" + faces[i].mexel[0] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[1]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.z.ToString()).Replace(',', '.') + ",\'1:" + faces[i].mexel[1] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[2]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.z.ToString()).Replace(',', '.') + ",\'2:" + faces[i].mexel[2] + "\');");
            file.WriteLine("text(" + (center.x.ToString()).Replace(',', '.') + "," +
                                  " " + (center.y.ToString()).Replace(',', '.') + "," +
                                  " " + (center.z.ToString()).Replace(',', '.') + ",\'" + i + "\');");
        }
        /// <summary>
        /// Function for printing face so that it can be displayed in Matlab.
        /// </summary>
        /// <param name="file">file for writing</param>
        /// <param name="i">face index [0..no faces-1]</param>
        /// <param name="str">display string</param>
        /// <param name="color">display color</param>
        /// <param name="normal">face normal direction as point vector</param>
        public void PrintFace_Matlab2(System.IO.TextWriter file, int i, string str, string color, Point normal)
        {
            file.WriteLine((str + " = [" + mexels[faces[i].mexel[0]].p.x +
                                  " " + mexels[faces[i].mexel[0]].p.y +
                                  " " + mexels[faces[i].mexel[0]].p.z + ";" +
                                        mexels[faces[i].mexel[1]].p.x +
                                  " " + mexels[faces[i].mexel[1]].p.y +
                                  " " + mexels[faces[i].mexel[1]].p.z + ";" +
                                        mexels[faces[i].mexel[2]].p.x +
                                  " " + mexels[faces[i].mexel[2]].p.y +
                                  " " + mexels[faces[i].mexel[2]].p.z + "];").Replace(',', '.'));
            file.WriteLine(str + "(4,:) = " + str + "(1,:);");
            file.WriteLine("fill3(" + str + "(:,1)," + str + "(:,2)," + str + "(:,3)," + color + ");");
            file.WriteLine("hold on;");
            Point center = GetFaceBaryCenter(i);
            normal.Normalize();
            Point n = center + normal;
            file.WriteLine("line([" + (center.x.ToString()).Replace(',', '.') +
                                  " " + (n.x.ToString()).Replace(',', '.') + "], " +
                                  " [" + (center.y.ToString()).Replace(',', '.') +
                                  " " + (n.y.ToString()).Replace(',', '.') + "], " +
                                  " [" + (center.z.ToString()).Replace(',', '.') +
                                  " " + (n.z.ToString()).Replace(',', '.') + "]);");
            file.WriteLine("text(" + (mexels[faces[i].mexel[0]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[0]].p.z.ToString()).Replace(',', '.') + ",\'0:" + faces[i].mexel[0] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[1]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[1]].p.z.ToString()).Replace(',', '.') + ",\'1:" + faces[i].mexel[1] + "\');");
            file.WriteLine("text(" + (mexels[faces[i].mexel[2]].p.x.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.y.ToString()).Replace(',', '.') + "," +
                                  " " + (mexels[faces[i].mexel[2]].p.z.ToString()).Replace(',', '.') + ",\'2:" + faces[i].mexel[2] + "\');");
            file.WriteLine("text(" + (center.x.ToString()).Replace(',', '.') + "," +
                                  " " + (center.y.ToString()).Replace(',', '.') + "," +
                                  " " + (center.z.ToString()).Replace(',', '.') + ",\'" + i + "\');");
        }
        /// <summary>
        /// Prints this mesh in matlab .m file format so that it can be displayed in Matlab.
        /// </summary>
        public void Print_Matlab()
        {
            System.IO.StreamWriter file = new System.IO.StreamWriter(@"s:\temp\harri\test.m", false);
            for (int i = 0; i < faces.Count; i++)
            {
                if (GetFaceFlag(i, TriMeshVOI.MESH_TEMP_ITEM)) PrintFace_Matlab(file, i, "face_" + i, "'r'");
                else PrintFace_Matlab(file, i, "face_" + i, "'b'");
                file.Flush();
            }
            file.Close();
        }
        /// <summary>
        /// Prints this mesh in matlab .m file format so that it can be displayed in Matlab.
        /// Only part of the mesh is printed starting from i'th face.
        /// </summary>
        /// <param name="i">face that is printed</param>
        /// <param name="depth">depth that is printed</param>
        public void Print_Matlab(int i, int depth)
        {
            List<int> included_faces = new List<int>();
            int d = 0;
            included_faces.Add(i);
            int start = 0;
            int end = included_faces.Count;
            int f = 0;
            double r = 0;
            double g = 0;
            double b = 0;
            System.IO.StreamWriter file = new System.IO.StreamWriter(@"s:\temp\harri\test.m", false);

            r = 0.25;
            g = 0.25;
            b = 0.25;
            if (GetFaceFlag(i, TriMeshVOI.MESH_INTERSECTING_ITEM)) r = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_VALID_ITEM)) g = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_UNVISITED_ITEM)) b = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_PARTIALLY_VALID_ITEM)) { g /= 2.0; }
            PrintFace_Matlab2(file, i, "face_" + i, "[" +
                                            r.ToString().Replace(',', '.') + " " +
                                            g.ToString().Replace(',', '.') + " " +
                                            b.ToString().Replace(',', '.') + "]");

            while (d++ < depth)
            {
                //go through next depth
                for (int k = start; k < end; k++)
                {
                    f = included_faces[k];
                    for (int j = 0; j < 3; j++)
                    {
                        if (!included_faces.Contains(faces[f].Ng[j]))
                        {
                            if (faces[f].Ng[j] == -1) continue;
                            included_faces.Add(faces[f].Ng[j]);
                            r = 0.25;
                            g = 0.25;
                            b = 0.25;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_INTERSECTING_ITEM)) r = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_VALID_ITEM)) g = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_UNVISITED_ITEM)) b = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_PARTIALLY_VALID_ITEM)) { g /= 2.0; }
                            PrintFace_Matlab2(file, faces[f].Ng[j], "face_" + faces[f].Ng[j], "[" +
                                                            r.ToString().Replace(',', '.') + " " +
                                                            g.ToString().Replace(',', '.') + " " +
                                                            b.ToString().Replace(',', '.') + "]");
                            file.Flush();
                        }
                    }
                }
                start = end;
                end = included_faces.Count;
            }
            file.Close();
        }

        /// <summary>
        /// Masks image with mesh.
        /// 
        /// Dunlavey MR, Efficient Polygon-Filling Algorithms for Raster Displays,
        /// ACM Transactions on Graphics, Vol. 2, No. 4, October 1983, Pages 264-273.
        /// </summary>
        /// <param name="mask">mask that will have 1 at border and 0 elsewhere</param>
        /// <param name="siz">voxel size</param>
        public void Mask(ref TPClib.Image.Image mask, Voxel siz)
        {
            int i = 0;
            int j = 0;

            //initialize output image as zero
            mask.Multiply(0.0f);

            //mask all faces
            Console.WriteLine("masking mesh " + SizeFaces + " " + SizeMexels);

            //leaves = tree.root.GetLeaves_Recursive();                   
            List<Point> line = new List<Point>();
            List<Point> line1 = new List<Point>();
            List<Point> line2 = new List<Point>();
            List<Point> line3 = new List<Point>();
            for (i = 0; i < faces.Count; i++)
            {
                //draw lines
                Point p1_real = mexels[faces[i].mexel[0]].p / siz;
                Point p2_real = mexels[faces[i].mexel[1]].p / siz;
                Point p3_real = mexels[faces[i].mexel[2]].p / siz;
                if (p1_real.x < 0) p1_real.x = 0;
                else if (Math.Ceiling(p1_real.x) >= mask.dimx) p1_real.x = mask.dimx - 1;
                if (p1_real.y < 0) p1_real.y = 0;
                else if (Math.Ceiling(p1_real.y) >= mask.dimy) p1_real.y = mask.dimy - 1;
                if (p1_real.z < 0) p1_real.z = 0;
                else if (Math.Ceiling(p1_real.z) >= mask.dimz) p1_real.z = mask.dimz - 1;

                if (p2_real.x < 0) p2_real.x = 0;
                else if (Math.Ceiling(p2_real.x) >= mask.dimx) p2_real.x = mask.dimx - 1;
                if (p2_real.y < 0) p2_real.y = 0;
                else if (Math.Ceiling(p2_real.y) >= mask.dimy) p2_real.y = mask.dimy - 1;
                if (p2_real.z < 0) p2_real.z = 0;
                else if (Math.Ceiling(p2_real.z) >= mask.dimz) p2_real.z = mask.dimz - 1;

                if (p3_real.x < 0) p3_real.x = 0;
                else if (Math.Ceiling(p3_real.x) >= mask.dimx) p3_real.x = mask.dimx - 1;
                if (p3_real.y < 0) p3_real.y = 0;
                else if (Math.Ceiling(p3_real.y) >= mask.dimy) p3_real.y = mask.dimy - 1;
                if (p3_real.z < 0) p3_real.z = 0;
                else if (Math.Ceiling(p3_real.z) >= mask.dimz) p3_real.z = mask.dimz - 1;

                line1 = TPClib.Image.ImageUtils.maskBresenhamLine(ref mask, p1_real, p2_real);
                line2 = TPClib.Image.ImageUtils.maskBresenhamLine(ref mask, p2_real, p3_real);
                line3 = TPClib.Image.ImageUtils.maskBresenhamLine(ref mask, p3_real, p1_real);

                //draw lines between points of lines
                for (j = 0; j < (int)line1.Count; j++)
                {
                    Point p_j = line1[j];
                    TPClib.Image.ImageUtils.maskBresenhamLine(ref mask,  p_j,  p3_real);
                }
                for (j = 0; j < (int)line2.Count; j++)
                {
                    Point p_j = line2[j];
                    TPClib.Image.ImageUtils.maskBresenhamLine(ref mask,  p_j,  p1_real);
                }
                for (j = 0; j < (int)line3.Count; j++)
                {
                    Point p_j = line3[j];
                    TPClib.Image.ImageUtils.maskBresenhamLine(ref mask,  p_j,  p2_real);
                }
            }
            return;
        }
        /// <summary>
        /// Prints face into file in Matlab processable format.
        /// </summary>
        /// <param name="filename">filename (*.m)</param>
        /// <param name="i">face index [0..no faces-1]</param>
        /// <param name="depth">radius of faces that are printed around selected one</param>
        public void Print_Matlab_WithNormals(string filename, int i, int depth)
        {
            List<int> included_faces = new List<int>();
            int d = 0;
            included_faces.Add(i);
            int start = 0;
            int end = included_faces.Count;
            int f = 0;
            double r = 0;
            double g = 0;
            double b = 0;
            System.IO.StreamWriter file = new System.IO.StreamWriter(filename, false);

            r = 0;
            g = 0;
            b = 0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_INTERSECTING_ITEM)) r = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_VALID_ITEM)) g = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_UNVISITED_ITEM)) b = 1.0;
            if (GetFaceFlag(i, TriMeshVOI.MESH_PARTIALLY_VALID_ITEM)) { r /= 2.0; g /= 2.0; b /= 2.0; }
            UpdateNormal(i);
            PrintFace_Matlab2(file, i, "face_" + i, "[" +
                                            r.ToString().Replace(',', '.') + " " +
                                            g.ToString().Replace(',', '.') + " " +
                                            b.ToString().Replace(',', '.') + "]", faces[i].n);

            while (d++ < depth)
            {
                //go through next depth
                for (int k = start; k < end; k++)
                {
                    f = included_faces[k];
                    for (int j = 0; j < 3; j++)
                    {
                        if (!included_faces.Contains(faces[f].Ng[j]))
                        {
                            included_faces.Add(faces[f].Ng[j]);
                            r = 0;
                            g = 0;
                            b = 0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_INTERSECTING_ITEM)) r = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_VALID_ITEM)) g = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_UNVISITED_ITEM)) b = 1.0;
                            if (GetFaceFlag(faces[f].Ng[j], TriMeshVOI.MESH_PARTIALLY_VALID_ITEM)) { r /= 2.0; g /= 2.0; b /= 2.0; }
                            UpdateNormal(faces[f].Ng[j]);
                            PrintFace_Matlab2(file, faces[f].Ng[j], "face_" + faces[f].Ng[j], "[" +
                                                            r.ToString().Replace(',', '.') + " " +
                                                            g.ToString().Replace(',', '.') + " " +
                                                            b.ToString().Replace(',', '.') + "]", faces[faces[f].Ng[j]].n);
                            file.Flush();
                        }
                    }
                }
                start = end;
                end = included_faces.Count;
            }
            file.Close();
        }
        /// <summary>
        /// Tests mesh consistency. Face neighbour index validity.
        /// </summary>
        public void TestConsistency_FaceNgs()
        {
            if (!TESTS_ENABLED) return;
            for (int i = 0; i < faces.Count; i++)
                for (int j = 0; j < 3; j++)
                    if (faces[i].Ng[j] < 0 || faces[i].Ng[j] >= faces.Count)
                        throw new TriMeshException("Invalid face neighbour.");
        }
        /// <summary>
        /// Tests mesh consistency. Face normals validity. 
        /// </summary>
        public void TestConsistency_FaceNormals()
        {
            if (!TESTS_ENABLED) return;
            for (int j = 0; j < faces.Count; j++)
                UpdateNormal(j);
            for (int j = 0; j < faces.Count; j++)
            {
                if (GetFaceFlag(j, TriMeshVOI.MESH_DELETED_ITEM)) continue;
                for (int k = 0; k < 3; k++)
                {
                    for (int i = 0; i < 3; i++)
                    {
                        if (faces[j].Ng[i] == -1)
                        {
                            PrintFace_Matlab(j, "Invalid", "r");
                            if (faces[j].Ng[0] != -1)
                            {
                                PrintFace_Matlab(faces[j].Ng[0], "Ng_0", "b");
                                if (faces[faces[j].Ng[0]].Ng[0] != -1 && faces[faces[j].Ng[0]].Ng[0] != j) PrintFace_Matlab(faces[faces[j].Ng[0]].Ng[0], "Ng_00", "g");
                                if (faces[faces[j].Ng[0]].Ng[1] != -1 && faces[faces[j].Ng[0]].Ng[1] != j) PrintFace_Matlab(faces[faces[j].Ng[0]].Ng[1], "Ng_01", "g");
                                if (faces[faces[j].Ng[0]].Ng[2] != -1 && faces[faces[j].Ng[0]].Ng[2] != j) PrintFace_Matlab(faces[faces[j].Ng[0]].Ng[2], "Ng_02", "g");
                            }
                            if (faces[j].Ng[1] != -1)
                            {
                                PrintFace_Matlab(faces[j].Ng[1], "Ng_1", "b");
                                if (faces[faces[j].Ng[1]].Ng[0] != -1 && faces[faces[j].Ng[1]].Ng[0] != j) PrintFace_Matlab(faces[faces[j].Ng[1]].Ng[0], "Ng_10", "g");
                                if (faces[faces[j].Ng[1]].Ng[1] != -1 && faces[faces[j].Ng[1]].Ng[1] != j) PrintFace_Matlab(faces[faces[j].Ng[1]].Ng[1], "Ng_11", "g");
                                if (faces[faces[j].Ng[1]].Ng[2] != -1 && faces[faces[j].Ng[1]].Ng[2] != j) PrintFace_Matlab(faces[faces[j].Ng[1]].Ng[2], "Ng_12", "g");
                            }
                            if (faces[j].Ng[2] != -1)
                            {
                                PrintFace_Matlab(faces[j].Ng[2], "Ng_2", "b");
                                if (faces[faces[j].Ng[2]].Ng[0] != -1 && faces[faces[j].Ng[2]].Ng[0] != j) PrintFace_Matlab(faces[faces[j].Ng[2]].Ng[0], "Ng_20", "g");
                                if (faces[faces[j].Ng[2]].Ng[1] != -1 && faces[faces[j].Ng[2]].Ng[1] != j) PrintFace_Matlab(faces[faces[j].Ng[2]].Ng[1], "Ng_21", "g");
                                if (faces[faces[j].Ng[2]].Ng[2] != -1 && faces[faces[j].Ng[2]].Ng[2] != j) PrintFace_Matlab(faces[faces[j].Ng[2]].Ng[2], "Ng_22", "g");
                            }
                            throw new TriMeshException("Invalid face index.");
                        }
                        if (GetFaceFlag(faces[j].Ng[i], TriMeshVOI.MESH_DELETED_ITEM)) continue;

                        for (int n = 0; n < 3; n++)
                        {
                            //mexel after common mexel must be different from each other
                            if (faces[faces[j].Ng[i]].mexel[n] == faces[j].mexel[k])
                            {
                                int l = (n == 2 ? 0 : n + 1);
                                int m = (k == 2 ? 0 : k + 1);
                                if (faces[j].mexel[m] == faces[faces[j].Ng[i]].mexel[l])
                                {
                                    Print_Matlab_WithNormals("TestConsistency_FaceNormals_error_report.m", j, 7);
                                    throw new TriMeshException("Invalid neighouring normals");
                                }
                                break;
                            }
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Tests mesh consistency. Face neighbouring mexel index validity.
        /// </summary>
        public void TestConsistency_FaceMexels()
        {
            if (!TESTS_ENABLED) return;
            for (int j = 0; j < faces.Count; j++)
            {
                if (GetFaceFlag(j, TriMeshVOI.MESH_DELETED_ITEM)) continue;
                for (int i = 0; i < 3; i++)
                {
                    if (faces[j].mexel[i] == -1) throw new TriMeshException("Invalid mexel index.");
                    if (faces[j].mexel[i] >= faces.Count) throw new TriMeshException("Invalid mexel index.");
                }
            }
        }
        /// <summary>
        /// Tests mesh consistency. Duplicate face corner indexes.
        /// </summary>
        public void TestConsistency_MexelFaces()
        {
            if (!TESTS_ENABLED) return;
            int i = 0;
            for (int j = 0; j < mexels.Count; j++)
            {
                if (GetFaceFlag(j, TriMeshVOI.MESH_DELETED_ITEM)) continue;
                for (i = 0; i < TriMeshVOI.Mexel_N; i++)
                {
                    if (mexels[j].triface[i] == -1) break;
                    if (mexels[j].triface[i] >= faces.Count) throw new TriMeshException("Invalid face index.");
                    if (i > 0 && mexels[j].triface[i - 1] == mexels[j].triface[i])
                        throw new TriMeshException("Duplicate face index.");
                }
                //all mexel must have at least 2 neighbouring faces
                if (i < 2) throw new TriMeshException("Invalid number of trifaces.");
            }
        }
        /// <summary>
        /// Tests mesh consistency. Two common neighbours of all neighbouring faces.
        /// </summary>
        public void TestConsistency_getTwoCommonNeighbours()
        {
            if (!TESTS_ENABLED) return;
            for (int j = 0; j < mexels.Count; j++)
            {
                if (GetMexelFlag(j, TriMeshVOI.MESH_DELETED_ITEM)) continue;
                for (int i = 0; i < TriMeshVOI.Mexel_N; i++)
                {
                    if (mexels[j].Ng[i] == -1) break;
                    if (GetMexelFlag(mexels[j].Ng[i], TriMeshVOI.MESH_DELETED_ITEM)) continue;
                    if (j == mexels[j].Ng[i]) throw new TriMeshException("Invalid neighbour index.");
                    GetTwoCommonNeighbours(j, mexels[j].Ng[i]);
                }
            }
        }
        /// <summary>
        /// Tests mesh consistency. Same corner indexes in two or more different faces.
        /// </summary>
        public void TestConsistency_DuplicateFaces()
        {
            if (!TESTS_ENABLED) return;
            for (int i = 0; i < faces.Count; i++)
            {
                for (int j = i + 1; j < faces.Count; j++)
                {
                    if (faces[i].HasMexel(faces[j].mexel[0]) &&
                        faces[i].HasMexel(faces[j].mexel[1]) &&
                        faces[i].HasMexel(faces[j].mexel[2]) &&
                        faces[j].HasMexel(faces[i].mexel[0]) &&
                        faces[j].HasMexel(faces[i].mexel[1]) &&
                        faces[j].HasMexel(faces[i].mexel[2]))
                        throw new TriMeshException("Duplicate face.");
                }
            }
        }
    }
}
