﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Volume of interest (VOI) class is a 3D volume that contains many 2D ROIs. VOI class
    /// handles the filling of VOIMatrix object, which is the matrix containing the region of interest
    /// information. It can be retrieved with GetMatrix() method.
    /// </summary>
    public class VOI : ROIObject
    {
        /// <summary>
        /// Name of VOI
        /// </summary>
        public String Name;
        /// <summary>
        /// Color of VOI
        /// </summary>        
        public int Color;
        /// <summary>
        /// Col
        /// </summary>
        public int Col;
        /// <summary>
        /// Gets the amount of ROIs in the VOI
        /// </summary>
        public int Count { get { return ROIs.Count; } }

        private List<ROI> ROIs;

        /// <summary>
        /// Constructor with zoom
        /// </summary>
        /// <param name="zoom">Zoom factor for VOI</param>
        public VOI(Double zoom)
            : base()
        {
            zoomFactor = zoom;
        }
        /// <summary>
        /// Default construtor
        /// </summary>
        public VOI()
            : base()
        {
            ROIs = new List<ROI>();
        }

        /// <summary>
        /// Claculates some function inside the ROI and returns the result for every image frame.
        /// </summary>
        /// <param name="image">Image reference</param>
        /// <param name="function">Function which is applied to pixels inside of the VOI</param>
        /// <returns>Result of function for every image frame.</returns>
        public Double[] Calculate( ref Image.Image image, ROICalculator.func function )
        {
            UpdateMatrix(image.width, image.height, image.planes);
            return (obj_matrix as VOI_Matrix).Calculate(ref image, function );
        }

        /// <summary>
        /// Returns the rois which are on given z slice.
        /// </summary>
        /// <param name="z">Z cordinate of slice</param>
        /// <returns>An array of ROIs which belong to the slice.</returns>
        public virtual ROI[] GetRois(int z)
        {
            List<ROI> return_list = new List<ROI>();

            foreach (ROI r in ROIs)
            {
                if (r.Z == z && r.Header.Direction == Slice.Direction.Transaxial ) return_list.Add(r);
            }

            return return_list.ToArray();
        }


        /// <summary>
        /// Fills the matrix with VOI.
        /// </summary>
        protected override void FillMatrix()
        {
            foreach (ROI roi in this.ROIs)
            {                
                //ROI_Matrix r = (obj_matrix as VOI_Matrix)[(int)roi.Z];
                
                if( roi.Header.Direction == Slice.Direction.Transaxial )
                {
                    if (roi.Z < 0 || roi.Z >= (obj_matrix as VOI_Matrix).Depeth) continue;
                    roi.Fill(width,height);
                    ROI_Matrix roim = roi.GetMatrix();
                    VOI_Matrix mat = (obj_matrix as VOI_Matrix);
                    mat.Union(roim, new Slice_Transaxial((int)roi.Z, mat.Width, mat.Height, mat.Depeth ) );
                }
                else if (roi.Header.Direction == Slice.Direction.Coronal)
                {
                    if (roi.Z < 0 || roi.Z >= (obj_matrix as VOI_Matrix).Height) continue;
                    VOI_Matrix mat = (obj_matrix as VOI_Matrix);
                    roi.Fill(width, mat.Depeth);
                    ROI_Matrix roim = roi.GetMatrix();
                    mat.Union(roim, new Slice_Coronal((int)roi.Z, mat.Width, mat.Height, mat.Depeth));
                }
                else if (roi.Header.Direction == Slice.Direction.Sagittal)
                {
                    if (roi.Z < 0 || roi.Z >= (obj_matrix as VOI_Matrix).Width) continue;
                    VOI_Matrix mat = (obj_matrix as VOI_Matrix);
                    roi.Fill(height, mat.Depeth);
                    ROI_Matrix roim = roi.GetMatrix();
                    mat.Union(roim, new Slice_Sagittal((int)roi.Z, mat.Width, mat.Height, mat.Depeth));
                }
                
                //roi.Fill((int)width, (int)height);                
                //r.Union( roi.GetMatrix() );
            }
            notReady = false;
        }

        /// <summary>
        /// Creates new matrix if necessary. Takes width,height as input
        /// </summary>
        /// <param name="dimensions">X,Y Dimensions are needed</param>
        protected override void CreateMatrix(params int[] dimensions)
        {
            if (dimensions.Length < 3) throw new TPCROIException("CreateMatrix: Too few parameters. X and Y must be defined.");

            if (notReady || obj_matrix == null || dimensions[0] != obj_matrix.Width || dimensions[1] != obj_matrix.Height)
            {
                obj_matrix = new VOI_Matrix(dimensions[0], dimensions[1], dimensions[2] );
                notReady = true;
            }
        }

        /// <summary>
        /// Changes the filling method of all ROIs in the VOI.
        /// </summary>
        public override Fill_Method FillMethod
        {
            get
            {
                return fill_method;
            }
            set
            {
                foreach (ROI r in ROIs)
                {
                    r.FillMethod = value;
                }
                fill_method = value;
            }
        }

        /// <summary>
        /// Gets the Matrix of VOI
        /// </summary>
        /// <returns>Matrix of VOI</returns>
        public new VOI_Matrix GetMatrix()
        {
            if (notReady)
            {
                if (obj_matrix != null)
                {
                    // If there is some kind of Fill() performed earlier, we can just update the
                    // matrix and return it
                    UpdateMatrix(obj_matrix.Width, obj_matrix.Height, (obj_matrix as VOI_Matrix).Depeth);
                    return (obj_matrix as VOI_Matrix);
                }
                else return null;
            }

            else return (obj_matrix as VOI_Matrix);
        }

        /// <summary>
        /// Iterator for SimpleVOI.
        /// </summary>
        /// <returns>Iterator object</returns>
        public IEnumerator<ROI> GetEnumerator()
        {
            for (int i = 0; i < this.ROIs.Count; i++)
            {
                yield return ROIs[i];
            }
        }

        /// <summary>
        /// Adds new ROI to VOI.
        /// </summary>
        /// <param name="roi">ROI to add.</param>
        public void Add(ROI roi)
        {
            roi.VOIParent = this;
            ROIs.Add(roi);
            notReady = true;
        }

        /// <summary>
        /// Replaces one ROI by another
        /// </summary>
        /// <param name="ROInum">Index number of ROI to replace</param>
        /// <param name="roi">ROI which will replace the ROI at given index</param>
        public void SetRoi(int ROInum, ROI roi)
        {
            if (ROInum < 0 || ROInum >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            roi.VOIParent = this;

            ROIs.Insert(ROInum, roi);
            notReady = true;
        }

        /// <summary>
        /// Gets one ROI from VOI
        /// </summary>
        /// <param name="ROInum">Index number of ROI</param>
        /// <returns>The ROI with given index</returns>
        public ROI GetRoi(int ROInum)
        {
            if (ROInum < 0 || ROInum >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            return ROIs[ROInum];
        }

        /// <summary>
        /// Removes one ROI from VOI
        /// </summary>
        /// <param name="ROInum">ROI number [0..No ROIs-1]</param>
        public void Remove(int ROInum)
        {
            if (ROInum < 0 || ROInum >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            ROIs.RemoveAt(ROInum);
            notReady = true;
        }

        /// <summary>
        /// Catenates another VOI to this VOI
        /// </summary>
        /// <param name="from">Source VOI</param>
        public void Catenate(VOI from)
        {
            if (from == null) throw new TPCROIException("Catenate: Cannot catenate null VOI.");

            foreach (ROI r in from)
            {
                ROI roi = r.Clone();
                roi.VOIParent = this;
                this.Add(roi);
                //zoom = (int)roi.ZoomFactor;
                notReady = true;
            }
        }

    }
}
