﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Matrix class for VOI. Gives the possibility to calculate functions and retrieve pixel values when
    /// Image data is given
    /// </summary>
    public class VOI_Matrix : MatrixObject
    {
        /// <summary>
        /// Returns the depeth(z) of matrix
        /// </summary>
        public int Depeth
        {
            get
            {
                return matrix.Length;
            }
        }

        /// <summary>
        /// Claculates some function inside the ROI and returns the array
        /// of results for every frame of image.
        /// </summary>
        /// <param name="image">Image reference</param>
        /// <param name="function">Function which is used for calculation. Can be for example Mean,Min,Max...</param>
        /// <returns>Result of function inside the VOI as array for every image frame</returns>
        public Double[] Calculate(ref Image.Image image, ROICalculator.func function)
        {
            // Frame length must be given
            int frames;
            if (image.dim.Length > 3) frames = image.dim.GetDimension(3);
            else frames = 1;

            ROICalculator rc = ROICalculator.Create(function, frames);

            for (int z = 0; z < depeth; z++)
            {
                ROI_Matrix roi_matrix = matrix[z];                                
                Calculate(ref image, roi_matrix.cells, new Slice_Transaxial(z, image.width, image.height, image.planes), rc);
            }

            return rc.GetResult();
        }

        /// <summary>
        /// Claculates some function inside the ROI, but doesn't return the result. The
        /// Values are only added to the calculator and result must be got by caller. This
        /// is good when function result of many different objects is needed, for example
        /// (mean of union of many VOIs/ROIs inside the image).
        /// </summary>
        /// <param name="img">Image reference</param>
        /// <param name="calculator">Calculator reference where the result is added to</param>
        public void Calculate(ref Image.Image img, ref ROICalculator calculator)
        {
            for (int z = 0; z < depeth; z++)
            {
                ROI_Matrix roi_matrix = matrix[z];
                Calculate(
                    ref img, 
                    roi_matrix.cells, 
                    new Slice_Transaxial(z, img.width, img.height, img.planes), 
                    calculator);
            }
        }

        /// <summary>
        /// Returns the pixels inside the given image as list.
        /// </summary>
        /// <param name="image">Image data</param>
        /// <returns>Pixels inside the VOI as list</returns>
        public List<ROIValuePoint> GetPixels(ref Image.Image image)
        {
            List<ROIValuePoint> result = new List<ROIValuePoint>();
            for (int z = 0; z < depeth; z++)
            {
                ROI_Matrix roi_matrix = matrix[z];
                result.AddRange( GetPixels(
                    ref image,
                    roi_matrix.cells,
                    new Slice_Transaxial(z, image.width, image.height, image.planes)) );
            }
            return result;
        }

        private ROI_Matrix[] matrix;
        private int depeth;

        /// <summary>
        /// Constructs VOI_Matrix object
        /// </summary>
        /// <param name="width">Width of matrix</param>
        /// <param name="height">Height of matrix</param>
        /// <param name="depeth">Depeth of matrix</param>
        public VOI_Matrix( int width, int height, int depeth )
        {
            this.width = width;
            this.height = height;
            this.depeth = depeth;

            matrix = new ROI_Matrix[depeth];
            for (int i = 0; i < depeth; i++) matrix[i] = new ROI_Matrix(width, height);
        }

        /// <summary>
        /// Gets one slice of VOI_Matrix as ROI_Matrix along z-axis.
        /// </summary>
        /// <param name="z">Z position of slice</param>
        /// <returns>Slice at depeth z as ROI_Matrix</returns>
        public ROI_Matrix this[int z]
        {
            get
            {
                if (z < 0 || z >= depeth) throw new TPCROIException("VOI_Matrix: Z index out of bounds.");

                return matrix[z];
            }
        }

        /// <summary>
        /// Gets one value from VOI_Matrix
        /// </summary>
        /// <param name="x">X coordinate</param>
        /// <param name="y">Y coordinate</param>
        /// <param name="z">Z coordinate</param>
        /// <returns>Value of (x,y,z) in matrix</returns>
        public byte this[int x, int y, int z]
        {
            get
            {
                if (x < 0 || x >= width) throw new TPCROIException("VOI_Matrix: X index out of bounds.");
                if (y < 0 || y >= height) throw new TPCROIException("VOI_Matrix: Y index out of bounds.");
                if (z < 0 || z >= depeth) throw new TPCROIException("VOI_Matrix: Z index out of bounds.");

                return matrix[z][x, y];
            }
            set
            {
                if (x < 0 || x >= width) throw new TPCROIException("VOI_Matrix: X index out of bounds.");
                if (y < 0 || y >= height) throw new TPCROIException("VOI_Matrix: Y index out of bounds.");
                if (z < 0 || z >= depeth) throw new TPCROIException("VOI_Matrix: Z index out of bounds.");

                matrix[z][x, y] = value;
            }
        }

        /// <summary>
        /// Gets 2D matrix from 3D matrix at given depeth 
        /// </summary>
        /// <param name="z">The z coordinate of ROI</param>
        /// <returns>The 2D ROI matrix at given depeth</returns>
        public ROI_Matrix GetRoiMatrix( int z )
        {
            if (z < 0 || z >= depeth) throw new TPCROIException("VOI_Matrix: Z index out of bounds.");
            return matrix[z];
        }

        /// <summary>
        /// Gets 2D matrix from 3D matrix by slice information 
        /// </summary>
        /// <param name="slice">The slice object containing the information which pixels to get</param>
        /// <returns>The 2D ROI matrix</returns>
        public ROI_Matrix GetRoiMatrix( Slice slice )
        {
            if (!slice.CheckDimensions(width, height, depeth))
            {
                throw new TPCROIException("GetRoiMatrix: Dimensions of slice do not match the dimensions of matrix.");
            }

            ROI_Matrix rmat = new ROI_Matrix(slice.Width, slice.Height);

            for (int y = 0; y < slice.Height; y++)
            {
                for (int x = 0; x < slice.Width; x++)
                {
                    slice.X2D = x;
                    slice.Y2D = y;
                    rmat[x, y] = matrix[slice.Z][slice.X, slice.Y];
                }
            }

            return rmat;
        }

        /// <summary>
        /// Gets 2D matrix from 3D matrix by depeth and direction 
        /// </summary>
        /// <param name="depeth">Depeth of 2D matrix</param>
        /// <param name="direction">Direction of the 2D matrix</param>
        /// <returns>The 2D ROI matrix</returns>
        public ROI_Matrix GetRoiMatrix(int depeth, Slice.Direction direction)
        {
            Slice slice = Slice.Create(direction, depeth, this.width, this.height, this.depeth);
            return GetRoiMatrix(slice);
        }

        /// <summary>
        /// Adds all 1's in 2D matrix to VOI matrix. Or operation is used for bits.
        /// </summary>
        /// <param name="r_matrix">2D matrix including the data to be OR-operated.</param>
        /// <param name="slice">reference slice</param>
        public void Union(ROI_Matrix r_matrix, Slice slice)
        {
            if ( !slice.CheckDimensions(this.width, this.height, this.depeth) )
                throw new TPCROIException("Union: Dimensions of the slice doesn't match the dimensions of VOIMatrix.");

            /*if (z < 0 || z >= depeth)
                throw new TPCROIException("SetSliceUnion: Index out of bounds.");*/

            for (int x = 0; x < r_matrix.Width; x++)
            {
                for (int y = 0; y < r_matrix.Height; y++)
                {
                    slice.X2D = x;
                    slice.Y2D = y;
                    matrix[slice.Z][slice.X, slice.Y] |= r_matrix[x, y];
                }
            }
        }

        /// <summary>
        /// Calculates the union of two matrix
        /// </summary>
        /// <param name="v_matrix">3D matrix including the data to be orred.</param>
        public void Union(VOI_Matrix v_matrix)
        {
            // finding of smallest dimensions (needed if matrix size's differ)
            int smallw = this.width;
            int smallh = this.height;
            int smalld = this.depeth;
            if (v_matrix.width < smallw) smallw = v_matrix.width;
            if (v_matrix.height < smallh) smallh = v_matrix.height;
            if (v_matrix.depeth < smalld) smalld = v_matrix.depeth;

            for( int z=0; z<smalld; z++ )
            {
                ROI_Matrix rm1 = matrix[z];
                ROI_Matrix rm2 = v_matrix[z];

                for (int y = 0; y < smallh; y++)
                {
                    for (int x = 0; x < smallw; x++)
                    {
                        rm1[x, y] |= rm2[x, y];   
                    }
                }
            }
        }

        /// <summary>
        /// Calculates the union of two matrix
        /// </summary>
        /// <param name="v_matrix">3D matrix as single array including the data to be orred.</param>
        public void Union(byte[] v_matrix)
        {
            // finding of smallest dimensions (needed if matrix size's differ)
            int smallw = this.width;
            int smallh = this.height;
            int smalld = this.depeth;
            int num = 0;

            for (int z = 0; z < smalld; z++)
            {
                ROI_Matrix rm1 = matrix[z];

                for (int y = 0; y < smallh; y++)
                {
                    for (int x = 0; x < smallw; x++)
                    {
                        rm1[x, y] |= v_matrix[num];
                        num++;
                    }
                }
            }
        }

        /// <summary>
        /// Returns contents of Matrix as single array
        /// </summary>
        /// <returns>Matrix as byte array</returns>
        public byte[] ToArray()
        {
            int smallw = this.width;
            int smallh = this.height;
            int smalld = this.depeth;
            int num = 0;
            byte[] res = new byte[width * height * depeth];

            for (int z = 0; z < smalld; z++)
            {
                ROI_Matrix rm1 = matrix[z];

                for (int y = 0; y < smallh; y++)
                {
                    for (int x = 0; x < smallw; x++)
                    {
                        res[num] = rm1[x, y];
                        num++;
                    }
                }
            }
            return res;
        }

        /// <summary>
        /// Clears the contents of the matrix
        /// </summary>
        public override void Clear()
        {
            for (int z = 0; z < depeth; z++)
            {
                matrix[z].Clear();
            }
        }

    }
}
