﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Xml.XPath;
using System.IO;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Reads/writes information for Vinci software (Vinci version 2.37.0) ROI file. 
    /// Information can be taken from vinci main header(vpx)
    /// Or xml file generated by roireport.
    /// </summary>
    class VinciROIFile : ROIFile
    {
        /// <summary>
        /// VOI and ROI data of file.
        /// </summary>
        private VOI Data;

        private int SamplingRes;
        private Double OffsetX;
        private Double OffsetY;
        private Double OffsetZ;
        private Double PosX;
        private Double PosY;
        private Double PosZ;
        private Double SmplPixelSize;
        private bool TruePixel;

        /// <summary>
        /// Creates the Vinci ROI file object
        /// </summary>
        public VinciROIFile()
        {
            imageGiven = false;
            Header.Resolution_X = 0;
            Header.Resolution_Y = 0;

            Init();
        }
        /// <summary>
        /// Creates the Vinci ROI file object
        /// </summary>
        public VinciROIFile(String filename)
        {
            imageGiven = false;
            this.filename = filename;
            
            Init();
        }

        private bool imageGiven;

        public void SetImageInformation(Image.ImageHeader image_header)
        {
            Header.Resolution_X = image_header.dimx;
            Header.Resolution_Y = image_header.dimy;
            Header.VoxelDimension_X = image_header.sizex;
            Header.VoxelDimension_Y = image_header.sizey;
            imageGiven = true;
        }

        public void SetImageInformation(int width, int height, double pixelXmm, double pixelYmm)
        {
            Header.Resolution_X = width;
            Header.Resolution_Y = height;
            Header.VoxelDimension_X = pixelXmm;
            Header.VoxelDimension_Y = pixelYmm;
            imageGiven = true;
        }

        private double minx = double.MaxValue, maxx = double.MinValue;
        private double miny = double.MaxValue, maxy = double.MinValue;


        /// <summary>
        /// Converts the 0-1 values in Vinci file to pixel coordinates
        /// </summary>
        /// <param name="value">0-1 coordinate value in Vinci file</param>
        /// <returns>Pixel coordinate of given percent value</returns>
        private double ScaleToImage(double value, int Reso, double VoxelDimension, double Offset, bool flip )
        {
            if (TruePixel)
            {
                double factor = (double)SamplingRes / (double)(Reso);

                // Zooming: 0-1 value is decreased by 0.5, multiplied and added
                value -= 0.5d;
                value = value * factor;
                value += (0.5d);

                // percentages are converted to pixels
                value += (Offset / (double)Reso) * factor;
            }
            else // Trilinear
            {
                double factor =
                    (this.SmplPixelSize * 256.0d) / // size of screen in Vinci (mm)
                    ((double)Reso * VoxelDimension); // size of image (mm)


                // Scaling the ROIfiles percent values of vinci screen to percentages of image
                double p = factor * (value - 0.5d);
                p = p + 0.5d;
                
                // Offset move
                if( flip ) p = p - (Offset) / ((double)Reso * VoxelDimension);
                else p = p + (Offset) / ((double)Reso * VoxelDimension);

                value = p;
            }

            // Converting percent value to pixel value
            //if (value < 0) value = -value;
            value = value * Reso;
            if (flip) value = Reso - value;

            return value;
        }

        /// <summary>
        /// Converts the 0-1 values in Vinci file to pixel coordinates
        /// </summary>
        /// <param name="value">0-1 x value in Vinci file</param>
        /// <returns>Pixel x coordinate of value</returns>
        /*private double ScaleToImageX(double value)
        {
            if (TruePixel)
            {
                double factor = (double)SamplingRes / (double)(Header.Resolution_X);

                // Zooming: 0-1 value is decreased by 0.5, multiplied and added
                value -= 0.5d;
                value = value * factor;
                value += (0.5d);

                // percentages are converted to pixels
                value += (OffsetX / (double)Header.Resolution_X) * factor;
            }
            else // Trilinear
            {
                double factor =
                    (this.SmplPixelSize * 256.0d) / // size of screen in Vinci (mm)
                    ((double)Header.Resolution_X * Header.VoxelDimension_X ); // size of image (mm)

                // Scaling the ROIfiles percent values of vinci screen to percentages of image
                double pixX = factor * (value - 0.5d);
                pixX = pixX + 0.5d;

                // Offset move
                pixX = pixX + (OffsetX) / ((double)Header.Resolution_X * Header.VoxelDimension_X);
                value = pixX;

                if (value < minx) minx = value;
                if (value > maxx) maxx = value;
            }

            // Converting percent value to pixel value
            value = value * Header.Resolution_X;

            return value;
        }

        /// <summary>
        /// Converts the 0-1 values in Vinci file to pixel coordinates
        /// </summary>
        /// <param name="value">0-1 x value in Vinci file</param>
        /// <returns>Pixel x coordinate of value</returns>
        private double ScaleToImageZ(double value)
        {
            // Calculate the percent value:
            value = value / (SmplPixelSize * 256.0);

            // Trilinear
            {
                double factor =
                    (this.SmplPixelSize * 256.0d) / // size of screen in Vinci (mm)
                    ((double)Header.Resolution_Z * Header.VoxelDimension_Z ); // size of image (mm)

                // Scaling the ROIfiles percent values of vinci screen to percentages of image
                double pixZ = factor * (value - 0.5d);
                pixZ = pixZ + 0.5d;

                // Offset move
                pixZ = pixZ + (OffsetZ) / ((double)Header.Resolution_Z * Header.VoxelDimension_Z);
                value = pixZ;

                if (value < minx) minx = value;
                if (value > maxx) maxx = value;
            }

            // Converting percent value to pixel value
            value = value * Header.Resolution_Z;

            return Header.Resolution_Z - value;
        }

        /// <summary>
        /// Converts the 0-1 values in Vinci file to pixel coordinates
        /// </summary>
        /// <param name="value">0-1 y value in Vinci file</param>
        /// <returns>Pixel y coordinate of value</returns>
        private double ScaleToImageY(double value)
        {
            // Y information is for some reason stored upside down
            value = 1.0d - value;

            if (TruePixel)
            {
                double factor = (double)SamplingRes / (double)Header.Resolution_Y;

                // Zooming: 0-1 value is decreased by 0.5, multiplied and added
                value -= 0.5d;
                value = value * factor;
                value += (0.5d);

                // percentages are converted to pixels
                value -= (OffsetY / (double)Header.Resolution_Y) * factor;
            }
            else // Trilinear
            {
                double factor = 
                    (this.SmplPixelSize * 256.0d) / // size of screen in Vinci (mm)
                    ((double)Header.Resolution_Y * Header.VoxelDimension_Y ); // size of image (mm)

                // Scaling the ROIfiles percent values of vinci screen to percentages of image
                double pixY = factor * (value - 0.5d);
                pixY = pixY + 0.5d;

                // Offset move
                pixY = pixY + (OffsetY) / ((double)Header.Resolution_Y * Header.VoxelDimension_Y );
                value = pixY;

                if (value < miny) miny = value;
                if (value > maxy) maxy = value;
            }

            // Converting percent value to pixel value
            value = value * Header.Resolution_Y;

            return value;
        }*/

        /// <summary>
        /// Writes ROI data as vinci project file (vpx).
        /// </summary>
        /// <param name="filename">Name of Vinci project file</param>
        public void ToProjectFile(String filename)
        {/*
            VinciROIFile file = new VinciROIFile(filename);
            file.ReadFile();


            filestream = new FileStream(filename, FileMode.Open, FileAccess.ReadWrite);
            XPathDocument doc = new XPathDocument(filestream);
            XPathNavigator navigator = doc.CreateNavigator();

            // In project file there should be some kind of image information
            navigator.MoveToRoot();
            XPathNodeIterator nodes = navigator.Select("/VinciXMLXchange/ProjectInfo/File");
            nodes.MoveNext();
            if (nodes.Current.Matches("/VinciXMLXchange/ProjectInfo/File")) result = true;

            // If we have only imported xml, we should have following information
            navigator.MoveToRoot();
            nodes = navigator.Select("/VinciXMLXchange/New");
            nodes.MoveNext();
            if (nodes.Current.Matches("/VinciXMLXchange/New")) result = true;

            filestream.Close();*/
        }

        /// <summary>
        /// Writes the ROI information to VinciROIfile. 
        /// </summary>
        public override void WriteFile()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;
            WriteData( ref fileWriter );

            filestream.Close();
        }

        /// <summary>
        /// Checks if the file is valid Vinci ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
        public override bool CheckFormat()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");
            if (!File.Exists(filename)) throw new TPCROIFileException("Cannot find ROI file: " + filename);


            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream;
            bool result = false;

            try
            {
                filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
                XPathDocument doc = new XPathDocument(filestream);
                XPathNavigator navigator = doc.CreateNavigator();

                // In project file there should be some kind of image information
                navigator.MoveToRoot();
                XPathNodeIterator nodes = navigator.Select("/VinciXMLXchange/ProjectInfo/File");
                nodes.MoveNext();
                if ( nodes.Current.Matches("/VinciXMLXchange/ProjectInfo/File")) result = true;

                // If we have only imported xml, we should have following information
                navigator.MoveToRoot();
                nodes = navigator.Select("/VinciXMLXchange/New");
                nodes.MoveNext();
                if (nodes.Current.Matches("/VinciXMLXchange/New")) result = true;

                filestream.Close();
            }
            catch (Exception)
            {
                result = false;
            }

            return result;
        }

        /// <summary>
        /// Reads Vinci ROI files.
        /// </summary>
        public override void ReadFile()
        {
            if (!CheckFormat()) throw new TPCROIFileException("Vinci file was not in correct format.");

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream;

            // At this point we can assume that the vinci file is valid and we can empty the previous data
            Init();

            try
            {
                filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
                //XmlTextReader reader = new XmlTextReader(filestream);
                XPathDocument doc = new XPathDocument(filestream);

                XPathNavigator navigator = doc.CreateNavigator();

                // Lets try to read file as vinci header file first
                bool success = ReadVinciHeaderFile(navigator);

                // If the reading did not success, it may be that the file is exported vinci ROI file
                if (!success) success = ReadVinciXmlFile(navigator);

                // If there was not success, file cannot be correct
                if (!success) throw new TPCROIFileException("Vinci file was not in correct format.");


                filestream.Close();

                Console.WriteLine("X-alue = (" + minx + " - " + maxx + ")" + "  Välimatka = " + (maxx - minx) );
                Console.WriteLine("Y-alue = (" + miny + " - " + maxy + ")" + "  Välimatka = " + (maxy - miny) );

            }
            catch (Exception e)
            {
                throw new TPCROIFileException("Cannot open file. File is not in correct format. " + e);
            }

            this.VOIs = new List<VOI>();
            VOIs.Add( Data );
        }

        /// <summary>
        /// Writes the contents of ImadeusROIFile to string
        /// </summary>
        /// <returns>String containing the contents of ImadeusFile</returns>
        public override string ToString()
        {
            return "e"; // WriteHeader() + WriteData();
        }

        #region private members

        private new void Init()
        {
            // The resolution values and voxel dimensions must be kept if those
            // are given earlier
            int TempResX = Header.Resolution_X;
            int TempResY = Header.Resolution_Y;
            double TempVoxX = Header.VoxelDimension_X;
            double TempVoxY = Header.VoxelDimension_Y;

            base.Init();

            if (imageGiven)
            {
                Header.Resolution_X = TempResX;
                Header.Resolution_Y = TempResY;
                Header.VoxelDimension_X = TempVoxX;
                Header.VoxelDimension_Y = TempVoxY;
            }

            Data = new VOI();
            SamplingRes = 256;
            OffsetX = 0;
            OffsetY = 0;
            OffsetZ = 0;
            PosX = 0;
            PosY = 0;
            PosZ = 0;
            SmplPixelSize = 1.0d;
            TruePixel = true;
        }

        /// <summary>
        /// Tries to load image dimensions by information got from vinci vpx(project) file
        /// </summary>
        /// <param name="navigator">Navigator object to XML</param>
        /// <returns>True, if success</returns>
        private bool TryToReadImageDimensions(XPathNavigator navigator)
        {
            Console.WriteLine("You have not given image dimension information. Trying to read those from image file...");
            navigator.MoveToRoot();

            XPathNodeIterator nodes = navigator.Select("/VinciXMLXchange/ProjectInfo/File");

            nodes.MoveNext();
            if (nodes.Current.Matches("/VinciXMLXchange/ProjectInfo/File"))
            {
                // The file section was found from vinci header file
                //nodes = navigator.Select("/VinciXMLXchange/ProjectInfo/File");
                String imagefilename = nodes.Current.Value;

                String fullname = Path.GetDirectoryName(this.filename) + "\\" + imagefilename;
                
                Console.WriteLine("Image file path in header file:" + fullname);

                Image.FileType itype = Image.ImageFile.ResolveFileFormat(fullname);

                // Next we read the image
                ImageFile imagefile;
                if      (itype == TPClib.Image.FileType.Analyze75) imagefile = new Analyze75File(fullname);
                else if (itype == TPClib.Image.FileType.DICOM) imagefile = new DicomFile(fullname);
                else if (itype == TPClib.Image.FileType.Ecat7) imagefile = new Ecat7File(fullname);
                else return false;

                Console.WriteLine("Image file recognized to " + itype );

                if( imagefile == null ) return false;

                imagefile.ReadFile();
                Header.Resolution_X = imagefile.header.dimx;
                Header.Resolution_Y = imagefile.header.dimy;
                Header.Resolution_Z = imagefile.header.dimz;
                Header.VoxelDimension_X = imagefile.header.sizex;
                Header.VoxelDimension_Y = imagefile.header.sizey;
                Header.VoxelDimension_Z = imagefile.header.sizez;

                imageGiven = true;
            }
            else return false; // failure. file was not correct Vinci header file

            Console.WriteLine("Image width and height information retrieved succesfully.");
            return true;
        }

        /// <summary>
        /// Reads ROIs from Vinci's project file "vpx".
        /// </summary>
        /// <param name="navigator">Navigator to xml file</param>
        /// <returns>True if success</returns>
        private bool ReadVinciHeaderFile(XPathNavigator navigator)
        {
            navigator.MoveToRoot();

            //XPathNodeIterator root = navigator.MoveToRoot();
            XPathNodeIterator nodes = navigator.Select("/VinciXMLXchange/ParameterSection/Use/Action/New");
            //navigator.ComparePosition(nodes);

            nodes.MoveNext();
            if (nodes.Current.Matches("/VinciXMLXchange/ParameterSection/Use/Action/New"))
            {
                // The file is normal vinci header file
                nodes = navigator.Select("/VinciXMLXchange/ParameterSection/Use/Action/New");
            }
            else return false; // failure. file was not correct Vinci header file

            if (!imageGiven)
            {
                if( !TryToReadImageDimensions( navigator ) )
                    throw new TPCROIFileException("You have not given image information. Reading of Vinci ROIs need image info.");
            }

            // We need some information about image and Vinci settings before we can convert vinci ROIs
            // To our format
            ReadImageInfo(navigator);

            // Geometrical information of all ROIs
            ReadGeometry(nodes);

            // Success
            return true;
        }

        private void ReadImageInfo(XPathNavigator navigator)
        {
            /*
            // Lets try to read file first as truepixel
            bool success = ReadTruePixelInformation(navigator);

            // If there was not success, we try file as trilinear
            if (!success) success = ReadTrilinearPixelInformation(navigator);
            if (!success) throw new Exception();
             */

            XPathNodeIterator nodes = navigator.Select(
                "/VinciXMLXchange/ParameterSection/Use/Action/New/Create/ImageSettings/OtherSettings/SetSamplingReslTP");
            nodes.MoveNext();

            // Truepixel mode has some additional options. If these exists, the image is handled as truepixel
            if (nodes.Current.Matches("/VinciXMLXchange/ParameterSection/Use/Action/New/Create/ImageSettings/OtherSettings/SetSamplingReslTP"))
            {
                Console.WriteLine("File is TruePixel file.");

                // The file is normal vinci header file
                nodes = navigator.Select("/VinciXMLXchange/ParameterSection/Use/Action/New/Create/ImageSettings/OtherSettings/SetSamplingReslTP");
                nodes.MoveNext();

                // Zooming of TruePixel is hadled with SamplingRes
                SamplingRes = Convert.ToInt32(nodes.Current.Value);

                this.TruePixel = true;
            }
            else
            {
                Console.WriteLine("File is Trilinear file.");
                this.TruePixel = false;
            }

            nodes = navigator.Select("/VinciXMLXchange/ParameterSection/Use/Action/New/Create/ImageSettings/Reslice");
            nodes.MoveNext();

            String line = nodes.Current.SelectSingleNode("Offset").Value;
            String[] words = line.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            this.OffsetX = HelpFunctions.StrToDouble(words[0]);
            this.OffsetY = HelpFunctions.StrToDouble(words[1]);
            this.OffsetZ = HelpFunctions.StrToDouble(words[2]);

            line = nodes.Current.SelectSingleNode("SmplPixelSize").Value;
            words = line.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            SmplPixelSize = HelpFunctions.StrToDouble(words[0]);

            line = nodes.Current.SelectSingleNode("Pos").Value;
            words = line.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            this.PosX = HelpFunctions.StrToDouble(words[0]);
            this.PosY = HelpFunctions.StrToDouble(words[1]);
            this.PosZ = HelpFunctions.StrToDouble(words[2]);

            Console.WriteLine("OffsetX=" + OffsetX);
            Console.WriteLine("OffsetY=" + OffsetY);
            Console.WriteLine("OffsetZ=" + OffsetZ);

            Console.WriteLine("PosX=" + PosX);
            Console.WriteLine("SmplPixelSize=" + SmplPixelSize );
        }


            /*
            foreach (XPathNavigator item in nodes)
            {
                XPathNavigator target = item.SelectSingleNode("Target");
                if (target == null) break;

                String jee = target.SelectSingleNode("Type").Value;

                Console.WriteLine(jee);
                if (jee.Equals("Roi2DRect")) readRect(item);
                else if (jee.Equals("Roi2DEllipse")) readEllipse(item);
                else if (jee.Equals("Roi2DPolygon")) readPolygon(item);
                else throw new Exception();
            }*/



        /// <summary>
        /// Reads Vinci's Exported XML ROI files. There is not header information available so the
        /// ROIs are probably not in right places.
        /// </summary>
        /// <param name="navigator">Navigator object to XML file</param>
        /// <returns>True if reading was success</returns>
        private bool ReadVinciXmlFile(XPathNavigator navigator)
        {
            if (!imageGiven)
                throw new TPCROIFileException("Cannot read vinci file. You have not given image dimension information.");

            navigator.MoveToRoot();

            // Lets try if the file would be exported XML ROI file
            XPathNodeIterator nodes = navigator.Select("/VinciXMLXchange/New");
            nodes.MoveNext();
            if (nodes.Current.Matches("/VinciXMLXchange/New"))
            {
                // File is vinci roi file
                nodes = navigator.Select("/VinciXMLXchange/New");
            }
            else return false;

            // At this point we can assume that the vinci file is valid and we can empty the previous data
            Init();

            // Geometrical information of all ROIs
            ReadGeometry(nodes);

            return true;
        }

        private void ReadGeometry(XPathNodeIterator nodes)
        {
            foreach (XPathNavigator item in nodes)
            {
                XPathNavigator target = item.SelectSingleNode("Target");
                if (target == null) break;

                String jee = target.SelectSingleNode("Type").Value;

                Console.WriteLine(jee);
                if (jee.Equals("Roi2DRect")) readRect(item);
                else if (jee.Equals("Roi2DEllipse")) readEllipse(item);
                else if (jee.Equals("Roi2DPolygon")) readPolygon(item);
                else throw new Exception();
            }
        }

        private void readRect(XPathNavigator element)
        {
            XPathNavigator target = element.SelectSingleNode("Create");
            if (target == null) return;

            String name = target.SelectSingleNode("SetRectName").Value;            
            String ortho = target.SelectSingleNode("SetOrthoPart").Value;   

            bool paintfill = target.SelectSingleNode("SetPaintFilled").Value.Equals("On");
            Double rotation = HelpFunctions.StrToDouble( target.SelectSingleNode("SetRotation").Value );
            String line = target.SelectSingleNode("SetRectGeometry").Value;

            string[] words = line.Split( new char[] {' ', '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries );

            // Must have x and y values.
            if (words.Length % 2 == 1) throw new TPCROIFileException("File was not in correct format.");


            int ResoX = 1; double VoxelDimX = 1; double OffX = 0; bool FlipX = false;
            int ResoY = 1; double VoxelDimY = 1; double OffY = 0; bool FlipY = false;
            int ResoZ = 1; double VoxelDimZ = 1; double OffZ = 0; bool FlipZ = false;
            double depthPos = 0;
            Slice.Direction direction = Slice.Direction.Transaxial;
            if (ortho.Equals("0"))
            {
                // Transaxial:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Y; VoxelDimY = Header.VoxelDimension_Y; OffY = OffsetY; FlipY = true;
                ResoZ = Header.Resolution_Z; VoxelDimZ = Header.VoxelDimension_Z; OffZ = OffsetZ; FlipZ = true;
                depthPos = PosZ;
                direction = Slice.Direction.Transaxial;
            }
            if (ortho.Equals("1"))
            {
                // Coronal:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_Y; VoxelDimZ = Header.VoxelDimension_Y; OffZ = OffsetY; FlipZ = false;
                depthPos = PosY;
                direction = Slice.Direction.Coronal;
            }
            if (ortho.Equals("2"))
            {
                // Sagittal:
                ResoX = Header.Resolution_Y; VoxelDimX = Header.VoxelDimension_Y; OffX = OffsetY; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_X; VoxelDimZ = Header.VoxelDimension_X; OffZ = OffsetX; FlipZ = false;
                depthPos = PosX;
                direction = Slice.Direction.Sagittal;
            }
            
            // cordinates are in point1, point2 format. We must calculate the width, height and the upperleft corner
            double x1 = ScaleToImage( HelpFunctions.StrToDouble(words[0]), ResoX, VoxelDimX, OffX, FlipX );
            double y1 = ScaleToImage(HelpFunctions.StrToDouble(words[1]), ResoY, VoxelDimY, OffY, FlipY );
            double x2 = ScaleToImage( HelpFunctions.StrToDouble(words[2]), ResoX, VoxelDimX, OffX, FlipX );
            double y2 = ScaleToImage(HelpFunctions.StrToDouble(words[3]), ResoY, VoxelDimY, OffY, FlipY );

            double x = x1;
            double y = y1;

            // lets find the upperleft corner
            if (x2 < x1) x = x2;
            if (y2 < y1) y = y2;

            double width = Math.Abs(x2 - x1);
            double height = Math.Abs(y2 - y1);

            RectangleROI roi = new RectangleROI(x + (width / 2.0d), y + (height / 2.0d), width, height);

            roi.CoordinateSystem = Coordinate_System.Pixel;
            roi.FillMethod = Fill_Method.Imadeus;
            roi.Header.ROIName = name;
            roi.Angle = rotation;
            roi.Z = (int)(ScaleToImage(depthPos / (SmplPixelSize * 256.0), ResoZ, VoxelDimZ, OffZ, FlipZ ));

            roi.Header.Direction = direction;
            Data.Add(roi);
        }

        private void readEllipse(XPathNavigator element)
        {
            XPathNavigator target = element.SelectSingleNode("Create");
            if (target == null) return;

            String name = target.SelectSingleNode("SetEllipseName").Value;
            bool paintfill = target.SelectSingleNode("SetPaintFilled").Value.Equals("On");
            Double rotation = HelpFunctions.StrToDouble(target.SelectSingleNode("SetRotation").Value);
            String line = target.SelectSingleNode("SetEllipseGeometry").Value;
            String ortho = target.SelectSingleNode("SetOrthoPart").Value;            

            string[] words = line.Split(new char[] { ' ', '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

            // Must have x and y values.
            if (words.Length % 2 == 1) throw new TPCROIFileException("File was not in correct format.");


            int ResoX = 1; double VoxelDimX = 1; double OffX = 0; bool FlipX = false;
            int ResoY = 1; double VoxelDimY = 1; double OffY = 0; bool FlipY = false;
            int ResoZ = 1; double VoxelDimZ = 1; double OffZ = 0; bool FlipZ = false;
            double depthPos = 0;
            Slice.Direction direction = Slice.Direction.Transaxial;
            if (ortho.Equals("0"))
            {
                // Transaxial:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Y; VoxelDimY = Header.VoxelDimension_Y; OffY = OffsetY; FlipY = true;
                ResoZ = Header.Resolution_Z; VoxelDimZ = Header.VoxelDimension_Z; OffZ = OffsetZ; FlipZ = true;
                depthPos = PosZ;
                direction = Slice.Direction.Transaxial;
            }
            if (ortho.Equals("1"))
            {
                // Coronal:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_Y; VoxelDimZ = Header.VoxelDimension_Y; OffZ = OffsetY; FlipZ = false;
                depthPos = PosY;
                direction = Slice.Direction.Coronal;
            }
            if (ortho.Equals("2"))
            {
                // Sagittal:
                ResoX = Header.Resolution_Y; VoxelDimX = Header.VoxelDimension_Y; OffX = OffsetY; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_X; VoxelDimZ = Header.VoxelDimension_X; OffZ = OffsetX; FlipZ = false;
                depthPos = PosX;
                direction = Slice.Direction.Sagittal;
            }

            // cordinates are in point1, point2 format. We must calculate the width, height and the upperleft corner
            double x1 = ScaleToImage(HelpFunctions.StrToDouble(words[0]), ResoX, VoxelDimX, OffX, false);
            double y1 = ScaleToImage(HelpFunctions.StrToDouble(words[1]), ResoY, VoxelDimY, OffY, true);
            double x2 = ScaleToImage(HelpFunctions.StrToDouble(words[2]), ResoX, VoxelDimX, OffX, false );
            double y2 = ScaleToImage(HelpFunctions.StrToDouble(words[3]), ResoY, VoxelDimY, OffY, true );


            double x = x1;
            double y = y1;

            // lets find the upperleft corner
            if (x2 < x1) x = x2;
            if (y2 < y1) y = y2;

            double width = Math.Abs(x2-x1);
            double height = Math.Abs(y2-y1);

            EllipseROI roi = new EllipseROI( x+(width/2.0d), y + (height/2.0d), width, height );
            roi.CoordinateSystem = Coordinate_System.Pixel;
            roi.FillMethod = Fill_Method.Imadeus;
            roi.Header.ROIName = name;
            roi.Angle = rotation;

            roi.Z = (int)(ScaleToImage(depthPos / (SmplPixelSize * 256.0), ResoZ, VoxelDimZ, OffZ, FlipZ));
            roi.Header.Direction = direction;

            Data.Add(roi);
        }

        private void readPolygon(XPathNavigator element)
        {
            XPathNavigator target = element.SelectSingleNode("Create");
            if (target == null) return;

            String name = target.SelectSingleNode("SetPolygonName").Value;
            bool paintfill = target.SelectSingleNode("SetPaintFilled").Value.Equals("On");
            Double rotation = HelpFunctions.StrToDouble(target.SelectSingleNode("SetRotation").Value);
            String ortho = target.SelectSingleNode("SetOrthoPart").Value;
            //String line = target.SelectSingleNode("SetPolygonGeometry").Value;

            TraceROI roi = new TraceROI();
            roi.CoordinateSystem = Coordinate_System.Pixel;
            roi.FillMethod = Fill_Method.Imadeus;
            roi.Header.ROIName = name;
            roi.Angle = rotation;



            int ResoX = 1; double VoxelDimX = 1; double OffX = 0; bool FlipX = false;
            int ResoY = 1; double VoxelDimY = 1; double OffY = 0; bool FlipY = false;
            int ResoZ = 1; double VoxelDimZ = 1; double OffZ = 0; bool FlipZ = false;
            double depthPos = 0;
            Slice.Direction direction = Slice.Direction.Transaxial;
            if (ortho.Equals("0"))
            {
                // Transaxial:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Y; VoxelDimY = Header.VoxelDimension_Y; OffY = OffsetY; FlipY = true;
                ResoZ = Header.Resolution_Z; VoxelDimZ = Header.VoxelDimension_Z; OffZ = OffsetZ; FlipZ = true;
                depthPos = PosZ;
                direction = Slice.Direction.Transaxial;
            }
            if (ortho.Equals("1"))
            {
                // Coronal:
                ResoX = Header.Resolution_X; VoxelDimX = Header.VoxelDimension_X; OffX = OffsetX; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_Y; VoxelDimZ = Header.VoxelDimension_Y; OffZ = OffsetY; FlipZ = false;
                depthPos = PosY;
                direction = Slice.Direction.Coronal;
            }
            if (ortho.Equals("2"))
            {
                // Sagittal:
                ResoX = Header.Resolution_Y; VoxelDimX = Header.VoxelDimension_Y; OffX = OffsetY; FlipX = false;
                ResoY = Header.Resolution_Z; VoxelDimY = Header.VoxelDimension_Z; OffY = OffsetZ; FlipY = true;
                ResoZ = Header.Resolution_X; VoxelDimZ = Header.VoxelDimension_X; OffZ = OffsetX; FlipZ = false;
                depthPos = PosX;
                direction = Slice.Direction.Sagittal;
            }

            // cordinates are in point1, point2 format. We must calculate the width, height and the upperleft corner
            /*double x1 = ScaleToImage(HelpFunctions.StrToDouble(words[0]), ResoX, VoxelDimX, OffX);
            double y1 = ScaleToImage(HelpFunctions.StrToDouble(words[1]), ResoY, VoxelDimY, OffY);
            double x2 = ScaleToImage(HelpFunctions.StrToDouble(words[2]), ResoX, VoxelDimX, OffX);
            double y2 = ScaleToImage(HelpFunctions.StrToDouble(words[3]), ResoY, VoxelDimY, OffY);*/

            // Mean values so that the polygon origo(rotate point) is correct
            double meanX = 0.0d;
            double meanY = 0.0d;
            
            // max and min values which are needed when the mean is calculated
            double minX = double.MaxValue;
            double minY = double.MaxValue;
            double maxX = double.MinValue;
            double maxY = double.MinValue;

            List<Double> x = new List<double>();
            List<Double> y = new List<double>();

            XPathNodeIterator points = target.Select("SetPolygonGeometry/Point");
            foreach( XPathNavigator point in points )
            {
                if (point == null) break;
                String line = point.Value;//point.SelectSingleNode("Point").Value;
                string[] words = line.Split(new char[] { ' ', '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);
                
                // There should always be x and y
                if (words.Length != 2) throw new Exception();

                double xx = ScaleToImage(HelpFunctions.StrToDouble(words[0]), ResoX, VoxelDimX, OffX, FlipX );
                double yy = ScaleToImage(HelpFunctions.StrToDouble(words[1]), ResoY, VoxelDimY, OffY, FlipY );

                // coordinates are placed to list, because the origo of ROI must be known
                // before adding to the ROIs point list. In Vinci file there is no origo points.
                // It must be calculated
                x.Add( xx );
                y.Add( yy );

                if (xx > maxX) maxX = xx;
                if (yy > maxY) maxY = yy;
                if (xx < minX) minX = xx;
                if (yy < minY) minY = yy;
            }

            // The origo axel which is used for rotating
            meanX = (maxX + minX) / 2.0d;
            meanY = (maxY + minY) / 2.0d;

            // Finally we add the points to the ROI
            for (int i = 0; i < x.Count; i++)
            {
                roi.AddPoint(x[i] - meanX, y[i] - meanY);
            }

            roi.X = meanX;
            roi.Y = meanY;

            roi.Z = (int)(ScaleToImage(depthPos / (SmplPixelSize * 256.0), ResoZ, VoxelDimZ, OffZ, FlipZ ));
            roi.Header.Direction = direction;

            Data.Add(roi);
        }

        private void WriteData( ref StreamWriter writer )
        {
            writer.Write("<VinciXMLXchange><Version>0.10</Version>\r\n");

            foreach (ROI r in Data)
            {
                if (r is TraceROI) WritePolygon(ref writer, r as TraceROI );
                if (r is EllipseROI) WriteEllipse(ref writer, r as EllipseROI);
                if (r is RectangleROI) WriteRectangle(ref writer, r as RectangleROI);

            }

            writer.WriteLine("</VinciXMLXchange>");
        }

        private void WriteEllipse(ref StreamWriter writer, EllipseROI roi)
        {
            StringBuilder str = new StringBuilder();

            // The direction of roi.
            int ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Transaxial) ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Coronal) ortho = 1;
            if (roi.Header.Direction == Slice.Direction.Sagittal) ortho = 2;

            str.Append("<New>\r\n\t<Target>\r\n");
            str.Append("\t\t<Type>Roi2DEllipse</Type>\r\n");
            str.Append("\t\t<Name>Roi2DEllipse</Name>\r\n");
            str.Append("\t</Target>\r\n");

            str.Append("\t<Create>\r\n");
            str.Append("\t\t<SetEllipseName>" + roi.Header.ROIName + "</SetEllipseName>\r\n");
            str.Append("\t\t<SetValidWithin>On</SetValidWithin>\r\n");
            str.Append("\t\t<SetPaintFilled>On</SetPaintFilled>\r\n");
            str.Append("\t\t<SetPaintBorder>On</SetPaintBorder>\r\n");
            str.Append("\t\t<SetEvaluate_On>On</SetEvaluate_On>\r\n");
            str.Append("\t\t<SetOrthoPart>" + ortho + "</SetOrthoPart>\r\n"); // this will be different some day
            str.Append("\t\t<SetColorIndex>5</SetColorIndex>\r\n");
            str.Append("\t\t<SetSelected>-1</SetSelected>\r\n");
            str.Append("\t\t<SetListType>ImageList</SetListType>\r\n");
            str.Append("\t\t<SetColorRef>0xff00ff</SetColorRef>\r\n");
            str.Append("\t\t<SetRotation>" + HelpFunctions.DoubleToStr( roi.Angle ) + "</SetRotation>\r\n");
            str.Append("\t\t<DisplaySizeDuringLastMove>256 256</DisplaySizeDuringLastMove>\r\n");

            Double width = roi.ShapeWidth;
            Double height = roi.ShapeHeight;
    
            String x = HelpFunctions.DoubleToStr(roi.X - (width/2.0d) );
            String y = HelpFunctions.DoubleToStr(roi.Y - (height / 2.0d));
            String x2 = HelpFunctions.DoubleToStr(roi.X + (width / 2.0d));
            String y2 = HelpFunctions.DoubleToStr(roi.Y + (height / 2.0d));

            str.Append("");
            str.Append("\t\t<SetEllipseGeometry>" + x + " " + y + " " + x2 + " " + y2);
            str.Append("</SetEllipseGeometry>\r\n");

            str.Append("\t</Create>\r\n");
            str.Append("</New>\r\n");

            writer.Write(str.ToString());
        }

        private void WriteRectangle(ref StreamWriter writer, RectangleROI roi)
        {
            StringBuilder str = new StringBuilder();

            // The direction of roi.
            int ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Transaxial) ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Coronal) ortho = 1;
            if (roi.Header.Direction == Slice.Direction.Sagittal) ortho = 2;

            str.Append("<New>\r\n\t<Target>\r\n");
            str.Append("\t\t<Type>Roi2DRect</Type>\r\n");
            str.Append("\t\t<Name>Roi2DRect</Name>\r\n");
            str.Append("\t</Target>\r\n");

            str.Append("\t<Create>\r\n");
            str.Append("\t\t<SetRectName>" + roi.Header.ROIName + "</SetRectName>\r\n");
            str.Append("\t\t<SetValidWithin>On</SetValidWithin>\r\n");
            str.Append("\t\t<SetPaintFilled>On</SetPaintFilled>\r\n");
            str.Append("\t\t<SetPaintBorder>On</SetPaintBorder>\r\n");
            str.Append("\t\t<SetEvaluate_On>On</SetEvaluate_On>\r\n");
            str.Append("\t\t<SetOrthoPart>"+ ortho + "</SetOrthoPart>\r\n"); // this will be different some day
            str.Append("\t\t<SetColorIndex>5</SetColorIndex>\r\n");
            str.Append("\t\t<SetSelected>-1</SetSelected>\r\n");
            str.Append("\t\t<SetListType>ImageList</SetListType>\r\n");
            str.Append("\t\t<SetColorRef>0xff00ff</SetColorRef>\r\n");
            str.Append("\t\t<SetRotation>" + HelpFunctions.DoubleToStr( roi.Angle ) + "</SetRotation>\r\n");
            str.Append("\t\t<DisplaySizeDuringLastMove>256 256</DisplaySizeDuringLastMove>\r\n");

            Double width = roi.ShapeWidth;
            Double height = roi.ShapeHeight;

            String x = HelpFunctions.DoubleToStr(roi.X - (width / 2.0d));
            String y = HelpFunctions.DoubleToStr(roi.Y - (height / 2.0d));
            String x2 = HelpFunctions.DoubleToStr(roi.X + (width / 2.0d));
            String y2 = HelpFunctions.DoubleToStr(roi.Y + (height / 2.0d));

            str.Append("");
            str.Append("\t\t<SetRectGeometry>" + x + " " + y + " " + x2 + " " + y2);
            str.Append("</SetRectGeometry>\r\n");

            str.Append("\t</Create>\r\n");
            str.Append("</New>\r\n");

            writer.Write(str.ToString());
        }

        private void WritePolygon(ref StreamWriter writer, TraceROI roi )
        {
            StringBuilder str = new StringBuilder();

            // The direction of roi.
            int ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Transaxial) ortho = 0;
            if (roi.Header.Direction == Slice.Direction.Coronal) ortho = 1;
            if (roi.Header.Direction == Slice.Direction.Sagittal) ortho = 2;

            str.Append("<New>\r\n\t<Target>\r\n");
            str.Append("\t\t<Type>Roi2DPolygon</Type>\r\n");
            str.Append("\t\t<Name>Roi2DPolygon</Name>\r\n");
            str.Append("\t</Target>\r\n");

            str.Append("\t<Create>\r\n");
            str.Append("\t\t<SetPolygonName>" + roi.Header.ROIName + "</SetPolygonName>\r\n");
            str.Append("\t\t<SetAutoName/>\r\n");
            str.Append("\t\t<SetValidWithin>On</SetValidWithin>\r\n");
            str.Append("\t\t<SetPaintFilled>On</SetPaintFilled>\r\n");
            str.Append("\t\t<SetPaintBorder>On</SetPaintBorder>\r\n");
            str.Append("\t\t<SetEvaluate_On>On</SetEvaluate_On>\r\n");
            str.Append("\t\t<SetOrthoPart>" + ortho + "</SetOrthoPart>\r\n"); // this will be different some day
            str.Append("\t\t<SetColorIndex>5</SetColorIndex>\r\n");
            str.Append("\t\t<SetShowPointHandles>On</SetShowPointHandles>\r\n");
            str.Append("\t\t<SetSelected>-1</SetSelected>\r\n");
            str.Append("\t\t<SetListType>ImageList</SetListType>\r\n");
            str.Append("\t\t<SetColorRef>0xff00ff</SetColorRef>\r\n");
            str.Append("\t\t<SetRotation>" + HelpFunctions.DoubleToStr( roi.Angle ) + "</SetRotation>\r\n");
            str.Append("\t\t<DisplaySizeDuringLastMove>256 256</DisplaySizeDuringLastMove>\r\n");
            str.Append("\t\t<SetPolygonGeometry>\r\n");
            
            foreach( ROIPoint point in roi.OriginalPoints )
            {
                str.Append("\t\t\t<Point>" + 
                    HelpFunctions.DoubleToStr( point.x ) + " " +
                    HelpFunctions.DoubleToStr( point.y ) + 
                    "</Point>\r\n");
            }

            str.Append("\t\t</SetPolygonGeometry>\r\n");
            str.Append("\t</Create>\r\n");            
            str.Append("</New>\r\n");

            writer.Write( str.ToString() );
        }

        #endregion
    }
}
