﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using TPClib;
using TPClib.Image;

namespace TPCConsoleApplication
{
    /// <summary>
    /// Test bench for console program
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_Program : NUnitTestbench
    {
        /// <summary>
        /// Test printing of help information
        /// </summary>
        [Test]
        public void Test1_0_Help()
        {
            Console.WriteLine(System.IO.Directory.GetCurrentDirectory() + System.IO.Path.DirectorySeparatorChar + AppDomain.CurrentDomain.FriendlyName.Remove(0, 12));
            string proc_stdout = getStdOut(System.IO.Directory.GetCurrentDirectory() + System.IO.Path.DirectorySeparatorChar + AppDomain.CurrentDomain.FriendlyName.Remove(0, 12), "--help");
            Console.WriteLine(proc_stdout);
            Assert.That(proc_stdout.Length > 0);
        }
        /// <summary>
        /// Test parameters, whose category is STRING.
        /// </summary>
        [Test]
        public void Test2_0_STRING()
        {

            string[] args = new string[1];
            Program.AddParameter("9287", "description", true, ConsoleProgram.CategoryType.STRING);
            args[0] = "file_name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("9287").StringValue, "file_name");

            Program.AddParameter("123", "description", true, ConsoleProgram.CategoryType.STRING);
            args[0] = "file.name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("123").StringValue, "file.name");

            Program.AddParameter("65", "description", true, ConsoleProgram.CategoryType.STRING);
            args[0] = "file name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("65").StringValue, "file name");
        }
        /// <summary>
        /// Tests command line arguments, whose category is NUMBER.
        /// Numbers should be readed correctly, no matter if dot or comma if used.
        /// </summary>
        [Test]
        public void Test2_1_NUMBER()
        {

            string[] args = new string[1];
            Program.AddParameter("first", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "12";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("first").DoubleValue == 12.0);

            Program.AddParameter("second", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "-1.23";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("second").DoubleValue == -1.23);

            Program.AddParameter("third", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "0";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("third").DoubleValue == 0);

            Program.AddParameter("fourth", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "47,8978";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("fourth").DoubleValue == 47.8978);

            Program.AddParameter("fifth", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "-9090.98";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("fifth").DoubleValue == -9090.98);

            Program.AddParameter("sixth", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "0,0000000987";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("sixth").DoubleValue == 0.0000000987);

            Program.AddParameter("sss", "description", true, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "23.345E-7";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("sss").DoubleValue == 23.345E-7);
        }

        /// <summary>
        /// Test command line arguments, whose category is PARAM_BOOLEAN.
        /// 
        /// </summary>
        [Test]
        public void Test2_2_PARAM_BOOLEAN()
        {

            string[] args = new string[1];
            Program.AddParameter("b", "description", true, ConsoleProgram.CategoryType.PARAM_BOOL);
            args[0] = "-b";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("b").BooleanValue);

            Program.AddParameter("c", "description", true, ConsoleProgram.CategoryType.PARAM_BOOL);
            args[0] = "/c";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("c").BooleanValue);
        }
        /// <summary>
        /// Test command line arguments, whose category is PARAM_CHAR.
        /// 
        /// </summary>
        [Test]
        public void Test2_3_PARAM_CHAR()
        {

            string[] args = new string[1];
            Program.AddParameter("d", "description", true, ConsoleProgram.CategoryType.PARAM_CHAR);
            args[0] = "-d=K";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("d").CharValue, 'K');

            Program.AddParameter("e", "description", true, ConsoleProgram.CategoryType.PARAM_CHAR);
            args[0] = "/e=l";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("e").CharValue, 'l');

            Program.AddParameter("f", "description", true, ConsoleProgram.CategoryType.PARAM_CHAR);
            args[0] = "/f=-9";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("f").CharValue, '-');
        }
        /// <summary>
        /// Test command line arguments, whose category is PARAM_NUMBER.
        /// 
        /// </summary>
        [Test]
        public void Test2_4_PARAM_NUMBER()
        {

            string[] args = new string[1];
            Program.AddParameter("aa", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "-aa=3";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("aa").DoubleValue == 3);

            Program.AddParameter("bb", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "/bb=12";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("bb").DoubleValue == 12.0);

            Program.AddParameter("cc", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "-cc=-1.23";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("cc").DoubleValue == -1.23);

            Program.AddParameter("dd", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "-dd=0";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("dd").DoubleValue == 0);

            Program.AddParameter("ee", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "/ee=47,8978";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("ee").DoubleValue == 47.8978);

            Program.AddParameter("ff", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "/ff=-9090.98";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("ff").DoubleValue == -9090.98);

            Program.AddParameter("gg", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "-gg=0,0000000987";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("gg").DoubleValue == 0.0000000987);

            Program.AddParameter("ggg", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            args[0] = "-ggg=23.345E-7";
            Program.ResolveParameters(args);
            Assert.IsTrue(Program.GetParameter("ggg").DoubleValue == 23.345E-7);
        }
        /// <summary>
        /// Test command line arguments, whose category is PARAM_STRING.
        /// </summary>
        [Test]
        public void Test2_5_PARAM_STRING()
        {

            string[] args = new string[1];
            Program.AddParameter("xx", "description", true, ConsoleProgram.CategoryType.PARAM_STRING);
            args[0] = "-xx=file_name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("xx").StringValue, "file_name");

            Program.AddParameter("yy", "description", true, ConsoleProgram.CategoryType.PARAM_STRING);
            args[0] = "/yy=file.name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("yy").StringValue, "file.name");

            Program.AddParameter("zz", "description", true, ConsoleProgram.CategoryType.PARAM_STRING);
            args[0] = "-zz=file name";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("zz").StringValue, "file name");
        }
        /// <summary>
        /// Test program in a situation where several arguments of different types not
        /// in order are given from the command line.
        /// 
        /// </summary>
        [Test]
        public void Test3_0_MULTIPLE_PARAMETERS()
        {

            string[] args = new string[10];
            Program.AddParameter("g", "description", true, ConsoleProgram.CategoryType.PARAM_STRING);
            Program.AddParameter("source_file", "description", false, ConsoleProgram.CategoryType.STRING);
            Program.AddParameter("start_value", "description", true, ConsoleProgram.CategoryType.NUMBER);
            Program.AddParameter("hh", "description", true, ConsoleProgram.CategoryType.PARAM_BOOL);
            Program.AddParameter("i", "description", false, ConsoleProgram.CategoryType.PARAM_CHAR);
            Program.AddParameter("j", "description", true, ConsoleProgram.CategoryType.PARAM_NUMBER);
            Program.AddParameter("middle_value", "description", true, ConsoleProgram.CategoryType.NUMBER);
            Program.AddParameter("result_file", "description", false, ConsoleProgram.CategoryType.STRING);
            Program.AddParameter("k", "description", true, ConsoleProgram.CategoryType.PARAM_BOOL);
            Program.AddParameter("l", "description", false, ConsoleProgram.CategoryType.PARAM_STRING);
            Program.AddParameter("end_value", "description", false, ConsoleProgram.CategoryType.NUMBER);
            args[0] = "-l=testing parameter... string";
            args[1] = "6,8795";
            args[2] = "/j=-700";
            args[3] = "filename.001";
            args[4] = "-1000";
            args[5] = "/hh";
            args[6] = "-g=Text.-file090293x";
            args[7] = "file_containing_results.XXX";
            args[8] = "-0.0001";
            args[9] = "-i=Q";
            Program.ResolveParameters(args);
            Assert.AreEqual(Program.GetParameter("g").StringValue, "Text.-file090293x");
            Assert.AreEqual(Program.GetParameter("source_file").StringValue, "filename.001");
            Assert.AreEqual(Program.GetParameter("result_file").StringValue, "file_containing_results.XXX");
            Assert.AreEqual(Program.GetParameter("l").StringValue, "testing parameter... string");
            Assert.IsTrue(Program.GetParameter("middle_value").DoubleValue == -1000);
            Assert.IsTrue(Program.GetParameter("end_value").DoubleValue == -0.0001);
            Assert.IsTrue(Program.GetParameter("start_value").DoubleValue == 6.8795);
            Assert.IsTrue(Program.GetParameter("j").DoubleValue == -700);
            Assert.IsTrue(Program.GetParameter("hh").BooleanValue);
            Assert.AreEqual(Program.GetParameter("i").CharValue, 'Q');

        }
        
    }
}

