/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;
using System.Runtime.InteropServices.ComTypes;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib
{
    /// <summary>
    /// Radioactivity unit
    /// </summary>
    public enum Activity_unit {
        /// <summary>
        /// Unknown unit
        /// </summary>
        Unknown = 0,
        /// <summary>
        /// Bequerel
        /// </summary>
        Bq = 1,
        /// <summary>
        /// Kilobequerel
        /// </summary>
        kBq = 1000,
        /// <summary>
        /// Megabequerel
        /// </summary>
        MBq = 1000000
    }
    /// <summary>
    /// Data unit type
    /// </summary>
    public enum Data_unit
    {
        /// <summary>
        /// Becquerels per millilitre
        /// </summary>
        BQ_per_ML,
        /// <summary>
        /// Becquerels per cubic centimeter
        /// </summary>
        BQ_per_CC,
        /// <summary>
        /// KiloBecquerels per millilitre
        /// </summary>
        KBQ_per_ML,
        /// <summary>
        /// Counts 
        /// </summary>
        COUNTS,
        /// <summary>
        /// Seconds * KiloBecquerels / millilitre
        /// </summary>
        SECtimesKBQ_per_ML,
        /// <summary>
        /// 1 / Seconds
        /// </summary>
        ONE_per_SEC,
        /// <summary>
        /// 1 / Minutes
        /// </summary>
        ONE_per_MIN,
        /// <summary>
        /// Millilitres / Millilitres
        /// </summary>
        ML_per_ML,
        /// <summary>
        /// Millilitres / Desilitres
        /// </summary>
        ML_per_DL,
        /// <summary>
        /// Millilitres / (Millilitres * Minutes)
        /// </summary>
        ML_per_MLtimesMIN,
        /// <summary>
        /// Millilitres / (Desilitres * Minutes)
        /// </summary>
        ML_per_DLtimesMIN,
        /// <summary>
        /// Number of counts / Millilitres
        /// </summary>
        NCI_per_ML,
        /// <summary>
        /// MegaBecquerels / Millilitres
        /// </summary>
        MBQ_per_ML,
        /// <summary>
        /// Moles / (100Grams * Minutes)
        /// </summary>
        UMOL_per_100GtimesMIN,
        /// <summary>
        /// Milligrams / (100Grams * Minutes)
        /// </summary>
        MG_per_100GtimesMIN,
        /// <summary>
        /// Unknown unit
        /// </summary>
        Unknown,
        /// <summary>
        /// Unknown calibrated unit
        /// </summary>
        Unknown_Calibrated,
        /// <summary>
        /// Unit undefined
        /// </summary>
        Undefined,
        /// <summary>
        /// Micromoles in millilitre
        /// </summary>
        UMOL_per_ML,
        /// <summary>
        /// Millimoles in litre
        /// </summary>
        MMOL_per_L
    }
    /// <summary>
    /// Class representing data unit. Only static predefined instances can be used. 
    /// Use array <code>units</code> if new unit is added to class.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public sealed class UnitConverter
    {
        /// <summary>
        /// List of all available units. This is the only place where unit parameters 
        /// are defined. List is ordered by unit names so that longer names are first. 
        /// </summary>
        private static string[] unit_names = new string[] { 

        };
        /// <summary>
        /// Constructs radioactivity unit from unit's name. 'Undefined' is used if 
        /// no proper match was found.
        /// </summary>
        /// <param name="name1">unit enumerator</param>
        public static Activity_unit CreateActivityUnit(string name)
        {
            name = name.Trim().ToLowerInvariant();
            //convert synonyms
            if (name.Equals("mbq") || name.Equals("mega-bequerel") || name.Equals("mega-bq")) return Activity_unit.MBq;
            if (name.Equals("kbq") || name.Equals("milo-bequerel") || name.Equals("kilo-bq")) return Activity_unit.kBq;
            if (name.Equals("bq") || name.Equals("bequerel")) return Activity_unit.Bq;
            return Activity_unit.Unknown;
        }
        /// <summary>
        /// Constructs unit from unit's name. 'Undefined' is used if 
        /// no proper match was found.
        /// </summary>
        /// <param name="name1">isotope name, for example 'Bq/ml'</param>
        public static Patient_orientation CreatePatientOrientation(string name)
        {
           
            name = name.Trim().ToLowerInvariant();
            //convert synonyms
            if (name.Equals("ffdl")) return Patient_orientation.Feet_first_Decubitus_Left;
            if (name.Equals("ffdr")) return Patient_orientation.Feet_first_Decubitus_Right;
            if (name.Equals("ffp")) return Patient_orientation.Feet_first_Prone;
            if (name.Equals("ffs")) return Patient_orientation.Feet_first_Supine;
            if (name.Equals("hfdl")) return Patient_orientation.Head_first_Decubitus_Left;
            if (name.Equals("hfdr")) return Patient_orientation.Head_first_Decubitus_Right;
            if (name.Equals("hfp")) return Patient_orientation.Head_first_Prone;
            if (name.Equals("hfs")) return Patient_orientation.Head_first_Supine;
            return Patient_orientation.Unknown;
        }
        /// <summary>
        /// Constructs unit from unit's name. 'Undefined' is used if 
        /// no proper match was found.
        /// </summary>
        /// <param name="name1">isotope name, for example 'Bq/ml'</param>
        public static Data_unit CreateDataUnit(string name)
        {
            name = name.Trim().ToLowerInvariant();
            //convert synonyms
            if(name.Equals("unknown_calibrated")) return Data_unit.Unknown_Calibrated;
            if(name.Equals("umol/(100g*min)")) return Data_unit.UMOL_per_100GtimesMIN;
            if(name.Equals("mg/(100g*min)")) return Data_unit.MG_per_100GtimesMIN;
            if(name.Equals("ml/(ml*min)")) return Data_unit.ML_per_MLtimesMIN;
            if(name.Equals("ml/(dl*min)")) return Data_unit.ML_per_DLtimesMIN;
            if(name.Equals("sec*kBq/ml")) return Data_unit.SECtimesKBQ_per_ML;
            if (name.Equals("mbq/ml") || name.Equals("mbqml")) return Data_unit.MBQ_per_ML;
            if (name.Equals("kbq/ml") || name.Equals("kbqml")) return Data_unit.KBQ_per_ML;
            if(name.Equals("nci/ml")) return Data_unit.NCI_per_ML;
            if(name.Equals("mmol/l")) return Data_unit.MMOL_per_L;
            if(name.Equals("bq/ml") || name.Equals("bqml")) return Data_unit.BQ_per_ML;
            if(name.Equals("bq/cc") || name.Equals("bqcc")) return Data_unit.BQ_per_CC;
            if(name.Equals("1/sec")) return Data_unit.ONE_per_SEC;
            if(name.Equals("1/min")) return Data_unit.ONE_per_MIN;
            if(name.Equals("ml/ml")) return Data_unit.ML_per_ML;
            if(name.Equals("ml/dl")) return Data_unit.ML_per_DL;
            if(name.Equals("um/ml")) return Data_unit.UMOL_per_ML;
            if (name.Contains("counts")) return Data_unit.COUNTS;
            if (name.Contains("unknown")) return Data_unit.Unknown;
            if (name.Contains("undefined")) return Data_unit.Undefined;
            return Data_unit.Undefined;
        }
        /// <summary>
        /// Constructs string from unit. 
        /// </summary>
        /// <param name="name">isotope</param>
        /// <returns>isotope as a string</returns>
        public static string ConvertToString(Isotope_enumerator isotope)
        {
            return Isotope.ConvertToString(isotope);
        }
        /// <summary>
        /// Constructs string from unit. 
        /// </summary>
        /// <param name="name">data unit</param>
        /// <returns>data unit as a string</returns>
        public static string ConvertToString(Data_unit unit)
        {
            switch(unit) {
                case Data_unit.Unknown_Calibrated: return "unknown_calibrated";
                case Data_unit.UMOL_per_100GtimesMIN: return "umol/(100g*min)";
                case Data_unit.MG_per_100GtimesMIN: return "mg/(100g*min)";
                case Data_unit.Undefined: return "<undefined>";
                case Data_unit.ML_per_MLtimesMIN: return "ml/(ml*min)";
                case Data_unit.ML_per_DLtimesMIN: return "ml/(dl*min)";
                case Data_unit.SECtimesKBQ_per_ML: return "sec*kBq/ml";
                case Data_unit.Unknown: return "unknown";
                case Data_unit.COUNTS: return "counts";
                case Data_unit.MBQ_per_ML: return "mbq/ml";
                case Data_unit.KBQ_per_ML: return "kbq/ml";
                case Data_unit.NCI_per_ML: return "nci/ml";
                case Data_unit.MMOL_per_L: return "mmol/l";
                case Data_unit.BQ_per_ML: return "bq/ml";
                case Data_unit.ONE_per_SEC: return "1/sec";
                case Data_unit.ONE_per_MIN: return "1/min";
                case Data_unit.ML_per_ML: return "ml/ml";
                case Data_unit.ML_per_DL: return "ml/dl";
                case Data_unit.UMOL_per_ML: return "um/ml";
                default:
                    return unit.ToString();
            }
        }
        /// <summary>
        /// Gets conversion factor from unit to another.
        /// </summary>
        /// <param name="source">source unit</param>
        /// <param name="target">target unit</param>
        /// <returns>multiplication value to convert to target unit</returns>
        public static double GetConversionFactor(Activity_unit source, Activity_unit target)
        {
            if(source == target) return 1.0;
            switch (source) {
                case Activity_unit.Bq:
                    switch (target) {
                        case Activity_unit.kBq: return 0.001;
                        case Activity_unit.MBq: return 0.000001;
                    }
                    break;
                case Activity_unit.kBq:
                    switch (target)
                    {
                        case Activity_unit.Bq: return 1000.0;
                        case Activity_unit.MBq: return 0.001;
                    }
                    break;
                case Activity_unit.MBq:
                    switch (target)
                    {
                        case Activity_unit.Bq: return 1000000.0;
                        case Activity_unit.kBq: return 1000.0;
                    }
                    break;
            }
            throw new Exception("Internal error: unit "+source.ToString()+" is not converted");
        }
        /// <summary>
        /// Gets conversion factor from unit to another.
        /// </summary>
        /// <param name="source">source unit</param>
        /// <param name="target">target unit</param>
        /// <returns>multiplication value to convert to target unit</returns>
        /// <exception cref="TPCInvalidArgumentsException">if conversion is not valid</exception>
        public static double GetConversionFactor(Data_unit source, Data_unit target)
        {
            if (!IsValidConversion(source, target))
                throw new TPCInvalidArgumentsException("Cannot convert unit from "+source+" to "+target);
            #region switch_clause_through_all_possibilities
            switch (source)
            {
                
                case Data_unit.KBQ_per_ML:
                    switch (target)
                    {
                        case Data_unit.BQ_per_CC: return 1000.0;
                        case Data_unit.BQ_per_ML: return 1000.0;
                        case Data_unit.MBQ_per_ML: return 0.001;
                    }
                    break;
                case Data_unit.MBQ_per_ML:
                    switch (target)
                    {
                        case Data_unit.BQ_per_CC: return 1000000.0;
                        case Data_unit.BQ_per_ML: return 1000000.0;
                        case Data_unit.KBQ_per_ML: return 0.001;
                    }
                    break;
                case Data_unit.BQ_per_CC:
                    switch (target)
                    {
                        case Data_unit.KBQ_per_ML: return 0.001;
                        case Data_unit.MBQ_per_ML: return 0.000001;
                    }
                    break;
                case Data_unit.BQ_per_ML:
                    switch (target)
                    {
                        case Data_unit.KBQ_per_ML: return 0.001;
                        case Data_unit.MBQ_per_ML: return 0.000001;
                    }
                    break;
                case Data_unit.ML_per_DLtimesMIN:
                    switch (target)
                    {
                        case Data_unit.ML_per_MLtimesMIN: return 0.01;
                    }
                    break;
                case Data_unit.ML_per_MLtimesMIN:
                    switch (target)
                    {
                        case Data_unit.ML_per_DLtimesMIN: return 100.0;
                    }
                    break;
                case Data_unit.ML_per_DL:
                    switch (target)
                    {
                        case Data_unit.ML_per_ML: return 0.01;
                    }
                    break;
                case Data_unit.ML_per_ML:
                    switch (target)
                    {
                        case Data_unit.ML_per_DL: return 100.0;
                    }
                    break;
                case Data_unit.ONE_per_MIN:
                    switch (target)
                    {
                        case Data_unit.ONE_per_SEC: return 60.0;
                    }
                    break;
                case Data_unit.ONE_per_SEC:
                    switch (target)
                    {
                        case Data_unit.ONE_per_MIN: return 1.0 / 60.0;
                    }
                    break;
                case Data_unit.UMOL_per_ML:
                    switch (target)
                    {
                        case Data_unit.MMOL_per_L: return 0.000001;
                    }
                    break;
                case Data_unit.MMOL_per_L:
                    switch (target)
                    {
                        case Data_unit.UMOL_per_ML: return 1000000.0;
                    }
                    break;
                default:
                    throw new TPCException("Unsupported conversion "+source.ToString()+" to "+target.ToString());
            }
            #endregion
            throw new TPCException("Internal error, should not be here.");
        }
        /// <summary>
        /// Returns names of all available units.
        /// </summary>
        /// <returns>array having unit names</returns>
        public static string[] GetAllUnitnames() {
            return Enum.GetNames(typeof(Data_unit));
        }
        /// <summary>
        /// Tests whether conversion is valid from unit to another.
        /// </summary>
        /// <param name="source">source unit</param>
        /// <param name="target">target unit</param>
        /// <returns>true if source unit can be converter to target unit</returns>
        public static bool IsValidConversion(Data_unit source, Data_unit target)
        {
            if (source == target) return true;
            if (source == Data_unit.Unknown) return false;
            if (source == Data_unit.Undefined) return false;
            if (source == Data_unit.Unknown_Calibrated) return false;
            if (target == Data_unit.Unknown) return false;
            if (target == Data_unit.Undefined) return false;
            if (target == Data_unit.Unknown_Calibrated) return false;
            #region switch_clause_through_all_possibilities
            switch (source) {
                case Data_unit.KBQ_per_ML:
                case Data_unit.MBQ_per_ML:
                case Data_unit.BQ_per_ML:
                    switch (target) {
                        case Data_unit.BQ_per_CC: 
                        case Data_unit.BQ_per_ML: 
                        case Data_unit.KBQ_per_ML: 
                        case Data_unit.MBQ_per_ML: 
                            return true;
                        default: return false;
                    }
                case Data_unit.BQ_per_CC:
                    switch (target)
                    {
                        case Data_unit.BQ_per_ML:
                        case Data_unit.KBQ_per_ML:
                        case Data_unit.MBQ_per_ML:
                            return true;
                        default: return false;
                    }
                case Data_unit.ML_per_DLtimesMIN:
                case Data_unit.ML_per_MLtimesMIN:
                    switch (target)
                    {
                        case Data_unit.ML_per_DLtimesMIN:
                        case Data_unit.ML_per_MLtimesMIN:
                            return true;
                        default: return false;
                    }
                case Data_unit.ML_per_DL:
                case Data_unit.ML_per_ML:
                    switch (target)
                    {
                        case Data_unit.ML_per_DL:
                        case Data_unit.ML_per_ML:
                            return true;
                        default: return false;
                    }
                case Data_unit.ONE_per_MIN:
                case Data_unit.ONE_per_SEC:
                    switch (target)
                    {
                        case Data_unit.ONE_per_MIN:
                        case Data_unit.ONE_per_SEC:
                            return true;
                        default: return false;
                    }
                case Data_unit.UMOL_per_ML:
                case Data_unit.MMOL_per_L:
                    switch (target)
                    {
                        case Data_unit.UMOL_per_ML:
                        case Data_unit.MMOL_per_L:
                            return true;
                        default: return false;
                    }
                default:
                    return false;
            }
            #endregion
        }
    }
}
