/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Text;
using System.Runtime.InteropServices.ComTypes;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib {
	
    /// <summary>
    /// Vector class. Internal representation is double[], and should be generally
    /// interchangeable with it (implicit casting is provided).
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public class Vector
    {
        /// <summary>
        /// Vector values
        /// </summary>
        private double[] val;

        /// <summary>
        /// Number of scalar components.
        /// </summary>
        public readonly int Dim;

        /// <summary>
        /// Number of elements in this Vector (readonly)
        /// </summary>
        /// <returns>number of elements</returns>
        public int Length
        {
            get { return Dim; }
        }

        /// <summary>
        /// Constructor; vector of n-dimensions
        /// </summary>
        /// <param name="dims">Dimension of the new vector</param>
        public Vector(int dims)
        {
            val = new double[dims];
            this.Dim = dims;
        }

        /// <summary>
        /// Constructor; initialize with values from a
        /// </summary>
        /// <param name="a">Array of vector elements</param>
        public Vector(int[] a)
        {
            this.val = new double[a.Length];
            this.Dim = a.Length;
            for (int i = 0; i < this.Dim; i++)
            {
                val[i] = a[i];
            }
        }

        /// <summary>
        /// Constructor; initialize with values from a
        /// </summary>
        /// <param name="a">Array of vector elements</param>
        public Vector(params double[] a)
        {
            this.val = new double[a.Length];
            this.Dim = a.Length;
            for (int i = 0; i < this.Dim; i++)
            {
                val[i] = a[i];
            }
        }

        /// <summary>
        /// Concatenates a value into end of vector. Note that this method might be slow 
        /// because of reallocation of underlying double array.
        /// </summary>
        /// <param name="d">value to be added</param>
        public static Vector Concat(Vector v, double d) {
            double[] values = v.val;
            int val_length = values.Length;

            Array.Resize<double>(ref values, val_length + 1);
            values[values.Length - 1] = d;
            return new Vector(values);
        }

        /// <summary>
        /// Vector addition a+b
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Sum vector a+b</returns>
        public static Vector operator +(Vector a, Vector b)
        {
            if (a.Dim != b.Dim) throw new ArgumentException("Vector dimensions don't match");

            Vector res = new Vector(a.Dim);
            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = a[i] + b[i];
            }
            return res;
        }

        /// <summary>
        /// Vector subtraction a-b
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Difference vector a-b</returns>
        public static Vector operator -(Vector a, Vector b)
        {
            if (a.Dim != b.Dim)
                throw new ArgumentException("Vector dimensions don't match");

            Vector res = new Vector(a.Dim);
            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = a[i] - b[i];
            }
            return res;
        }

        /// <summary>
        /// Scalar product c*a
        /// </summary>
        /// <param name="c">Scalar c</param>
        /// <param name="a">Vector a</param>
        /// <returns>Result vector c*a</returns>
        public static Vector operator *(double c, Vector a)
        {
            return a * c;
        }

        /// <summary>
        /// Scalar product a*c
        /// </summary>
        /// <param name="c">Scalar c</param>
        /// <param name="a">Vector a</param>
        /// <returns>Result vector a*c</returns>
        public static Vector operator *(Vector a, double c)
        {
            Vector res = new Vector(a.Dim);
            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = c * a[i];
            }
            return res;
        }

        /// <summary>
        /// Division by scalar a/c
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="c">Scalar c</param>
        /// <returns>Result vector a/c</returns>
        public static Vector operator /(Vector a, double c)
        {
            if (c == 0) throw new DivideByZeroException();
            Vector res = new Vector(a.Dim);
            for (int i = 0; i < res.Dim; i++)
            {
                res[i] = a[i] / c;
            }
            return res;
        }

        /// <summary>
        /// Negate vector
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <returns>Vector -a</returns>
        public static Vector operator -(Vector a)
        {
            return (-1 * a);
        }

        /// <summary>
        /// Implicit cast to double[] array
        /// </summary>
        /// <param name="v">Vector to cast</param>
        /// <returns>Array holding the vector elements</returns>
        public static implicit operator double[](Vector v)
        {
            return v.val;
        }
        /// <summary>
        /// Cast from double value
        /// </summary>
        /// <param name="d">value that is casted</param>
        /// <returns>double value as Vector object</returns>
        public static implicit operator Vector(double d)
        {
            return new Vector(d);
        }

        /// <summary>
        /// Implicit cast of double array to Vector
        /// </summary>
        /// <param name="arr">Array to cast</param>
        /// <returns>Vector holding the array elements</returns>
        public static implicit operator Vector(double[] arr)
        {
            return new Vector(arr);
        }

        /// <summary>
        /// Implicit cast of int array to Vector
        /// </summary>
        /// <param name="arr">Array to cast</param>
        /// <returns>Vector holding the array elements</returns>
        public static implicit operator Vector(int[] arr)
        {
            return new Vector(arr);
        }

        /// <summary>
        /// Equivalence operator
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>True, if every element in a is equivalent with corresponding element in b</returns>
        public static bool operator ==(Vector a, Vector b)
        {
            if (a.Dim == b.Dim)
            {
                for (int i = 0; i < a.Dim; i++)
                {
                    if (a[i] != b[i]) return false;
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Inequivalence operator
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>True, if at least one element in a is not equivalent with corresponding element in b</returns>
        public static bool operator !=(Vector a, Vector b)
        {
            return !(a == b);
        }

        /// <summary>
        /// Dot (or inner) product of vectors
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Inner product ab</returns>
        public static double Dot(Vector a, Vector b)
        {
            if (a.Dim != b.Dim) throw new ArgumentException("Vector dimensions don't match");

            double res = 0.0;
            for (int i = 0; i < a.Dim; i++)
            {
                res += a[i] * b[i];
            }
            return res;
        }

        /// <summary>
        /// Equals method
        /// </summary>
        /// <param name="obj">Object to compare this Vector to</param>
        /// <returns>True, if obj is equivalent with this Vector</returns>
        public override bool Equals(object obj)
        {
            if (obj.GetType() == this.GetType())
            {
                return this == (Vector)obj;
            }
            return false;
        }

        /// <summary>
        /// Hash code of this vector
        /// </summary>
        /// <returns>Hash code</returns>
        public override int GetHashCode()
        {
            int hash = 1;
            for (int i = 0; i < Dim; i++)
            {
                hash *= this[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Array copy of this Vector
        /// </summary>
        /// <returns>Array holding copies of this vectors elements</returns>
        public double[] ToArray()
        {
            double[] copy = new double[this.Dim];
            Array.Copy(this.val, copy, this.Dim);
            return copy;
        }

        /// <summary>
        /// Index operator
        /// </summary>
        /// <param name="i">Index</param>
        /// <returns>i:th element of this Vector</returns>
        public double this[int i]
        {
            get { return val[i]; }
            set { val[i] = value; }
        }

        /// <summary>
        /// Vector of dimension 'n' with all elements equal to 'c'
        /// </summary>
        /// <param name="n">Dimension</param>
        /// <param name="c">Constant value c</param>
        /// <returns>Vector (c,c,...,c) of n elements</returns>
        public static Vector Fill(int n, double c)
        {
            double[] val = new double[n];
            for (int i = 0; i < n; i++)
            {
                val[i] = c;
            }
            return new Vector(val);
        }

        /// <summary>
        /// Sum of several Vectors
        /// </summary>
        /// <param name="varr">Array of Vectors</param>
        /// <returns>Sum of all Vectors in 'varr'</returns>
        public static Vector Sum(params Vector[] varr)
        {
            Vector sum = varr[0];

            for (int i = 1; i < varr.Length; i++)
            {
                sum += varr[i];
            }

            return sum;
        }

        /// <summary>
        /// Unit vector of n-dimensions, axis specifies
        /// the index of the non-zero element
        /// </summary>
        /// <param name="dim">Dimension</param>
        /// <param name="axis">Index of the non-zero element</param>
        /// <returns>Unit Vector</returns>
        public static Vector Unit(int dim, int axis)
        {
            Vector v = new Vector(dim);
            v[axis] = 1.0;
            return v;
        }

        /// <summary>
        /// Multiplication by elements
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Result vector (a[0]*b[0], a[1]*b[1], ... , a[n]*b[n])</returns>
        public static Vector operator *(Vector a, Vector b)
        {
            int dim = a.Length;
            if (b.Length != dim) throw new ArgumentException("Vector dimensions don't match");

            Vector c = new Vector(dim);
            for (int i = 0; i < dim; i++)
            {
                c[i] = a[i] * b[i];
            }
            return c;
        }

        /// <summary>
        /// Division by elements
        /// </summary>
        /// <param name="a">Vector a</param>
        /// <param name="b">Vector b</param>
        /// <returns>Result vector (a[0]/b[0], a[1]/b[1], ... , a[n]/b[n])</returns>
        public static Vector operator /(Vector a, Vector b)
        {
            int dim = a.Length;
            if (b.Length != dim) throw new ArgumentException("Vector dimensions don't match");

            Vector c = new Vector(dim);
            for (int i = 0; i < dim; i++)
            {
                c[i] = a[i] / b[i];
            }

            return c;
        }

        /// <summary>
        /// Get the string representation of this Vector
        /// </summary>
        /// <returns>String reperesentation of this Vector</returns>
        public string MakeString()
        {
            StringBuilder sb = new StringBuilder();
            sb.Append("[");
            foreach (double s in val)
                sb.Append(s.ToString("F4") + " ");
            sb.Append("]");
            return sb.ToString();
        }

        /// <summary>
        /// Prints contents of this vector as string
        /// </summary>
        /// <param name="writer">writer where data is is written</param>
        public void PrintAsString(System.IO.TextWriter writer)
        {
            writer.Write("[");
            if (val.Length > 0)
                writer.Write(val[0]);
            for (int i = 1; i < val.Length; i++)
            {
                writer.Write(" " + val[i]);
            }
            writer.Write("]\n");
        }

        /// <summary>
        /// String representation
        /// </summary>
        /// <returns>String representation of this vector</returns>
        public override string ToString()
        {
            return ToString("");
        }

        /// <summary>
        /// String representation with formatted numeric output
        /// </summary>
        /// <param name="format">Format string; all format strings applicable to double are accepted</param>
        /// <returns>String representation of this vector</returns>
        public string ToString(string format)
        {
            int n = this.Length;

            if (n < 1)
                return "[empty]";

            string[] strs = new string[n];
            int maxlen = 0;

            for (int i = 0; i < n; i++)
            {
                strs[i] = this[i].ToString(format);
                if (strs[i].Length > maxlen)
                {
                    maxlen = strs[i].Length;
                }
            }

            StringBuilder sb = new StringBuilder("[");

            for (int i = 0; i < n; i++)
            {
                sb.Append(strs[i].PadLeft(maxlen) + "");
            }

            return sb.ToString().Trim() + "]";
        }

        /// <summary>
        /// Euclidean norm |v|
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Norm of Vector v</returns>
        public static double Norm(Vector v)
        {
            return Math.Sqrt(Vector.SqNorm(v));
        }

        /// <summary>
        /// Square of the euclidean norm |v|*|v|
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Square of the norm of Vector v</returns>
        public static double SqNorm(Vector v)
        {
            return Vector.Dot(v, v);
        }

        /// <summary>
        /// Smallest value in Vector v
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Minimal element in v</returns>
        public static double Min(Vector v)
        {
            double min = v[0];

            for (int i = 1; i < v.Dim; i++)
            {
                if (v[i] < min) min = v[i];
            }

            return min;
        }

        /// <summary>
        /// Largest value in v
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Maximal element in v</returns>
        public static double Max(Vector v)
        {
            double max = v[0];

            for (int i = 1; i < v.Dim; i++)
            {
                if (v[i] > max) max = v[i];
            }

            return max;
        }

        /// <summary>
        /// Index of the largest value in v
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Maximal element in v</returns>
        public static int MaxIndex(Vector v)
        {
            double max = v[0];
            int maxindex = 0;

            for (int i = 1; i < v.Dim; i++)
            {
                if (v[i] > max)
                {
                    max = v[i];
                    maxindex = i;
                }
            }

            return maxindex;
        }

        /// <summary>
        /// Index of the smallest value in v
        /// </summary>
        /// <param name="v">Vector v</param>
        /// <returns>Index of the maximal element in v</returns>
        public static int MinIndex(Vector v)
        {
            double min = v[0];
            int minindex = 0;

            for (int i = 1; i < v.Dim; i++)
            {
                if (v[i] > min)
                {
                    min = v[i];
                    minindex = i;
                }
            }

            return minindex;
        }

        /// <summary>
        /// Gets a range of elements from vector
        /// </summary>
        /// <param name="v">Source vector</param>
        /// <param name="s">Start index</param>
        /// <param name="r">Range length</param>
        /// <returns>Vector with elements {v[s],v[s+1],...,v[e]}</returns>
        public static Vector Range(Vector v, int s, int r)
        {
            if ((s < 0) || (r < 0) || (r + s > v.Length)) throw new ArgumentException("Index out of bounds");

            Vector res = new Vector(r);

            for (int i = 0; i < r; i++)
            {
                res[i] = v[s + i];
            }

            return res;
        }

        /// <summary>
        /// Generates 'n' random vectors of 'dim' dimensions (uniform distribution),
        /// with upper and lower limits for each dimension
        /// </summary>
        /// <param name="n">Number of vectors to generate</param>
        /// <param name="dim">Dimension of generated vectors</param>
        /// <param name="par_high">Upper limit for all dimensions</param>
        /// <param name="par_low">Lower limit for all dimensions</param>
        /// <returns>Array of random vectors</returns>
        public static Vector[] RandomPoints(int n, int dim, double par_high, double par_low)
        {
            return RandomPoints(n, Vector.Fill(dim, par_high), Vector.Fill(dim, par_low));
        }

        /// <summary>
        /// Generates 'n' random vectors (uniform distribution),
        /// with upper and lower limits in each dimension in par_high and par_low
        /// </summary>
        /// <param name="n">Number of vectors to generate</param>
        /// <param name="par_high">Upper limits for each dimension</param>
        /// <param name="par_low">Lower limits for each dimension</param>
        /// <returns>Array of random vectors</returns>
        public static Vector[] RandomPoints(int n, Vector par_high, Vector par_low)
        {
            Vector[] vectors = new Vector[n];

            for (int i = 0; i < n; i++)
            {
                vectors[i] = RandomPoint(par_high, par_low, i);
            }

            return vectors;
        }

        /// <summary>
        /// Generates a random vector (uniform distribution),
        /// with upper and lower limits defined in 'par_high' and 'par_low'
        /// </summary>
        /// <param name="par_high">Upper limits</param>
        /// <param name="par_low">Lower limits</param>
        /// <param name="seed">seed number for random number generator</param>
        /// <returns>A random vector</returns>
        public static Vector RandomPoint(Vector par_high, Vector par_low, int seed)
        {
            if (par_high.Dim != par_low.Dim)
                throw new ArgumentException("Limit vector dimensions don't match");

            Random rand = new Random(seed);
            Vector v = new Vector(par_high.Dim);
            for (int i = 0; i < v.Dim; i++)
            {
                if (par_high[i] < par_low[i])
                    throw new ArgumentException("The Low limit was larger than the high limit.");
                v[i] = rand.NextDouble() * (par_high[i] - par_low[i]) + par_low[i];
            }
            return v;
        }
    }
}
