/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices.ComTypes;
using System.Runtime.InteropServices;
using TPClib.Interfaces.Matlab;

namespace TPClib
{
    /// <summary>
    /// Represents voxel in 3D-space that has a volume.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ievent))]
    public class Voxel
    {
        /// <summary>
        /// Holds x-coordinate dimension
        /// </summary>
        public float sizex { 
            get { return (float)size.x; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for x-dimension."); size.x = value; }
        }
        /// <summary>
        /// Holds y-coordinate dimension
        /// </summary>
        public float sizey { 
            get { return (float)size.y; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for y-dimension."); size.y = value; }
        }
        /// <summary>
        /// Holds z-coordinate dimension
        /// </summary>
        public float sizez { 
            get { return (float)size.z; }
            set { if (value < 0) throw new TPCException("Tried to set negative size for z-dimension."); size.z = value; }
        }
        /// <summary>
        /// Holds volume of voxel in units
        /// </summary>
        public float volume { get { return (float)(size.x * size.y * size.z); } }
        /// <summary>
        /// Point holding voxel dimension information.
        /// </summary>
        private Point size = new Point();
        /// <summary>
        /// Constructs voxel from another voxel
        /// </summary>
        /// <param name="v">voxel</param>
        public Voxel(Voxel v)
        {
            size = new Point(v.sizex, v.sizey, v.sizez);
        }
        /// <summary>
        /// Constructs voxel with desired size
        /// </summary>
        /// <param name="sizex">x-coord size</param>
        /// <param name="sizey">y-coord size</param>
        /// <param name="sizez">z-coord size</param>
        public Voxel(float sizex, float sizey, float sizez)
        {
            if (sizex <= 0) throw new TPCException("Tried to create Voxel with x-dimension that is not above zero:" + sizex);
            if (sizey <= 0) throw new TPCException("Tried to create Voxel with y-dimension that is not above zero:" + sizey);
            if (sizez <= 0) throw new TPCException("Tried to create Voxel with z-dimension that is not above zero:" + sizez);
            size = new Point(sizex, sizey, sizez);
        }
        /// <summary>
        /// Constructs voxel with desired size
        /// </summary>
        /// <param name="sizex">x-coord size</param>
        /// <param name="sizey">y-coord size</param>
        /// <param name="sizez">z-coord size</param>
        public Voxel(double sizex, double sizey, double sizez)
            : this((float)sizex, (float)sizey, (float)sizez)
        {
        }
        /// <summary>
        /// Constructs cubic voxel of size 1.0 x 1.0 x 1.0 units
        /// </summary>
        public Voxel()
        {
            size = new Point(1.0, 1.0, 1.0);           
        }
        /// <summary>
        /// Compares voxel equality
        /// </summary>
        /// <param name="v1">1st voxel</param>
        /// <param name="v2">2nd voxel</param>
        /// <returns>true if voxel dimensions are equal</returns>
        public static bool operator ==(Voxel v1, Voxel v2) {
            return !(v1 != v2);
        }
        /// <summary>
        /// Compares voxels inequality
        /// </summary>
        /// <param name="v1">1st tested voxel</param>
        /// <param name="v2">2nd tested voxel</param>
        /// <returns>true if voxel sizes are ineqault at any axis</returns>
        public static bool operator !=(Voxel v1, Voxel v2)
        {
            if (v1.sizex != v2.sizex) return true;
            if (v1.sizey != v2.sizey) return true;
            if (v1.sizez != v2.sizez) return true;
            return false;
        }
        /// <summary>
        /// Multiplication operator
        /// </summary>
        /// <param name="p1">1st voxel size that is multiplied</param>
        /// <param name="d">multiplicator</param>
        public static Voxel operator *(Voxel p1, double d)
        {
            Voxel r = new Voxel();
            r.sizex = (float)(p1.sizex * d);
            r.sizey = (float)(p1.sizey * d);
            r.sizez = (float)(p1.sizez * d);
            return r;
        }
        /// <summary>
        /// Compares two objects
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if object is voxel having equal dimensions</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is Voxel)) return false;
            return (sizex == (obj as Voxel).sizex) && (sizey == (obj as Voxel).sizey) && (sizez == (obj as Voxel).sizez);
        }
        /// <summary>
        /// Returns voxel as a string
        /// </summary>
        /// <returns>string representation</returns>
        public override string ToString()
        {
            return "Voxel[" + sizex.ToString("G").Replace(',', '.') + " x " +
                              sizey.ToString("G").Replace(',', '.') + " x " +
                              sizez.ToString("G").Replace(',', '.') + "]";
        }
        /// <summary>
        /// Operators != and == require implementation. Just calls 
        /// parent's method.
        /// </summary>
        /// <returns>hashcode for this object</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        /// <summary>
        /// Gets array representation of voxel dimensions.
        /// </summary>
        /// <returns>array holding voxel size in x,y,z order</returns>
        public double[] ToArray() {
            return new double[] { size.x, size.y, size.z };
        }
        /// <summary>
        /// Implicit casting to 3D point object
        /// </summary>
        /// <param name="v">input voxel</param>
        /// <returns>point holding voxel size as its coordinates</returns>
        public static implicit operator Point(Voxel v) {
            return new Point(v.sizex, v.sizey, v.sizez);
        }
    }
}
