/*
   
    openDICOM.NET openDICOM# 0.1.1

    openDICOM# provides a library for DICOM related development on Mono.
    Copyright (C) 2006-2007  Albert Gnandt

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA


    $Id: DataElement.cs 48 2007-03-28 13:49:15Z agnandt $
*/
using System;
using System.IO;
using openDicom.Registry;
using openDicom.DataStructure;
using openDicom.Encoding;

namespace openDicom.DataStructure.DataSet
{

    /// <summary>
    ///     This class represents a DICOM data element.
    /// </summary>
    public sealed class DataElement: IComparable, IDicomStreamMember
    {
        /// <summary>
        /// Parent object in nested tag hierarchy
        /// </summary>
        public object parent = null;
        /// <summary>
        /// Gets parent object in nested tag hierarchy
        /// </summary>
        public object getParent()
        {
            return parent;
        }

        private Tag tag = null;
        /// <summary>
        ///     Data element tag.
        /// </summary>
        public Tag Tag
        {
            get 
            {
                if (tag != null) 
                    return tag; 
                else
                    throw new DicomException("DataElement.Tag is null.", 
                        (Tag) null);
            }
        }

        private ValueRepresentation vr = null;
        /// <summary>
        ///     Data element value representation.
        /// </summary>
        public ValueRepresentation VR
        {
            get 
            {
                if (vr != null) 
                    return vr; 
                else
                    throw new DicomException("DataElement.VR is null.",
                        this.Tag);
            }
        }

        private ValueLength valueLength = null;
        /// <summary>
        ///     Data element value length.
        /// </summary>
        public ValueLength ValueLength
        {
            get 
            {
                if (valueLength != null) 
                    return valueLength; 
                else
                    throw new DicomException("DataElement.ValueLength is null.",
                        this.Tag);
            }
        }

        private Value value = null;
        /// <summary>
        ///     Data element value.
        /// </summary>
        public Value Value
        {
            get 
            {
                if (value != null) 
                    return value; 
                else
                    throw new DicomException("DataElement.Value is null.",
                        this.Tag);
            }
        }

        /// <summary>
        ///     DICOM stream position of this data element instance.
        /// </summary>
        public long StreamPosition
        {
            get { return Tag.StreamPosition; }
        }

        private TransferSyntax transferSyntax = TransferSyntax.Default;
        /// <summary>
        ///     Transfer syntax of this data element instance.
        /// </summary>
        public TransferSyntax TransferSyntax
        {
            set
            {
                if (value == null)
                    transferSyntax = TransferSyntax.Default;
                else
                    transferSyntax = value;
            }

            get { return transferSyntax; }
        }

    
        /// <summary>
        ///     Creates a new data element instance from specified DICOM
        ///     output stream using the DICOM default transfer syntax.
        /// </summary>
        public DataElement(Stream stream): this(stream, null) {}

        /// <summary>
        ///     Creates a new data element instance from specified DICOM
        ///     output stream using specified DICOM transfer syntax.
        /// </summary>
        public DataElement(Stream stream, TransferSyntax transferSyntax)
        {
            TransferSyntax = transferSyntax;
            LoadFrom(stream);
        }

        /// <summary>
        ///     Creates a new data element instance from specified DICOM
        ///     output stream using specified DICOM transfer syntax.
        /// </summary>
        public DataElement(Stream stream, TransferSyntax transferSyntax, object obj)
        {
            TransferSyntax = transferSyntax;
            parent = obj;
            LoadFrom(stream);
        }
        /// <summary>
        /// Creates data element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        public DataElement(string tag, string vr): this(tag, vr, null) {}
        /// <summary>
        /// Creates data element.
        /// </summary>
        /// <see cref="TransferSyntax"/>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        /// <param name="transferSyntax">element transfer syntac</param>
        public DataElement(string tag, string vr, TransferSyntax transferSyntax)
        {
            TransferSyntax = transferSyntax;
            this.tag = new Tag(tag, TransferSyntax);
            this.vr = ValueRepresentation.GetBy(vr, Tag);
        }

        /// <summary>
        /// Creates data element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="value">value at tag</param>
        public DataElement(Tag tag, Value value)
        {
            this.tag = tag;
            this.value = value;
        }
        /// <summary>
        /// Creates data element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        public DataElement(Tag tag, ValueRepresentation vr)
        {
            this.tag = tag;
            TransferSyntax = Tag.TransferSyntax;
            if (vr == null) vr = ValueRepresentation.GetBy(Tag);
            this.vr = vr;
        }

        /// <summary>
        ///     Re-creates a data element instance from specified DICOM
        ///     input stream using <see cref="DataElement.TransferSyntax" />.
        /// </summary>
        /// <param name="stream">input stream</param>
        public void LoadFrom(Stream stream)
        {
            tag = new Tag(stream, TransferSyntax);
            //sequence
            vr = ValueRepresentation.LoadFrom(stream, Tag);
            valueLength = new ValueLength(stream, VR);
            value = new Value(stream, VR, ValueLength, this);
        }

        /// <summary>
        ///     Writes a DICOM stream member instance to a specified
        ///     DICOM output stream.
        /// </summary>      
        public void SaveTo(Stream stream)
        {
            tag.SaveTo(stream);
            value.VR.SaveTo(stream);
            value.SaveTo(stream);     
        }

        /// <summary>
        ///     Implementation of the IComparable interface. So use
        ///     of this class within collections is guaranteed.
        /// </summary>
        public int CompareTo(object obj)
        {
            return Tag.CompareTo(((DataElement) obj).Tag);
        }
        /// <summary>
        /// Returns string representation of this object. 
        /// </summary>
        /// <returns>this object as a string</returns>
        public override string ToString()
        {
            //resolve object type
            if (value == null) return "<null>";

            object[] data_array = value.ToArray();
            if (data_array.Length == 0) return "<no data>";
            string r = "";
            for (int i = 0; i < data_array.Length; i++) {
                switch (data_array[i].GetType().Name) {
                    case "PersonName":
                        r += " " + ((openDicom.Encoding.Type.PersonName)data_array[i]).FullName;
                        break;
                    case "Decimal":
                        r += " " + ((Decimal)data_array[i]).ToString().Replace(',','.');
                        break;
                    case "String":
                        r += " " + ((String)data_array[i]).ToString();
                        break;
                    case "UInt16[]":
                        UInt16[] uint16s = (UInt16[])data_array[i];
                        foreach (UInt16 b in uint16s)
                            r += " " + b.ToString();
                        break;
                    case "UInt16":
                        r += " " + ((UInt16)data_array[i]).ToString();
                        break;
                    case "Int16":
                        r += " " + ((Int16)data_array[i]).ToString();
                        break;
                    case "UInt32":
                        r += " " + ((UInt32)data_array[i]).ToString();
                        break;
                    case "Int32":
                        r += " " + ((Int32)data_array[i]).ToString();
                        break;
                    case "UInt64":
                        r += " " + ((UInt64)data_array[i]).ToString();
                        break;
                    case "Int64":
                        r += " " + ((Int64)data_array[i]).ToString();
                        break;
                    case "Sequence":
                        r += " Sequence";
                        break;
                    case "NestedDataSet":
                        r += " NestedDataSet";
                        break;
                    case "Byte[]":
                        Byte[] bytes = (Byte[])data_array[i];
                        foreach(Byte b in bytes)
                            r += " " + b.ToString();
                        break;
                    case "Uid":
                        if (TPClib.Image.DicomFile.uidDictionary.GetDictionaryEntry((Uid)data_array[i]) != null)
                            r += " " + TPClib.Image.DicomFile.uidDictionary.GetDictionaryEntry((Uid)data_array[i]).ToString();
                        else
                            r += " (dictionary entry not found)";
                        break;
                    case "DateTime":
                        r += " " + ((System.DateTime)data_array[i]).ToString("HH:mm:ss dd:MM:yyyy");
                        break;
                    case "TimeSpan":
                        r += " " + ((TimeSpan)data_array[i]).ToString();
                        break;
                    case "Single":
                        r += " " + ((Single)data_array[i]).ToString("0.00000").Replace(',','.');
                        break;
                    case "Double":
                        r += " " + ((Double)data_array[i]).ToString("0.0000000000").Replace(',','.');;
                        break;
                    case "Age":
                        openDicom.Encoding.Type.Age age = (openDicom.Encoding.Type.Age)data_array[i];
                        r += " " + age.AgeValue.ToString();
                        switch (age.Context)
                        {
                            case openDicom.Encoding.Type.AgeContext.Days:
                                r += " days";
                                break;
                            case openDicom.Encoding.Type.AgeContext.Months:
                                r += " months";
                                break;
                            case openDicom.Encoding.Type.AgeContext.Weeks:
                                r += " weeks";
                                break;
                            case openDicom.Encoding.Type.AgeContext.Years:
                            default:
                                r += " years";
                                break;
                        }
                        break;
                    default:
                        r+=" unrecognized type = ["+data_array[i].GetType().Name+"]";
                        break;
                }
            }
            return r.TrimStart(' ');
        }
        /// <summary>
        /// Explicit cast operator.
        /// </summary>
        /// <param name="e">dataelement</param>
        /// <returns>string representation from ToString() method</returns>
        public static implicit operator string(DataElement e) {
            return e.ToString();
        }
    }
}
