/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Text;
using System.IO;
using System.Collections.Generic;

namespace TPClib
{
	public class ChangeLog : TPCFile
	{
		public const string COMMENTSTRING = @"#";
		public const string ITEMSTRING = @"--";

		public class Entry : IComparable
		{
			private List<ItemTag> tags = new List<ItemTag>();
			private List<string> items = new List<string>();

			private ItemTag versionTag;

			public class ItemTag
			{
				public const string TAGSTART = @"[";
				public const string TAGEND = @"]";
				public const string TAGSET = @"=";

				public string Name;

				public string Value;

				public ItemTag(string n = null, string v = null) { Name = n; Value = v; }

				public static ItemTag ParseTag(string s)
				{
					ItemTag t = new ItemTag();
					string parseLine = s.Trim();
					if (parseLine.StartsWith(TAGSTART) && parseLine.EndsWith(TAGEND))
					{
						string[] split = parseLine.Split(new string[] { TAGSET, TAGSTART, TAGEND }, StringSplitOptions.RemoveEmptyEntries);
						t.Name = split[0].Trim();
						if (split.Length > 1) t.Value = split[1].Trim();
						else t.Value = String.Empty;
					}
					return t;
				}

				public override string ToString()
				{
					return "[" + this.Name + " = " + this.Value + "]";
				}
			}

			public Entry(string version = null)
			{
				versionTag = new ItemTag("version", version);
				tags.Add(versionTag);
			}

			public bool IsEmpty { get { return versionTag.Value == null; } }

			public void AddTag(string name, string value)
			{
				AddTag(new ItemTag(name, value));
			}

			public void AddTag(ItemTag tag)
			{
				int index = tags.FindIndex(delegate(ItemTag t) { return t.Name == tag.Name; });
				if (index < 0) tags.Add(tag);
				else tags[index].Value = tag.Value;
			}

			public void AddItem(string item)
			{
				items.Add(item);
			}

			public override string ToString()
			{
				StringBuilder sb = new StringBuilder();
				foreach (ItemTag t in tags)
				{
					sb.AppendLine(t.ToString());
				}
				foreach (string s in items)
				{
					sb.AppendLine(ITEMSTRING + " " + s);
				}
				return sb.ToString();
			}

			public override bool Equals(object obj)
			{
				return CompareTo(obj) == 0;
			}

			public override int GetHashCode()
			{
				return versionTag.GetHashCode();
			}

			public int CompareTo(object obj)
			{
				if (obj is Entry)
				{
					string a = this.versionTag.Value;
					string b = (obj as Entry).versionTag.Value;
					if (a == null && b == null) return 0;
					else if (a == null) return -1;
					else if (b == null) return 1;
					else return this.versionTag.Value.CompareTo((obj as Entry).versionTag.Value);
				}
				else throw new ArgumentException("Wrong type for comparison or null value");
			}
		}

		private List<Entry> entries = new List<Entry>();

		public override void ReadFile()
		{
			IList<string> lines = ReadAllLines();
			Entry e = ReadEntry(lines);
			while (!e.IsEmpty)
			{
				entries.Add(e);
				e = ReadEntry(lines);
			}
		}

		public override void WriteFile()
		{
			using (StreamWriter sw = new StreamWriter(new FileStream(this.filename, FileMode.Create)))
			{
				sw.Write(this.ToString());
			}
		}

		public override string ToString()
		{
			StringBuilder sb = new StringBuilder();
			entries.Sort();
			foreach (Entry e in entries)
			{
				sb.AppendLine(e.ToString());
			}
			return sb.ToString();
		}

		private IList<string> ReadAllLines()
		{
			List<string> lines = new List<string>();
			using (StreamReader sr = new StreamReader(new FileStream(this.filename, FileMode.Open)))
			{
				while (!sr.EndOfStream)
				{
					string line = sr.ReadLine().Trim();
					if (!line.StartsWith(COMMENTSTRING) && line.Length > 0) lines.Add(line);
				}
			}
			return lines;
		}

		private Entry ReadEntry(IList<string> lines)
		{
			Entry e = new Entry();
			Entry.ItemTag tag;
			while (lines.Count > 0)
			{
				tag = Entry.ItemTag.ParseTag(lines[0]);
				if (tag.Name == null) break;
				e.AddTag(tag);
				lines.RemoveAt(0);
			}
			string itemString;
			while (lines.Count > 0 && lines[0].StartsWith(ITEMSTRING))
			{
				itemString = lines[0].Substring(ITEMSTRING.Length).Trim();
				lines.RemoveAt(0);
				while (lines.Count > 0 && !(lines[0].StartsWith(ITEMSTRING) || lines[0].StartsWith(Entry.ItemTag.TAGSTART)))
				{
					itemString = itemString + " " + lines[0];
					lines.RemoveAt(0);
				}
				e.AddItem(itemString);
			}
			return e;
		}
	}
}
