/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Curve
{
	/// <summary>
	/// CPT Time Activity Curve (TAC) file format. CPT is an ASCII data format for regional time-activity curves, generated by CTI Imagetool and Vinci
	/// </summary>
	public class CPTCurveFile : CurveFile
	{
		/// <summary>
		/// CPT file header
		/// </summary>
		public struct CPTHeader
		{
			/// <summary>
			/// Information about software that was used to create this data
			/// </summary>
			public string SoftwareDescription;
			/// <summary>
			/// Time of creation for data
			/// </summary>
			public System.DateTime CreationTime;
			/// <summary>
			/// Image file that was used to draw this ROIs
			/// </summary>
			public string InitialFilename;
			/// <summary>
			/// Associated Vinci project file name
			/// </summary>
			public string AssociatedProjectFilename;
			/// <summary>
			/// Vinci project signature
			/// </summary>
			public string ProjectSignature;
			/// <summary>
			/// Display size in pixels in Vinci when drawing ROI
			/// </summary>
			public int DisplaySize;
			/// <summary>
			/// Sampling size in pixels when drawing ROI
			/// </summary>
			public int SamplingSize;
			/// <summary>
			/// Interpolation method in display window
			/// </summary>
			public string Interpolation;
			/// <summary>
			/// Zoom method in display
			/// </summary>
			public string ZoomMethod;
			/// <summary>
			/// ROI thickness in planes
			/// </summary>
			public int ROIWidth;
			/// <summary>
			/// ROI masking interpolation method
			/// </summary>
			public string ROIInterpolation;
			/// <summary>
			/// ROI coordinate units
			/// </summary>
			public string Units;
			/// <summary>
			/// Name of the ROI
			/// </summary>
			public string ROIName;
		}

		private CPTHeader header;

		private static IFormatProvider format = System.Globalization.NumberFormatInfo.InvariantInfo;

		/// <summary>
		/// Default constructor.
		/// </summary>
		public CPTCurveFile(){}

		/// <summary>
		/// Constructor with filename.
		/// </summary>
		/// <param name="filename">CPT filename</param>
		public CPTCurveFile(string filename) { this.filename = filename; }

		/// <summary>
		/// CPT specific header information
		/// </summary>
		public CPTHeader Header { get { return header; } }

		/// <summary>
		/// Reads CPT file into tacs structures (define here which variables)
		/// </summary>
		public override void ReadFile()
		{
			// read the file to memory
			string[] lines = System.IO.File.ReadAllLines(filename);
			if (lines.Length < 12) throw new TPCCurveFileException("Missing header information in Vinci CPT file");
			//read description
			header.SoftwareDescription = lines[0].Trim().Remove(0, 1).Trim();
			//try to parse creation date
			System.DateTime.TryParse(lines[1].Remove(0, 13), out header.CreationTime);
			header.InitialFilename = lines[2].Substring(lines[2].IndexOf(':') + 1).Trim().Trim('\"');
			header.AssociatedProjectFilename = lines[3].Substring(lines[3].IndexOf(':') + 1).Trim().Trim('\"');
			header.ProjectSignature = lines[4].Substring(lines[4].IndexOf(':') + 1).Trim('\"', ' ');
			int.TryParse(lines[5].Substring(lines[5].IndexOf(':') + 1), out header.DisplaySize);
			int.TryParse(lines[6].Substring(lines[6].IndexOf(':') + 1), out header.SamplingSize);
			header.Interpolation = lines[7].Substring(lines[7].IndexOf(':') + 1).Trim();
			header.ZoomMethod = lines[8].Substring(lines[8].IndexOf(':') + 1).Trim();
			int.TryParse(lines[9].Substring(lines[9].IndexOf(':') + 1), out header.ROIWidth);
			header.ROIInterpolation = lines[10].Substring(lines[10].IndexOf(':') + 1).Trim();
			header.Units = lines[11].Substring(lines[11].IndexOf(':') + 1).Trim();

			//resolve ROI name
			int line_Index = 11;
			bool transaxial = false;
			while (!lines[++line_Index].StartsWith("# Using ROI "))
			{
			}
			//check if this ROI was drawn on transaxial slice
			if (lines[line_Index].Contains("TRANSAXIAL")) transaxial = true;
			string[] splitted = lines[line_Index].Split('"');
			if (splitted.Length != 3) throw new TPCCurveFileException("Invalid ROI name data at line " + (line_Index + 1));
			header.ROIName = splitted[1].Trim();

			//create data for mean,total, and std
			curves = new RegionalTACTable(new TACTable(3));
			curves.SetUnit(header.Units);
			curves.Timetype = FrameTimeCell.FrameTimetype.START_END;
			(curves as RegionalTACTable).curveheader[0].secondaryName = "ROI Avg";
			(curves as RegionalTACTable).curveheader[1].secondaryName = "ROI Total";
			(curves as RegionalTACTable).curveheader[2].secondaryName = "ROI Stdev";
			//ROI Volume in voxels
			int ROI_Vol;
			int.TryParse(splitted[2].Substring(splitted[2].IndexOf("size:") + 5, splitted[2].IndexOf("pixels") - (splitted[2].IndexOf("size:") + 5)), out ROI_Vol);
			(curves as RegionalTACTable).curveheader[0].ROIsize = ROI_Vol;
			(curves as RegionalTACTable).curveheader[1].ROIsize = ROI_Vol;
			(curves as RegionalTACTable).curveheader[2].ROIsize = ROI_Vol;

			//read until 1st frame number is found
			while (!lines[++line_Index].StartsWith("1"))
			{
			}

			//resolve number of frames
			int frames = lines.Length - line_Index;

			//ROI average intensity
			double ROI_Avg;
			//ROI Total intensity
			double ROI_Total;
			//ROI Standard deviation of intensity around average
			double Stdev;
			//Frame start and duration in sec
			double startTime;
			double duration;

			bool first_TAC_line = true;

			IFormatProvider format = System.Globalization.NumberFormatInfo.InvariantInfo;
			System.Globalization.NumberStyles style = System.Globalization.NumberStyles.Float;

			//read lines 
			for (; line_Index < lines.Length; line_Index++)
			{
				splitted = lines[line_Index].Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
				if (splitted.Length != 11) throw new TPCCurveFileException("Invalid number of items in CPT TAC data in line " + (line_Index + 1) + ", should be 11");

				//add new row information
				double.TryParse(splitted[3], style, format, out ROI_Avg);
				double.TryParse(splitted[5], style, format, out ROI_Total);
				double.TryParse(splitted[6], style, format, out Stdev);
				curves.AddRow(new double[] { ROI_Avg, ROI_Total, Stdev });
				double.TryParse(splitted[7], style, format, out startTime);
				double.TryParse(splitted[8], style, format, out duration);
				(curves as RegionalTACTable).SetTimeCell(curves.Rows - 1, new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END, startTime, startTime + duration));

				//set ROI size and plane number if this is the first line that is read
				//corresponding in formation in other lines is ignored for now, since there is no need for 
				//suport for dynamically changing ROIs
				if (first_TAC_line)
				{
					if (transaxial)
					{
						(curves as RegionalTACTable).curveheader[0].plane = splitted[1].Trim();
						(curves as RegionalTACTable).curveheader[1].plane = splitted[1].Trim();
						(curves as RegionalTACTable).curveheader[2].plane = splitted[1].Trim();
					}
					first_TAC_line = false;
				}
			}
		}
		/// <summary>
		/// Writes TACTable tacs into CPT file. 
		/// </summary>
		public override void WriteFile()
		{
			if (curves.Columns < 3)
				throw new TPCCurveFileException("CPT format must have at least 3 TACs (Avg, Total, Std) to be written");
			if (curves.GetTimeCell(0).type != FrameTimeCell.FrameTimetype.START_END)
				throw new TPCCurveFileException("CPT supports only start-end type of frame times");
			List<string> lines = new List<string>();
			lines.Add("# Written with " + TPClib.Version.GetVersionInformation());
			lines.Add("# created at " + DateTime.Now.ToString("MMM dd yyyy HH:mm:SS"));
			lines.Add("# ROI initially defined on file: \"" + header.InitialFilename + "\"");
			lines.Add("# associated project file: \"" + header.AssociatedProjectFilename + "\"");
			lines.Add("# project signature: \"" + header.ProjectSignature + "\"");
			lines.Add("# Display Size: " + header.DisplaySize);
			lines.Add("# Sampling Size: " + header.SamplingSize);
			lines.Add("# Interpolation: " + header.Interpolation);
			lines.Add("# Zoom Method: " + header.ZoomMethod);
			lines.Add("# ROI Width (+/- Planes): " + header.ROIWidth);
			lines.Add("# Interpolation: " + header.ROIInterpolation);
			lines.Add("# Units: " + curves.Unit.ToString().ToLowerInvariant());
			lines.Add("");
			lines.Add("# Using ROI \"" + header.ROIName + "\" (TRANSAXIAL), size: " + (curves as RegionalTACTable).curveheader[0].ROIsize + " pixels (one slice)");
			lines.Add("# ROI surface[mm" + '\u00B2' + "]: " + (curves as RegionalTACTable).curveheader[0].ROIsize + ", ROI volume[mm" + '\u00B2' + "]: 0.0");
			lines.Add("");
			lines.Add("Frame Cut   ROI ID        ROI Avg    #pixels    ROI Total   %Stdev    Offset   Duration   ROI Surf.     ROI Vol.");
			lines.Add("                                     (screen)                          (sec)     (sec)      mmxmm       mmxmmxmm");
			for (int row_i = 0; row_i < curves.Rows; row_i++)
			{
				lines.Add((row_i + 1).ToString(format).PadRight(6) +
						  curves.GetTAC(0).Header.plane.PadRight(6) +
						  "1".PadRight(13) +
						  curves.GetTAC(0)[row_i].Value.ToString("E4", format).PadRight(13) +
						  (curves as RegionalTACTable).curveheader[0].ROIsize.ToString(format).PadRight(9) +
						  curves.GetTAC(1)[row_i].Value.ToString("E4", format).PadRight(14) +
						  curves.GetTAC(2)[row_i].Value.ToString("0.0", format).PadLeft(4) +
						  curves.GetTimeCell(row_i).start_time.ToString("0.0", format).PadLeft(11) +
						  (curves.GetTimeCell(row_i).end_time - curves.GetTimeCell(row_i).start_time).ToString("N1", format).PadLeft(10) +
						  new Single().ToString("E4", format).PadLeft(14) +
						  new Single().ToString("E4", format).PadLeft(14));
			}

			//Write all data at once into the file
			System.IO.File.WriteAllLines(filename, lines.ToArray());
		}

		/// <summary>
		/// Checks file format based on filename extension. This method reads 
		/// all of the data, since no magic number or static format recognition 
		/// for this file format is specified.
		/// </summary>
		/// <param name="filename">full path to file</param>
		/// <returns>true if file is a CPT file</returns>
		/// <exception cref="TPCIFFileException">if there was a read problem</exception>
		public static bool CheckFormat(string filename)
		{
			System.IO.FileInfo file = new System.IO.FileInfo(filename);
			if (String.Compare(file.Extension, ".cpt", true) == 0) //case insensitive comparison
			{
				//try to read file with readFile()
				try
				{
					CPTCurveFile cptfile = new CPTCurveFile();
					cptfile.filename = filename;
					cptfile.ReadFile();
					return true;
				}
				catch (Exception)
				{
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}
}
