/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Text;
using System.IO;
using TPClib.Common;

namespace TPClib.Curve
{
    #region exceptions
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCDFTCurveFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCDFTCurveFileException() : base("No message") {      
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCDFTCurveFileException(string message)
            : base(message)
        {
        }
    }
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCIFFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCIFFileException() : base("No message") {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCIFFileException(string message)
            : base(message)
        {
        }
    }
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCIDWCFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCIDWCFileException()
            : base("No message")
        {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCIDWCFileException(string message)
            : base(message)
        {
        }
    }  
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCCurveFileException : TPCException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCCurveFileException()
            : base("No message")
        {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCCurveFileException(string message)
            : base(message)
        {
        }
    }
    #endregion

    /// <summary>
    /// DFT-file format.
    /// </summary>
    public class DFTCurveFile : CurveFile
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public DFTCurveFile()
        {
            this.FileType = CurveFileType.Unknown_DFT;
        }

        /// <summary>
        /// constructor with filename
        /// </summary>
        /// <param name="filename">name of the DFT file</param>
        public DFTCurveFile(String filename)
        {
            this.FileType = CurveFileType.Unknown_DFT;
            this.filename = filename;
        }

        /// <summary>
        /// Reads the file. Filename must be given.
        /// </summary>
        /// <exception cref="TPCDFTCurveFileException">If filename is null.</exception>
        public override void ReadFile()
        {
            if (filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            
            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
			string text;
			using (StreamReader reader = new StreamReader(new FileStream(filename, FileMode.Open, FileAccess.Read), new ASCIIEncoding()))
			{
				text = reader.ReadToEnd();
			}
            // we need new reader to emptyheader generation because we need to read one line. This line
            // would be lost in readCurves() if the reader would be same one
            StringReader strReader2 = new StringReader(text);
            generateEmptyHeader(ref strReader2);
            strReader2.Close();

            string[] lines = File.ReadAllLines(filename);
            TimeUnit timeunit = TimeUnit.Minute;
            foreach (string line in lines)
            {
                if (line.StartsWith("# Activity units:"))
                {
                    string activity_units = line.Substring(line.LastIndexOf("units:") + 6).ToLower();
                    this.Curves.Unit = DataUnit.Parse(activity_units.Trim());
                }
                if (line.StartsWith("# Time units:"))
                {
                    string time_unit = line.Substring(line.LastIndexOf("units:") + 6).Trim().ToLower();
                    if (time_unit.Contains("s")) timeunit = TimeUnit.Second;
                }
                if (line.StartsWith("# unit :=")) 
                {
                    string activity_units = line.Substring(line.LastIndexOf("unit :=") + 7).ToLower();
                    this.Curves.Unit = DataUnit.Parse(activity_units.Trim());
                }
                if (line.StartsWith("# timeunit :=")) 
                {
                    string time_unit = line.Substring(line.LastIndexOf("unit :=") + 7).Trim().ToLower();
                    if (time_unit.Contains("sec")) timeunit = TimeUnit.Second;
                }
            }
			StringReader strReader = new StringReader(text);
			this.readCurves(ref strReader, timeunit);
            strReader.Close();
        }

        /// <summary>
        /// Reads DFT file.
        /// </summary>
        /// <param name="filename">filename of DFT tacs</param>
        /// <returns>DFT file that has all tacs in image</returns>
        /// <exception cref="TPCException">if reading of file failed</exception>
        public new static DFTCurveFile ReadFile(string filename)
        {
            CurveFileType filetype;
            try
            {
                filetype = CurveFile.ResolveFileFormat(filename);
            }
            catch (Exception)
            {
                throw new TPCException("Failed to recognize file " + filename);
            }
            switch (filetype)
            {
                case CurveFileType.Plasma_DFT:
                    DFTPlasmaCurveFile plasma_curve_file = new DFTPlasmaCurveFile(filename);
                    plasma_curve_file.ReadFile();
                    return plasma_curve_file;
                case CurveFileType.Regional_DFT:
                    DFTRegionalCurveFile regional_curve_file = new DFTRegionalCurveFile(filename);
                    regional_curve_file.ReadFile();
                    return regional_curve_file;
                case CurveFileType.Unknown_DFT:
                default:
                    DFTCurveFile curve_file = new DFTCurveFile(filename);
                    curve_file.ReadFile();
                    return curve_file;
            }

        }

        /// <summary>
        /// Creates a DFT file from contents of DFTRegionalcurvefile. Filename must be specified before
        /// calling this method
        /// </summary>
        public override void WriteFile()
        {
            if (this.filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            else WriteFile(this.filename);
        }

        /// <summary>
        /// Creates normal DFTCurvefile with no header. 
        /// </summary>
        /// <remarks>data is written with sec time unit</remarks>
        /// <param name="filename">Name of file to create.</param>
        public virtual void WriteFile( String filename )
        {
            WriteFile(filename, TimeUnit.Second);
        }

        /// <summary>
        /// Creates normal DFTCurvefile with no header.
        /// </summary>
        /// <param name="filename">Name of file to create.</param>
        /// <param name="timeunit">time unit for writing</param>
        public virtual void WriteFile(String filename, TimeUnit timeunit)
        {
            // Writing the text to file
			using (StreamWriter fileWriter = new StreamWriter(new FileStream(filename, FileMode.Create, FileAccess.Write), new ASCIIEncoding()))
			{
				fileWriter.AutoFlush = true;
				fileWriter.Write(GetDataAsString(timeunit));
			}
		}

        /// <summary>
        /// Returns a string representation of this object.
        /// </summary>
        /// <returns>This object as string</returns>
        public override string ToString()
        {
            return "DFTCurveFile[filename="+filename+" filetype="+FileType+"]";
        }

        /// <summary>
        /// Returns a String containing all information in form that would be written to file.
        /// </summary>
        /// <param name="time">Time format. Can be "(min)", "(sec)" or "."</param>
        /// <returns>Contents of file in string.</returns>
        protected string GetDataAsString(TimeUnit time)
        {
            StringBuilder sb = new StringBuilder();
            StringWriter writer = new StringWriter(sb);

			try
			{
				// Now we can write all the curve tacs;
				writeCurves(ref writer, time);

				// Comment lines
				foreach (String comment in Comments)
				{
					writer.WriteLine(comment);
				}
				writer.Flush();
			}
			catch (Exception e)
			{
				throw new TPCDFTCurveFileException("String writing failed: " + e + " -- " + e.StackTrace + "\r\n>>" + e.Source);
			}
			finally
			{
				writer.Close();
			}
            return sb.ToString();
        }

        /// <summary>
        /// Checks file format based on contents of the file.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT file</returns>
        /// <exception cref="TPCDFTCurveFileException">if there was a read problem</exception>
        public static CurveFileType CheckFormatByContents(string filename)
        {
            try
            {
                // we open the file and put its contents to String which is
                // read with StringReader. We can close the file after String is in memory
				string text;
				using (StreamReader reader = new StreamReader(new FileStream(filename, FileMode.Open, FileAccess.Read), new ASCIIEncoding()))
				{
					text = reader.ReadLine();
				}

				//Couldn't read single line
                if (text == null)
                    return CurveFileType.Unknown_DFT;

                // Next we must check what kind of file we are trying to read
                // We look into first 3 letters, which give a lot of information.
                // we ignore all the possible empty spaces                
                String StartString = text.Substring(0, (text.Length >= 10? 10 : text.Length));
                StartString.Trim();
                StartString = StartString.Substring(0, 3);

                // Case1: normal DFT file starts with string DFT
                if (StartString.Equals("DFT"))
                {
                    return CurveFileType.Regional_DFT;
                }

                // Case2: Typical plasma TAC starts with "# Injection"
                else if (StartString.Equals("# I"))
                {
                    return CurveFileType.Plasma_DFT;
                }
                else
                {
                    return CurveFileType.Unknown_DFT;
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read file:" + e.Message);
            }
        }


        /// <summary>
        /// Checks file format based on filename extension.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT file</returns>
        /// <exception cref="TPCDFTCurveFileException">if there was a read problem</exception>
        public static bool CheckFormat( String filename )
        {
            FileInfo file = new FileInfo(filename);
            if (String.Compare(file.Extension, "if", true) == 0)
            {
                return false;
            }
            else if (String.Compare(file.Extension, "idwc", true) == 0 || String.Compare(file.Extension, "idw", true) == 0)
            {
                return false;
            }
            else if (String.Compare(file.Extension, "xml", true) == 0 || String.Compare(file.Extension, "html", true) == 0)
            {
                return false;
            }
            
            //try to resolve format by reading its contents
            try
            {
                return ( !CheckFormatByContents(filename).Equals( "unknown" ) );
            }
            catch (Exception e)
            {
                throw new TPCCurveFileException("Fileformat not recognized."+e);
            }
        }

        #region private members

        /// <summary>
        /// Reads all the curve tacs from file with no header
        /// </summary>
        /// <param name="reader">string reader stream</param>
        /// <param name="time">time format of file as string</param>
        protected void readCurves(ref StringReader reader, TimeUnit time)
        {
            String[] words = null;
            TableCell[] curvecells = null;
            FrameTimeCell timecell = null;

            // Secons are chanced to minutes if the original file have had minute values
            Double timefactor = 1.0000d;
            if (time == TimeUnit.Minute) timefactor = 60.0000d;
            else timefactor = 1.0000d;

            int dataStartColumn = 1;
            if (Curves.Timetype == FrameTimeCell.FrameTimetype.START_END) dataStartColumn = 2;

            do
            {
                words = this.ReadWordsFromLine(ref reader);

                if (words != null)
                {
                    //ensure that line has at least one convertible value
                    if(words.Length == 0) continue;
                    try
                    {
                        StrToDouble(words[0]);
                    }
                    catch {
                        continue;
                    }

                    // we need to convert the curve strings to curve doubles
                    int numcols = Curves.Columns;

                    curvecells = new TableCell[numcols];

                    if( Curves.Timetype == FrameTimeCell.FrameTimetype.START_END )
                    {
                        timecell = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                        timecell.start_time = StrToDouble(words[0]);
                        timecell.end_time = StrToDouble(words[1]);
                    }
                    else 
                    {
                        timecell = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                        timecell.mid_time = StrToDouble(words[0]);
                    }

                    // minutes are chanced to seconds
                    timecell *= timefactor;

                    for (int i = 0; i < numcols; i++)
                    {
                        curvecells[i] = new TableCell();
                        if (words[i+dataStartColumn].Equals(".")) curvecells[i].Value = Double.NaN;
                        else
                        {
                            curvecells[i].Value = StrToDouble(words[i + dataStartColumn]);
                        }
                    }

                    // we add the new row to TACTable
                    Curves.AddRow(timecell, curvecells);
                }
            }
            while (words != null);
        }

        /// <summary>
        /// Converts strings to double in scientific mode with dot
        /// </summary>
        /// <param name="str">input string</param>
        /// <returns>a double which is converted from string</returns>
        protected Double StrToDouble(String str)
        {
            return Convert.ToDouble(str.TrimEnd(','), new System.Globalization.CultureInfo("en-GB"));
        }

        /// <summary>
        /// Creates a fixed size String from input string and result length.
        /// All extra space is filled with spaces. If the input string is longer than result,
        /// the remainders are cut off
        /// </summary>
        /// <param name="input">The input string</param>
        /// <param name="length">length if result string</param>
        /// <returns>input string, but with fixed length</returns>
        protected String FixStr(String input, int length)
        {
            // null input is treated like empty string 
            if (input == null) return new String(' ',length);

            int difference = length - input.Length;

            // the input string is longer than the boundaries given
            if (difference < 0) { return input.Substring(0, length); }
            else if (difference == 0) { return input; }
            else return input.PadRight(input.Length + difference, ' ');
        }



        /// <summary>
        /// Writes all the curve tacs to the file with no headers.
        /// </summary>
        /// <param name="writer">Streamwriter reference_times to the DFT file</param>
        /// <param name="time">Unit of time. Can be "(min)", "(sec)", or "."</param>
        protected void writeCurves(ref StringWriter writer, TimeUnit time)
        {
            // Secons are chanced to minutes if the original file have had minute values
            Double timefactor=1.0000d;
            if (time == TimeUnit.Minute) timefactor = 1.0000d / 60.0000d;
            else timefactor = 1.0000d;

            // header is now ready. Now we write the tacs
            for (int i = 0; i < Curves.Rows; i++)
            {
                FrameTimeCell timecell = Curves.GetTimeCell(i);

                // we chance the seconds to minutes if we must
                timecell *= timefactor;

                // if there are two time columns, we print both values inside 22 characters
                if (timecell.type == FrameTimeCell.FrameTimetype.START_END)
                {
                    writer.Write(FixStr(
                    (timecell.start_time).ToString("F3", new System.Globalization.CultureInfo("en-GB")), 11));
                    writer.Write(FixStr(
                    (timecell.end_time).ToString("F3", new System.Globalization.CultureInfo("en-GB")), 11));
                }
                else if (timecell.type == FrameTimeCell.FrameTimetype.START)
                {
                    // Starting time
                    writer.Write(FixStr("" + timecell.start_time.ToString("F3", new System.Globalization.CultureInfo("en-GB")), 22));
                }
                else
                {
                    // If the time cell is not StartEnd or start, we use mid time
                    writer.Write(FixStr("" + timecell.mid_time.ToString("F3", new System.Globalization.CultureInfo("en-GB")), 22));
                }

                // next we print every datacells of the row
                TableCell[] tabcells = Curves.GetRow(i);

                foreach (TableCell cell in tabcells)
                {
                    if (cell==null || Double.IsNaN(cell.Value))
                    {
                        writer.Write(FixStr(".", 20));
                    }
                    else
                    {
                        String s = ((Double)cell).ToString("e11", new System.Globalization.CultureInfo("en-GB"));
                        writer.Write(FixStr(s, 20));
                    }
                }
                writer.Write("\r\n");
            }
        }

        /// <summary>
        /// If the there is no header tacs in the file the columns are generated according to
        /// curve tacs. This function assumes that the time tacs is written to one column
        /// </summary>
        /// <param name="reader">string reader stream</param>
        protected void generateEmptyHeader(ref StringReader reader)
        {
            String[] words = ReadWordsFromLine(ref reader);

            // the TACtable is created. The empty headercell list are needed because TACtable needs
            // to know how many columns there must be
            Curves = new TACTable(FrameTimeCell.FrameTimetype.MID, words.Length-1);
        }
        #endregion
    }
}
