/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;

namespace TPClib.Curve
{
    /// <summary>
    /// Sort helper function for DFTCurveTAC. This is needed when colmuns are sorted by CurveName for example.
    /// Also Find operations are performed through this class.
    /// </summary>
    public class DFTTACComparer : TACComparer, IComparer<TAC>
    {
        /// <summary>
        /// The fields which are used for comparing
        /// </summary>
        public enum CompareField
        {
            /// <summary>
            /// Name of TAC region
            /// </summary>
            CurveName,
            /// <summary>
            /// Secondary name (sin or dx etc.)
            /// </summary>
            SecondaryName,
            /// <summary>
            /// Size of region in voxels
            /// </summary>
            ROISize,
            /// <summary>
            /// Region plane number
            /// </summary>
            Plane
        }
        /// <summary>
        /// The sorting field
        /// </summary>
        private CompareField compareElement = CompareField.CurveName;
        /// <summary>
        /// Default constructor. Inits comparefield to CurveName by default.
        /// </summary>
        public DFTTACComparer()
        {
            compareElement = CompareField.CurveName;
        }
        /// <summary>
        /// Constructor with comparefield parameter.
        /// </summary>
        /// <param name="cf">This parameter gives the sorting field</param>
        public DFTTACComparer(CompareField cf)
        {
            compareElement = cf;
        }

        /// <summary>
        /// Comparing function.
        /// </summary>
        /// <param name="pFirstObject">first object</param>
        /// <param name="pObjectToCompare">second object</param>
        /// <returns>result</returns>
        Int32 System.Collections.Generic.IComparer<TAC>.Compare(TAC pFirstObject, TAC pObjectToCompare)
        {
            if ((pObjectToCompare.Header is CurveHeader) && (pFirstObject.Header is CurveHeader))
            {
                switch (this.compareElement)
                {
                    case CompareField.CurveName:
                        return String.Compare(
                            ((CurveHeader)pFirstObject.Header).curveName + " " + ((CurveHeader)pFirstObject.Header).secondaryName,
                            ((CurveHeader)pObjectToCompare.Header).curveName + " " + ((CurveHeader)pObjectToCompare.Header).secondaryName, StringComparison.Ordinal);
                    case CompareField.SecondaryName:
                        return String.Compare(((CurveHeader)pFirstObject.Header).secondaryName, ((CurveHeader)pObjectToCompare.Header).secondaryName);
                    case CompareField.Plane:
                        return String.Compare(((CurveHeader)pFirstObject.Header).plane, ((CurveHeader)pObjectToCompare.Header).plane);
                    case CompareField.ROISize:
                        return System.Collections.Comparer.DefaultInvariant.Compare(((CurveHeader)pFirstObject.Header).ROIsize, ((CurveHeader)pObjectToCompare.Header).ROIsize);
                    default:
                        return 0;
                }
            }
            else throw new TPCException("Cannot compare different header types. All headers must be same type.");
        }

        /// <summary>
        /// Gets finding function based on FindValue. If FindValue is for example CurveName, returning
        /// delegate will be name finding function
        /// </summary>
        /// <returns>Function delegate for find function.</returns>
        public override Predicate<TAC> GetFindFunction()
        {
            if (this.compareElement == CompareField.CurveName) return new Predicate<TAC>(FindByName);
            if (this.compareElement == CompareField.SecondaryName) return new Predicate<TAC>(FindBySecondaryName);
            if (this.compareElement == CompareField.Plane) return new Predicate<TAC>(FindByPlane);
            if (this.compareElement == CompareField.ROISize) return new Predicate<TAC>(FindByROI);
            return new Predicate<TAC>(base.FindNone);
        }

        /// <summary>
        /// Delegate function that finds TACS by name field
        /// </summary>
        private bool FindByName(TAC t)
        {
            // The finding value must be string
            if (!(FindValue is String)) return false;

            if (((CurveHeader)t.Header).curveName.Contains((FindValue as string))) return true;
            return false;
        }

        /// <summary>
        /// Delegate function that finds TACS by secondary name field
        /// </summary>
        private bool FindBySecondaryName(TAC t)
        {
            // The finding value must be string
            if (!(FindValue is String)) return false;

            if (((CurveHeader)t.Header).secondaryName.Contains((FindValue as string))) return true;
            return false;
        }

        /// <summary>
        /// Delegate function that finds TACS by name field
        /// </summary>
        private bool FindByPlane(TAC t)
        {
            // The finding value must be string
            if (!(FindValue is String)) return false;

            if (((CurveHeader)t.Header).plane.Contains((FindValue as string))) return true;
            return false;
        }

        /// <summary>
        /// Delegate function that finds TACS by name field
        /// </summary>
        private bool FindByROI(TAC t)
        {
            // The finding value must be string
            if (!(FindValue is Double)) return false;

            if (((CurveHeader)t.Header).ROIsize == ((Double)FindValue)) return true;
            return false;
        }


    }
}