/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using TPClib.Common;

namespace TPClib.Curve
{
    /// <summary>
    /// Definitions for a set of tacs 
    /// Fields ordered as they were in the spefication.
    /// </summary>
    public class FITHeader
    {
        /// <summary>
        /// the name of the program that was used to fit the tacs
        /// or otherwise produce the fit file.
        /// </summary>
        public string Program { get; set; }

        /// <summary>
        /// program format version 
        /// </summary>
        public string Version { get; set; }

        /// <summary>
        /// the date and time of fitting
        /// </summary>
        public DateTime DateTime { get; set; }

        /// <summary>
        /// Knows if the file contains additional clock time
        /// </summary>
        private bool noClockInDateTime = true;

        /// <summary>
        /// The name of the datafile which was fitted
        /// </summary>
        public string DataFile { get; set; }

        /// <summary>
        /// The unit of the original tacs values
        /// </summary>
        public DataUnit DataUnit { get; set; }

        /// <summary>
        /// The unit of sample tacs
        /// </summary>
        public TimeUnit TimeUnit { get; set; }

        /// <summary>
        /// Number of regions
        /// </summary>
        public int NumberOfVOIs { get; set; }

        /// <summary>
        /// Header row titles
        /// </summary>
        private static string[] name = new string[] {
            "Program", "Date", "Data file",
            "Data unit", "Time unit", "Nr of VOIs"
        };

        /// <summary>
        /// Parses the string which contains the header tacs.
        /// </summary>
        /// <param name="line">The string to be parsed.</param>
        /// <remarks>
        /// PRE: The line must look like this:
        ///   FIT(formatVersion) (genProgram)
        ///   Date: (year-month-day) (hour:min:sec)
        ///   Data file: (dir\filename)
        ///   Data unit: (unit of orginal datafiles)
        ///   Time unit: (unit of sample tacs)
        ///   Nr of VOIs: (the number of fitted tacs that follow)
        /// </remarks>
        public void Parse(string line)
        {
			using (StringReader sr = new StringReader(line))
			{
				// the 1st line
				String format = sr.ReadLine();
				if (!format.StartsWith("FIT"))
					throw new TPCFITfileException("001 File doesn't start with FIT");

				String version = format.Substring(3, 1);
				if (!(version.CompareTo("1") == 0))
					throw new TPCFITfileException("Wrong versio[" + version + "]");

				String prog = format.Substring(5);
				prog = prog.TrimStart(new char[] { ' ' });
				this.Program = prog;

				// the 2nd line
				string temp = ParseLine(sr.ReadLine(), FITHeader.name[1]);
				char[] sep = new char[] { ' ' };
				string[] spl = temp.Split(sep, StringSplitOptions.RemoveEmptyEntries);

				// checks whether there
				//      "2005-08-02 12:31:00"
				//      or just "2005-08-02" without time
				if (spl.Length == 2)
					this.noClockInDateTime = false;
				else if (spl.Length == 1)
					this.noClockInDateTime = true;
				else throw new TPCFITfileException("DateTime is incorrect.");

				try
				{
					this.DateTime = DateTime.Parse(temp);

					// the 3rd line
					this.DataFile = ParseLine(sr.ReadLine(), FITHeader.name[2]);

					// the 4th line
					this.DataUnit = DataUnit.Parse(ParseLine(sr.ReadLine(), FITHeader.name[3]));

					// the 5th line
					String timeUnit = ParseLine(sr.ReadLine(), FITHeader.name[4]);
					this.TimeUnit = TimeUnit.Parse(timeUnit);

					// the 6th line
					String VOIs = ParseLine(sr.ReadLine(), FITHeader.name[5]);
					this.NumberOfVOIs = Int16.Parse(VOIs);
					if (this.NumberOfVOIs < 0)
						throw new Exception();
				}
				catch (Exception)
				{
					throw new TPCFITfileException("Header Format Error");
				}
			}
        }

        /// <summary>
        /// Parses one header line.
        /// </summary>
        /// <returns>parsed line</returns>
        /// <remarks>
        /// PRE:   line == 'a:b' where a,b are strings
        /// POST:  RESULT == b
        /// </remarks>
        private static string ParseLine(string line, string expectedName)
        {
            string[] col = line.Split(new char[] { ':' }, StringSplitOptions.None);

            if (col.Length != 2)
                throw new TPCFITfileException("File format Exception " + col.Length);

            if (col[0].CompareTo(expectedName) != 0)
                throw new TPCFITfileException("File format Exception [" + col[0] + "]");

            return col[1].TrimStart(' ');
        }

        /// <summary>
        /// Converts the header tacs to string.
        /// </summary>
        /// <returns>the header tacs in string.</returns>
        public string HeaderToString()
        {
            FITHeader h = this;

            IFormatProvider NUMBER_FORMAT =
                System.Globalization.CultureInfo.CreateSpecificCulture("en-us");

            string s = "FIT1 ".PadRight(12) + h.Program + "\r\n";
            s += (FITHeader.name[1] + ":").PadRight(12);

            if (h.noClockInDateTime)
                s += h.DateTime.ToString("yyyy-MM-dd") + " \r\n";
            else
                s += h.DateTime.ToString("yyyy-MM-dd HH:mm:ss") + "\r\n";

            s += (FITHeader.name[2] + ":").PadRight(12) + h.DataFile + "\r\n";
            s += (FITHeader.name[3] + ":").PadRight(12) + h.DataUnit + "\r\n";
            s += (FITHeader.name[4] + ":").PadRight(12) + h.TimeUnit + "\r\n";
            s += (FITHeader.name[5] + ":").PadRight(12) + h.NumberOfVOIs + "\r\n";

            return s;
        }
    }
}