/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.Data;

namespace TPClib.Curve
{
    /// <summary>
    /// Parameter value table. The type of values in the table are not predefined. 
    /// The table consists of regions in each row and parameter values in each column. 
    /// <see cref="Parameters"/>
    /// <see cref="Regions"/>
    /// </summary>
    public class ParameterTable
    {
        /// <summary>
        /// Class for handling parameterNames in parametertable.
        /// </summary>
        public class Parameters_class
        {
            /// <summary>
            /// Parent table that has the actual parameter tacs
            /// </summary>
            ParameterTable parent;

            /// <summary>
            /// Constructs new parameters_class for parent
            /// </summary>
            /// <param name="tab">table holding parameterNames for created object</param>
            public Parameters_class(ref ParameterTable tab)
            {
                parent = tab;
            }

            /// <summary>
            /// Gets the number of parameterNames in parametertable
            /// </summary>
            public int Count
            {
                get
                {
                    return parent.table.Columns.Count - 1;
                }
            }

            /// <summary>
            /// Gets/Sets single cell values inside the parametertable
            /// </summary>
            /// <param name="param_index">Index of the parameter</param>
            /// <param name="region_index">Index of the region</param>
            /// <returns>Single TableCell value</returns>
            public TableCell this[int param_index, int region_index]
            {
                get
                {
                    return Get(param_index, region_index);
                }
                set
                {
                    Set(param_index, region_index, value);
                }
            }

            /// <summary>
            /// Gets/Sets single cell values inside the parametertable
            /// </summary>
            /// <param name="param_name">Index of the parameter</param>
            /// <param name="region_index">Index of the region</param>
            /// <returns>Single TableCell value</returns>
            public TableCell this[String param_name, int region_index]
            {
                get
                {
                    int param_index = GetIndex(param_name);
                    return Get(param_index, region_index);
                }
                set
                {
                    int param_index = GetIndex(param_name);
                    Set(param_index, region_index, value);
                }
            }


            /// <summary>
            /// Gets and Sets parameterNames with given index
            /// </summary>
            /// <param name="index">Index of parameter to get/set</param>
            /// <returns>Parameter tacs for each region</returns>
            public TableCell[] this[int index]
            {
                get
                {
                    if (index < 0 || index > Count) throw new TPCParameterTableException("this[]: Index out of bounds.");
                    return this.Get(index);
                }
                set
                {
                    if (index < 0 || index > Count) throw new TPCParameterTableException("this[]: Index out of bounds.");
                    this.Set(index, value);
                }
            }

            /// <summary>
            /// Gets and Sets parameterNames with given parametername
            /// </summary>
            /// <param name="name">Name of parameter to get/set</param>
            /// <returns>Parameter tacs for each region</returns>
            public TableCell[] this[ String name ]
            {
                get
                {
                    int index = GetIndex(name);
                    return this.Get(index);
                }
                set
                {
                    int index = GetIndex(name);
                    this.Set(index, value);
                }
            }
            /// <summary>
            /// Adds new parameter. There can't be any regions in the table. If there is regions, use
            /// Add( name, cells[] ) instead.
            /// </summary>
            /// <param name="name">Name of new parameter.</param>
            public void Add(String name) {
                Add(new String[] { name });
            }
            /// <summary>
            /// Adds new parameter. There can't be any regions in the table. If there is regions, use
            /// Add( name, cells[] ) instead.
            /// </summary>
            /// <param name="name">Names of new parameters.</param>
            public void Add(String[] name)
            {
                if (parent.table.Rows.Count > 0) throw
                   new TPCParameterTableException("Add: Cannot add empty parameterNames when there is regions present.");

                foreach (String s in name)
                {
                    DataColumn col = new DataColumn(s, Type.GetType("TPClib.Curve.TableCell"));
                    parent.table.Columns.Add(col);
                }
            }


            /// <summary>
            /// Adds a parameter with given name and TableCell for every region. Length of cells must be
            /// same as number of regions in ParameterTable.
            /// </summary>
            /// <param name="name">parameterNames(columns) name that will be created</param>
            /// <param name="cells">TableCell for every region.</param>
            public void Add(String name, TableCell[] cells)
            {
                if (cells.Length != parent.table.Rows.Count) throw
                    new TPCParameterTableException("AddParameter: Length of cell array do not match the number of regions in the table.");

                DataColumn col = new DataColumn(name, Type.GetType("TPClib.Curve.TableCell"));
                parent.table.Columns.Add(col);
                //parent.AddColumn(name, Type.GetType("TPClib.Curve.TableCell"));

                if (cells.Length > 0) Set(Count - 1, cells);
            }

            /// <summary>
            /// Gets name of one parameter in parametertable
            /// </summary>
            /// <param name="index">Index of the column</param>
            /// <returns>Name of the column in String.</returns>
            public String GetName(int index)
            {
                if (index < 0 || index >= Count) throw new TPCParameterTableException("GetParameterName: Index out of bounds.");
                return parent.table.Columns[index+1].ColumnName;
            }
            /// <summary>
            /// Sets column name.
            /// </summary>
            /// <param name="index">column index [0..no columns-1]</param>
            /// <param name="name">new name for column</param>
            public void SetName(int index, String name)
            {
                if (index < 0 || index >= Count) throw new TPCParameterTableException("SetParameterName: Index out of bounds.");
                parent.table.Columns[index + 1].ColumnName = name;
            }

            /// <summary>
            /// Set a parameter with given name and TableCell for every region. Length of cells must be
            /// same as number of regions in ParameterTable. Old parameter tacs will be owerwritten.
            /// </summary>
            /// <param name="name">parameterNames(columns) name that will be created</param>
            /// <param name="cells">TableCell for every region.</param>
            public void Set(String name, TableCell[] cells)
            {
                if (cells.Length != parent.table.Rows.Count) throw
                    new TPCParameterTableException("SetParameter: Length of cell array do not match the number of regions in the table.");
                if (!parent.table.Columns.Contains(name))
                    throw new TPCParameterTableException("SetParameter: There is no parameter " + name + " in the ParameterTable");

                for (int i = 0; i < parent.Regions.Count; i++)
                {
                    parent.table.Rows[i][name] = cells[i];
                }
            }

            /// <summary>
            /// Gets a parameter for every region from parametertable. 
            /// </summary>
            /// <param name="index">Index of the parameter.</param>
            /// <returns>Parameter for every region in the ParameterTable.</returns>
            public TableCell[] Get(int index)
            {
                if (index < 0 || index >= parent.table.Columns.Count - 1) throw new TPCParameterTableException("GetParameter: Index out of bounds.");

                TableCell[] cells = new TableCell[parent.Regions.Count];

                for (int i = 0; i < parent.Regions.Count; i++)
                {
                    cells[i] = (parent.table.Rows[i][index + 1] as TableCell);
                }

                return cells;
            }

            /// <summary>
            /// Gets one TableCell from parametertable
            /// </summary>
            /// <param name="name">Name of parameter</param>
            /// <param name="region_index">Index of region</param>
            /// <returns>Single TableCell object</returns>
            public TableCell Get(String name, int region_index)
            {
                return Get( this.GetIndex(name), region_index );
            }

            /// <summary>
            /// Sets one TableCell to parametertable
            /// </summary>
            /// <param name="name">Name of parameter</param>
            /// <param name="region_index">Index of region</param>
            /// <param name="cell">Single TableCell.</param>
            public void Set( String name, int region_index, TableCell cell)
            {
                Set( GetIndex( name ), region_index, cell );
            }

            /// <summary>
            /// Gets a single parameter from parametertable. 
            /// </summary>
            /// <param name="param_index">Index of the parameter.</param>
            /// <param name="region_index">Index of region</param>
            /// <returns>Parameter for every region in the ParameterTable.</returns>
            public TableCell Get(int param_index, int region_index)
            {
                if (param_index < 0 || param_index >= Count) throw new TPCParameterTableException("Get: Index out of bounds.");
                if (region_index < 0 || region_index >= parent.table.Rows.Count) throw new TPCParameterTableException("Get: Index out of bounds.");

                return (parent.table.Rows[region_index][param_index + 1] as TableCell);
            }

            /// <summary>
            /// Changes one cell in parametertable
            /// </summary>
            /// <param name="param_index">Index of parameter</param>
            /// <param name="index">Index of region</param>
            /// <param name="value">Value which to be set</param>
            public void Set(int param_index, int index, TableCell value)
            {
                if (param_index < 0 || param_index >= Count) throw new TPCParameterTableException("Set: Index out of bounds.");
                if (index < 0 || index >= parent.table.Rows.Count ) throw new TPCParameterTableException("Set: Index out of bounds.");

                parent.table.Rows[index][param_index+1] = value;
            }

            /// <summary>
            /// Set a parameter with given index and TableCell for every region. Length of cells must be
            /// same as number of regions in ParameterTable. Old parameter tacs will be owerwritten.
            /// </summary>
            /// <param name="id">parameterNames(columns) index that will be owerwritten</param>
            /// <param name="cells">TableCell for every region.</param>
            public void Set(int id, TableCell[] cells)
            {
                if (cells.Length != parent.table.Rows.Count) throw
                    new TPCParameterTableException("Set: Length of cell array do not match the number of regions in the table.");
                if (id < 0 || id >= Count) throw new TPCParameterTableException("Set: Index out of bounds.");

                for (int i = 0; i < parent.Regions.Count; i++)
                {
                    parent.table.Rows[i][id + 1] = (cells[i]);
                }
            }

            /// <summary>
            /// Gets parameterNames by given name and returns an array of tablecells, which are the
            /// parameter for every region in the table. For example: GetParameter( "flow" )
            /// </summary>
            /// <param name="name">String of which is compared to parameter(column) names</param>
            /// <returns>Array of tablecells which are the parameter for every region in the table.</returns>
            public TableCell[] Get(String name)
            {
                if (!parent.table.Columns.Contains(name))
                    throw new TPCParameterTableException("GetParameter: There is no parameter " + name + " in the ParameterTable");

                TableCell[] cells = new TableCell[parent.Regions.Count];

                for (int i = 0; i < parent.Regions.Count; i++)
                {
                    cells[i] = (parent.table.Rows[i][name] as TableCell);
                }

                return cells;
            }

            /// <summary>
            /// Gets parameter index of parameter with given parameter name.
            /// </summary>
            /// <param name="name">Name of parameter</param>
            /// <returns>Index number of parameter</returns>
            public int GetIndex(String name)
            {
                if (!parent.table.Columns.Contains(name))
                    throw new TPCParameterTableException("GetIndex: There is no parameter " + name + " in the ParameterTable");

                int index = parent.table.Columns.IndexOf(name) - 1;
                if (index < 0) throw new TPCParameterTableException("GetIndex: Region header don't have parameter index.");

                return index;
            }


            /// <summary>
            /// Finds parameterNames by given string and returns an ParameterTable which contain all the
            /// parameterNames which name contain the search_string. For example: ParameterTable tab = FindParameters( "ow" )
            /// </summary>
            /// <param name="search_string">String of which is compared to parameter(column) names</param>
            /// <returns>Parametertable containing the matching parameterNames</returns>
            public ParameterTable Find(String search_string)
            {
                ParameterTable ret_table = new ParameterTable(parent);

                foreach (DataColumn col in parent.table.Columns)
                {
                    if (col.Ordinal != 0 && col.ColumnName.ToLower().Contains(search_string.ToLower()))
                    {
                        //ret_table.AddColumn(col.ColumnName, col.DataType);
                        ret_table.Parameters.Add(col.ColumnName, Get(col.ColumnName));
                    }
                }

                return ret_table;
            }


            /// <summary>
            /// Sorts the table by given parameter name. If multiple same name parameterNames are found, last one
            /// is used. If the name is "Region" the table is sorted by region name
            /// </summary>
            /// <param name="name">String of which is compared to parameter names</param>
            /// <param name="isDescending">True if the result is Descending order. False if increasing.</param>
            public void Sort(String name, bool isDescending)
            {
                // the name check. Last match is used
                string colname = "";

                foreach (DataColumn col in parent.table.Columns)
                {
                    if (col.ColumnName.ToLower().Equals(name.ToLower())) colname = col.ColumnName;
                }

                if (colname.Equals("")) throw new TPCParameterTableException("Sort: Cannot find parameter name: " + name);

                DataView view = new DataView(parent.table);
                if (isDescending) view.Sort = colname + " DESC";
                else view.Sort = colname;

                parent.table = view.ToTable();
            }

            /// <summary>
            /// Sorts the table by given parameter index.
            /// </summary>
            /// <param name="index">Index of parameter</param>
            /// <param name="isDescending">True if the result is Descending order. False if increasing.</param>
            public void Sort(int index, bool isDescending)
            {
                if (index < 0 || index >= Count) throw
                   new TPCParameterTableException("Sort: Index out of bounds.");

                Sort(parent.table.Columns[index].ColumnName, isDescending);
            }

            /// <summary>
            /// Removes one parameter from ParameterTable
            /// </summary>
            /// <param name="index">Index of parameter to remove</param>
            public void Remove(int index)
            {
                if (index < 0 || index >= Count) throw new TPCParameterTableException("Delete: Index out of bounds.");
                parent.table.Columns.RemoveAt(index+1);
            }


            /// <summary>
            /// Gets type of one parameter in parametertable
            /// </summary>
            /// <param name="index">Index of column</param>
            /// <returns>The type of that column</returns>
            protected Type GetType(int index)
            {
                if (index < 0 || index >= Count) throw new TPCParameterTableException("GetColumnType: Index out of bounds.");

                return parent.table.Columns[index+1].DataType;
            }

            /// <summary>
            /// Iterator for iterating through parameterNames.
            /// </summary>
            /// <returns>List of tablecells</returns>
            public IEnumerator<TableCell[]> GetEnumerator()
            {
                for (int i = 0; i < Count; i++)
                {
                    TableCell[] p = new TableCell[parent.table.Rows.Count];
                    for (int w = 0; w < parent.table.Rows.Count; w++)
                        p[w] = (parent.table.Rows[w][i + 1] as TableCell);
                    yield return p;
                }
            }


        }

        /// <summary>
        /// Class for handling regions in parametertable.
        /// </summary>
        public class Regions_class
        {
            private ParameterTable parent;

            /// <summary>
            /// Constructs new region class
            /// </summary>
            /// <param name="tab">reference_times to parent parametertable.</param>
            public Regions_class( ref ParameterTable tab )
            {
                parent = tab;
            }

            /// <summary>
            /// Gets the number of regions in the parametertable.
            /// </summary>
            public int Count
            {
                get
                {
                    return parent.table.Rows.Count;
                }
            }

            /// <summary>
            /// Gets and Sets parameterNames with given index
            /// </summary>
            /// <param name="index">Index of region to get/set parameter tacs</param>
            /// <returns>Parameter tacs</returns>
            public TableCell[] this[int index]
            {
                get
                {
                    if (index < 0 || index > Count) throw new TPCParameterTableException("this[]: Index out of bounds.");
                    return this.GetParameters(index);
                }
                set
                {
                    if (index < 0 || index > Count) throw new TPCParameterTableException("this[]: Index out of bounds.");
                    this.SetParameters(index, value);
                }
            }

            /// <summary>
            /// Gets/Sets single value inside the parametertable
            /// </summary>
            /// <param name="region_index">Index of region</param>
            /// <param name="param_index">Index of parameter</param>
            /// <returns>Single TableCell</returns>
            public TableCell this[int region_index, int param_index]
            {
                get
                {
                    return Get(region_index, param_index);
                }
                set
                {
                    Set(region_index, param_index, value);
                }
            }

            /// <summary>
            /// Removes one region from ParameterTable
            /// </summary>
            /// <param name="index">Index of region to remove</param>
            public void Remove( int index )
            {
                if (index < 0 || index >= Count) throw new TPCParameterTableException("Delete: Index out of bounds.");
                parent.table.Rows.RemoveAt(index);
            }


            /// <summary>
            /// Gets region(header) information with given index
            /// </summary>
            /// <param name="index">Index of region.</param>
            public RegionCell GetHeader(int index)
            {
                if (index < 0 || index >= parent.table.Rows.Count) throw new TPCParameterTableException("GetHeader: Index out of bounds.");

                return (parent.table.Rows[index][0] as RegionCell);
            }

            /// <summary>
            /// Gets region(header) information with given index
            /// </summary>
            /// <param name="index">Index of region.</param>
            /// <param name="cell">New HeaderCell</param>
            public void SetHeader(int index, RegionCell cell )
            {
                if (index < 0 || index >= parent.table.Rows.Count) throw new TPCParameterTableException("SetHeader: Index out of bounds.");

                parent.table.Rows[index][0] = cell;
            }

            /// <summary>
            /// Gets one TableCell from parametertable
            /// </summary>
            /// <param name="region_index">Index of region</param>
            /// <param name="param_index">Index of parameter</param>
            /// <returns>Single TableCell</returns>
            public TableCell Get(int region_index, int param_index)
            {
                if (region_index < 0 || region_index >= Count) throw
                     new TPCParameterTableException("Get: Region_Index out of bounds.");
                if (param_index < 0 || param_index >= parent.table.Columns.Count-1) throw
                     new TPCParameterTableException("Get: Parameter_Index out of bounds.");

                return (parent.table.Rows[region_index][param_index + 1] as TableCell);
            }

            /// <summary>
            /// Sets one TableCell in ParameterTable
            /// </summary>
            /// <param name="region_index">Index of region</param>
            /// <param name="param_index">Index of parameter</param>
            /// <param name="cell">TableCell which will be set to table</param>
            public void Set(int region_index, int param_index, TableCell cell )
            {
                if (region_index < 0 || region_index >= Count) throw
                     new TPCParameterTableException("Get: Region_Index out of bounds.");
                if (param_index < 0 || param_index >= parent.table.Columns.Count - 1) throw
                       new TPCParameterTableException("Get: Parameter_Index out of bounds.");

                parent.table.Rows[region_index][param_index + 1] = cell;
            }

            /// <summary>
            /// Gets parameterNames of one region with given region index
            /// </summary>
            /// <param name="index">Index of region</param>
            /// <returns>All TableCells containing the information for every parameter on the ParameterTable.</returns>
            public TableCell[] GetParameters(int index)
            {
                if (index < 0 || index >= parent.table.Rows.Count)
                    throw new TPCParameterTableException("GetFittedParameters: Index out of bounds.");

                TableCell[] cells = new TableCell[parent.table.Columns.Count - 1];
                for (int i = 0; i < parent.table.Columns.Count - 1; i++)
                {
                    cells[i] = (parent.table.Rows[index][i + 1] as TableCell).Clone();
                }

                return cells;
            }

            /// <summary>
            /// Gets all the parameterNames of given region name. If there is multiple regions with same name, the last will be
            /// given.
            /// </summary>
            /// <param name="RegionName">Name of region to fetch</param>
            /// <returns>All the parameterNames of the given region</returns>
            public TableCell[] GetParameters(String RegionName)
            {
                int index = -1;
                for (int i = 0; i < parent.Regions.Count; i++) if ((parent.table.Rows[i][0] as RegionCell).Region.Name.ToLower().Equals(RegionName.ToLower()))
                    {
                        index = i;
                    }
                if (index == -1) throw new TPCParameterTableException("There is no such region as " + RegionName);

                return GetParameters(index);
            }

            /// <summary>
            /// Sets parameterNames of one region. All the old tacs will be lost.
            /// </summary>
            /// <param name="index">Index of region.</param>
            /// <param name="cells">new values for region</param>
            public void SetParameters(int index, TableCell[] cells)
            {
                if (index < 0 || index >= parent.table.Rows.Count)
                    throw new TPCParameterTableException("GetFittedParameters: Index out of bounds.");

                for (int i = 1; i < parent.table.Columns.Count; i++)
                {
                    parent.table.Rows[index][i] = cells[i - 1];
                }
            }

            /// <summary>
            /// Sets parameterNames to region with given name. THIS IS CURRENTLY WORKING ONLY WITH RES HEADERS!
            /// </summary>
            /// <param name="RegionName">Name of region.</param>
            /// <param name="cells">new values that are inserted to region</param>
            public void SetParameters(String RegionName, TableCell[] cells)
            {
                int index = -1;
                for (int i = 0; i < parent.Regions.Count; i++) if ((parent.table.Rows[i][0] as RegionCell).Region.Name.ToLower().Equals(RegionName.ToLower()))
                    {
                        index = i;
                    }
                if (index == -1) throw new TPCParameterTableException("There is no such region as " + RegionName);

                SetParameters(index, cells);
            }

            /// <summary>
            /// Adds new region to Parametertable. Number of parameter values must match the number of parameter values in
            /// the parametertable.
            /// </summary>
            /// <param name="region_cell">The header cell containing the region information. For example region_cell.</param>
			/// <param name="parameters">List of TableCells containing the parameter values of the region.</param>
            /// <param name="index">Index of new region [0...Count-1]</param>
            public void AddAt(int index, RegionCell region_cell, TableCell[] parameters)
            {
                if (index < 0 || index > Count) throw
                   new TPCParameterTableException("AddAt: Index out of bounds.");
                if (region_cell == null) throw
                   new TPCParameterTableException("Add: region_cell cant be null.");
                if (parameters == null) throw
                   new TPCParameterTableException("Add: parameterNames cant be null.");
                if (parent.table.Rows.Count > 0)
                {
                    if (parameters.Length + 1 != parent.table.Columns.Count)
                        throw new TPCParameterTableException("Add: number of parameter values do not match the number of parameter values in the parametertable.");
                }
                else if (parent.table.Columns.Count == 0)
                {
                    //add columns for new row
                    for (int i = 0; i < parameters.Length + 1; i++)
                        parent.table.Columns.Add();
                }
                else if (parent.table.Columns.Count != parameters.Length + 1)
                {
                    throw new TPCParameterTableException("Add: number of parameter values do not match the number of parameter values in the parametertable.");
                }
                //place region cell in the first position
                DataRow row = parent.table.NewRow();
                row[0] = region_cell;
                //place parameter values in the following positions
                for (int i = 0; i < parent.table.Columns.Count - 1; i++)
                {
                    row[i + 1] = parameters[i];
                }
                parent.table.Rows.InsertAt(row, index);
            }

            /// <summary>
            /// Adds new region to Parametertable. number of parameter values must match the number of parameter values in
            /// the parametertable.
            /// </summary>
            /// <param name="region_cell">The header cell containing the region information. For example region_cell.</param>
            public void Add(RegionCell region_cell)
            {
                DataRow row = parent.table.NewRow();
                row[0] = region_cell;
                parent.table.Rows.Add(row);
            }

            /// <summary>
            /// Adds new region to Parametertable. number of parameterNames must match the number of parameterNames in
            /// the parametertable.
            /// </summary>
            /// <param name="region_cell">The header cell containing the region information. For example region_cell.</param>
			/// <param name="parameters">List of TableCells containing the parameter values of the region.</param>
            public void Add(RegionCell region_cell, TableCell[] parameters)
            {
                if (parameters.Length > 0)
                {
                    AddAt(Count, region_cell, parameters);
                }
                else // if there is only one cell, we must deal it differently
                {
                    Add(region_cell);
                }
            }


            /// <summary>
            /// Gets index of region with given region name. THIS WORKS ONLY WITH RES HEADERS!!
            /// </summary>
            /// <param name="name">Name of region</param>
            /// <returns>Index number of region</returns>
            public int GetIndex(String name)
            {
                int index = -1;
                for (int i = 0; i < parent.Regions.Count; i++) if ((parent.table.Rows[i][0] as RegionCell).Region.Name.ToLower().Equals(name.ToLower()))
                    {
                        index = i;
                    }
                if (index == -1) throw new TPCParameterTableException("There is no such region as " + name);

                return index;
            }

            /// <summary>
            /// Iterator for iterating through regions.
            /// </summary>
            /// <returns>List of tablecells</returns>
            public IEnumerator<TableCell[]> GetEnumerator()
            {
                for (int i=0; i<parent.table.Rows.Count; i++ )
                {
                    TableCell[] p = new TableCell[ parent.table.Columns.Count-1 ];
                    for( int w=0; w<parent.table.Columns.Count-1; w++ )
                        p[w] = (parent.table.Rows[i][w+1] as TableCell);
                    yield return p;
                }
            }
        }

        /// <summary>
        /// Table of tacs inside table.
        /// </summary>
        protected DataTable table;
        /// <summary>
        /// Parameter values in this table.
        /// </summary>
        public Parameters_class Parameters;
        /// <summary>
        /// Regions in this table.
        /// </summary>
        public Regions_class Regions;
        /// <summary>
        /// Parameter units as string
        /// </summary>
        private List<string> parameter_units;
        /// <summary>
        /// Gets parameter units, or null if units are not specified in constructor
        /// </summary>
        public String[] Parameter_Units {
            get {
                if (parameter_units == null) return null;
                return parameter_units.ToArray();
            }
            set
            {
                parameter_units = new List<string>(value);
            }
        }
        #region general methods

        /// <summary>
        /// Default constructor. 
        /// </summary>
        public ParameterTable()
        {
            Init();
        }

        /// <summary>
        /// Creates ParameterTable with Regions
        /// </summary>
        /// <param name="RegionHeaders">Array of RegionHeaderCells</param>
        public ParameterTable(RegionCell[] RegionHeaders )
        {
            Init();
            foreach (RegionCell rh in RegionHeaders)
            {
                this.Regions.Add(rh);
            }
        }
        /// <summary>
        /// Creates new ParameterTable with parameter names
        /// </summary>
        /// <param name="param_name">name for parameter.</param>
        /// <param name="unit">unit for parameter</param>
        public ParameterTable(String param_name, String unit)
        {
            Init();
            this.Parameters.Add(new string[] {param_name});
            parameter_units = new List<String>(new string[] {unit});
        }
        /// <summary>
        /// Creates new ParameterTable with parameter names
        /// </summary>
        /// <param name="param_names">N-length array of strings containing names for each parameter.</param>
        /// <param name="units">N-length array of units for each parameter</param>
        public ParameterTable(String[] param_names, String[] units)
        {
            if (param_names.Length != units.Length)
                throw new TPCCurveFileException("Number of parameter names "+param_names.Length+" does not match length of unit types "+units.Length);
            Init();
            this.Parameters.Add(param_names);
            parameter_units = new List<String>(units);
        }

        /// <summary>
        /// Creates new ParameterTable with parameter names
        /// </summary>
        /// <param name="param_names">Array of Strings containing names for each parameter.</param>
        public ParameterTable(String[] param_names)
        {
            Init();
            this.Parameters.Add(param_names);
        }

        /// <summary>
        /// Creates an empty parametertable (without parameterNames) with region information given by 
        /// another parametertable.
        /// </summary>
        /// <param name="source_structure">Parametertable which region name information is copied to new one.</param>
        public ParameterTable(ParameterTable source_structure)
        {
            Init();

            for( int i=0; i<source_structure.Regions.Count; i++ )
            {
                DataRow row = table.NewRow();
                row[0] = source_structure.Regions.GetHeader(i);

                table.Rows.Add( row ); 
            }
        }

        /// <summary>
        /// Initializes values of this object
        /// </summary>
        private void Init()
        {
            ParameterTable t = this;
            this.Parameters = new Parameters_class(ref t);
            this.Regions = new Regions_class(ref t);
            this.table = new DataTable();
            DataColumn col = new DataColumn("Region", Type.GetType("TPClib.Curve.RegionCell"));
            this.table.Columns.Add(col);
        }


        /// <summary>
        /// The zero-based indexes for parameter tables items.
        /// </summary>
        /// <param name="x">Index of parameter</param>
        /// <param name="y">Index of region</param>
        /// <returns>Single TableCell from parametertable</returns>
        public TableCell this[int x, int y]
        {
            get
            {
                if (x >= this.table.Columns.Count-1 || x < 0) throw new TPCParameterTableException("this[x,y]: Index out of bounds.");
                if (y >= this.table.Rows.Count || y < 0) throw new TPCParameterTableException("this[x,y]: Index out of bounds.");
                return (table.Rows[y][x+1] as TableCell);
            }
            set
            {
                if (x >= this.table.Columns.Count-1 || x<0 ) throw new TPCParameterTableException("this[x,y]: Index out of bounds.");
                if (y >= this.table.Rows.Count || y < 0) throw new TPCParameterTableException("this[x,y]: Index out of bounds.");
                table.Rows[y][x+1] = value;
            }

        }

        /// <summary>
        /// Creates a fixed size String from input string and result length.
        /// All extra space is filled with spaces. If the input string is longer than result,
        /// the remainders are cut off
        /// </summary>
        /// <param name="input">The input string</param>
        /// <param name="length">length if result string</param>
        /// <returns>string with fixed size</returns>
        protected String FixStr(String input, int length)
        {
            // null input is treated like empty string 
            if (input == null) return new String(' ', length);

            int difference = length - input.Length;

            // the input string is longer than the boundaries given
            if (difference < 0) { return input.Substring(0, length); }
            else if (difference == 0) { return input; }
            else return input.PadRight(input.Length + difference, ' ');
        }

        /// <summary>
        /// Checks if two parametertables are identical by contents.
        /// </summary>
        /// <param name="obj">Another parametertable</param>
        /// <returns>True if the tables are identical.</returns>
        public override bool Equals(Object obj)
        {
            if ( !(obj is ParameterTable) ) return false;

            ParameterTable tab = (obj as ParameterTable);

            for (int i = 0; i < Parameters.Count; i++)
            {
                if (!this.table.Columns[i+1].ColumnName.Equals(tab.table.Columns[i+1].ColumnName)) return false;
            }

            if (Regions.Count != tab.Regions.Count) return false;
            if (Parameters.Count != tab.Parameters.Count) return false;

            for (int y = 0; y < Regions.Count; y++)
            {
                for (int x = 0; x < Parameters.Count+1; x++)
                {
                    if (!table.Rows[y][x].Equals(tab.table.Rows[y][x]))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        /// <summary>
        /// Prints the contents of parametertable to stdout.
        /// </summary>
        public void WriteToSTDOut()
        {
            Console.WriteLine( ToString() );
        }

        /// <summary>
        /// Prints the contents of the parametertable to String
        /// </summary>
        /// <returns>String containing the contents of the parametertable</returns>
        public override string ToString()
        {
            StringBuilder str = new StringBuilder();

            // The parameter names
            for (int i = 0; i < table.Columns.Count; i++) str.Append( FixStr( table.Columns[i].ColumnName, 12 ) );
            str.Append("\n");

            // tacs
            for (int y = 0; y < table.Rows.Count; y++)
            {
                for (int x = 0; x < table.Columns.Count; x++)
                {
                    str.Append( FixStr( table.Rows[y][x].ToString(), 12 ) );
                }
                str.Append( "\n" );
            }

            return str.ToString();
        }

        #endregion
        /// <summary>
        /// Gets hash code of this object. Default bahaviour.
        /// </summary>
        /// <returns>returns hash code with default implementation</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }

}
