/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Curve
{
    /// <summary>
    /// Single RESCell containing result information. Value, standard deviation, upper CL and lower CL
    /// </summary>
    public class RESCell : TableCell
    {
        /// <summary>
        /// Standard deviation value for this parameter
        /// </summary>
        public Double SD;
        /// <summary>
        /// Upper confidence limit for this parameter
        /// </summary>
        public Double CL_upper;
        /// <summary>
        /// Lower confidence limit for this parameter
        /// </summary>
        public Double CL_lower;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="value">Value for the parameter.</param>
        public RESCell(Double value)
        {
            this.Value = value;
            this.SD = Double.NaN;
            this.CL_upper = Double.NaN;
            this.CL_lower = Double.NaN;
        }

        /// <summary>
        /// Constructor that takes parameter value and standard deviation in.
        /// </summary>
        /// <param name="value">The parameter value</param>
        /// <param name="sd">The standard deviation value</param>
        public RESCell(Double value, Double sd)
        {
            this.Value = value;
            this.SD = sd;
            this.CL_lower = Double.NaN;
            this.CL_upper = Double.NaN;
        }

        /// <summary>
        /// Constuctor that takes parameter value and limit information
        /// </summary>
        /// <param name="value">The parameter value</param>
        /// <param name="cl_upper">The upper confidence limit</param>
        /// <param name="cl_lower">The lower confidence limit</param>
        public RESCell(Double value, Double cl_upper, Double cl_lower)
        {
            this.Value = value;
            this.SD = Double.NaN;
            this.CL_upper = cl_upper;
            this.CL_lower = cl_lower;
        }


        /// <summary>
        /// Checks if two REScells are identical by contents
        /// </summary>
        /// <param name="obj">Another rescell</param>
        /// <returns>True if the cells are identical.</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is RESCell)) return false;

            RESCell rc = (obj as RESCell);

            if (!DoubleCheck( this.CL_lower , rc.CL_lower ) ) return false;
            if (!DoubleCheck(this.CL_upper, rc.CL_upper)) return false;
            if (!DoubleCheck(this.Value, rc.Value)) return false;
            if (!DoubleCheck(this.SD , rc.SD) ) return false;

            return true;
        }

        /// <summary>
        /// Returns copy of the object
        /// </summary>
        /// <returns>Copy of the object.</returns>
        public override TableCell Clone()
        {
            RESCell cell = new RESCell(this.Value);
            cell.CL_lower = this.CL_lower;
            cell.CL_upper = this.CL_upper;
            cell.SD = this.SD;
            return cell;
        }

        /// <summary>
        /// Prints the RESCells value to string
        /// </summary>
        /// <returns>She string containing the value of REScell</returns>
        public override String ToString()
        {
            return this.Value.ToString();
        }
        /// <summary>
        /// Gets hash code of this object. Default bahaviour.
        /// </summary>
        /// <returns>returns hash code with default implementation</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
