/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.Curve
{
    /// <summary>
    /// Time Activity Curve (TAC). Contains curveheader information and possibility to sort TACs by
    /// CurveName, secondaryname, ROIsize, Plane
    /// </summary>
    public class TAC
    {
        /// <summary>
        /// Header tacs for DFT TAC
        /// </summary>
        public CurveHeader Header;

        /// <summary>
        /// TACs Curve tacs
        /// </summary>
        protected List<TableCell> cells;

        /// <summary>
        /// Default constructor. All fields are set to null
        /// </summary>
        public TAC()
        {
            cells = new List<TableCell>();
            //curveDataUnit = Unit.Unknown;
            Header = new CurveHeader();
        }

        /// <summary>
        /// Removes all time elements from the TAC.
        /// </summary>
        public void Clear() {
            cells.Clear();
        }
        /// <summary>
        /// Adds an object to the end of the System.Collections.Generic.List{T}.
        /// </summary>
        /// <param name="cell">The object to be added to the end of the TAC. </param>
        public void Add(TableCell cell) {
            cells.Add(cell);
        }
        
        /// <summary>
        /// Initializes a DFTCurveTAC without time information
        /// </summary>
        /// <param name="curh">DFT Curve header information</param>
        /// <param name="tcells">An array of TableCells to fill TAC tacs</param>
        public TAC(CurveHeader curh, TableCell[] tcells)
        {
            if (tcells == null) throw new TPCTACTableException("TAC: Input tacs can't be null.");
            if (curh == null) Header = new CurveHeader();

            Header = curh;

            cells = new List<TableCell>();
            //curveDataUnit = Unit.Unknown;

            foreach (TableCell ftc in tcells)
            {
                TableCell tc = new TableCell();
                tc.Value = ftc.Value;
                cells.Add(tc);
            }
        }

        /// <summary>
        /// Constructs time activity curve from other time activity curve.
        /// </summary>
        /// <param name="cc">time activity curve that has tacs for new time activity curve</param>
        public TAC(TAC cc)
        {
            if (cc == null) throw new TPCTACTableException("TAC: Input tacs can't be null.");
            SetData(cc);
        }
        /// <summary>
        /// Sets tacs in time activity curve.
        /// </summary>
        /// <param name="cc">values that are placed into this time activity curve</param>
        public void SetData( TAC cc )
        {
            Header = cc.Header;

            if (cc.cells == null) throw new TPCTACTableException("TAC: Input tacs can't be null.");

            //curveDataUnit = cc.curveDataUnit;
            cells = new List<TableCell>();
            foreach (TableCell tc in cc.cells)
            {
                TableCell tabc = new TableCell();
                tabc.Value = tc.Value;
                cells.Add(tabc);
            }
        }
        /// <summary>
        /// Returns TAC values.
        /// </summary>
        /// <remarks>Note that actual unit of values can be found when retrieving TableCell items.</remarks>
        /// <returns>values as a double array</returns>
        public double[] GetValues() {
            List<double> values = new List<double>();
            for (int i = 0; i < cells.Count; i++) {
                values.Add(cells[i].Value);
            }
            return values.ToArray();
        }
        /// <summary>
        /// Compares two DFTCurveTACs and returns true if the contents are identical
        /// </summary>
        /// <param name="first">first TAC</param>
        /// <param name="second">second TAC</param>
        /// <returns>true if the TACs are identical</returns>
        public static bool Equals(TAC first, TAC second)
        {
            if (first == null || second == null) throw new TPCTACTableException("Cannot compare TACs. (Null reference_times)");
            if (first.cells.Count != second.cells.Count) return false;

            //if (first.curveDataUnit != second.curveDataUnit) return false;

            // TableCell comparing
            if (first.cells != null && second.cells != null)
            {
                if (first.cells.Count != second.cells.Count) return false;
                for (int i = 0; i < first.cells.Count; i++)
                {
                    if (first.cells[i].Value != second.cells[i].Value) return false;
                }
            }

            // header comparing
            if (first.Header == null && second.Header == null) { }
            else if (first.Header != null && second.Header != null)
            {
                first.Header.Equals(second.Header);
            }
            else
                return false; // another is null another not

            return true;
        }

        /// <summary>
        /// Explicit cast to array of doubles
        /// </summary>
        /// <param name="t">TAC to convert</param>
        /// <returns>Array of doubles</returns>
        public static explicit operator double[](TAC t)
        {
            int n = t.cells.Count;
            double[] arr = new double[n];
            for (int i = 0; i < n; i++)
            {
                arr[i] = (double)(t.cells[i]);
            }
            return arr;
        }

        /// <summary>
        /// Explicit cast to vector
        /// </summary>
        /// <param name="t">TAC to convert</param>
        /// <returns>Vector containing TAC values</returns>
        public static explicit operator Vector(TAC t)
        {
            int n = t.cells.Count;
            Vector arr = new Vector(n);
            for (int i = 0; i < n; i++)
            {
                arr[i] = (double)(t.cells[i]);
            }
            return arr;
        }

        /// <summary>
        /// Gets String representation of this object.
        /// </summary>
        /// <returns>TAC as string</returns>
        public override string ToString()
        {
            String text = "";
            if (this.Header != null) text = this.Header.ToString();
            
            foreach (TableCell tc in this.cells)
            {
               text += tc.Value.ToString() + "\n";
            }
            return text;
        }
        /// <summary>
        /// Gets TAC value at index
        /// </summary>
        /// <param name="i">the zero-based inxed for tacs</param>
        /// <returns>tacs cell</returns>
        public TableCell this[int i] {
            get {
                return cells[i];
            }
        }
        /// <summary>
        /// Casting operator from Tablecell[] list
        /// </summary>
        /// <param name="list">original list</param>
        /// <returns>TAC containing list values and empty header</returns>
        public static explicit operator TAC (TableCell[] list) {
            return new TAC(new CurveHeader(), list);
            
        }
        /// <summary>
        /// Number of time points
        /// </summary>
        public int Length { get { return cells.Count; } }

        /// <summary>
        /// Copies the elements of TAC to a new array.
        /// </summary>
        /// <returns>An array containing activity values of the TAC.</returns>
        public TableCell[] ToArray() { return cells.ToArray(); }
    }

}
