/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Curve
{
    /// <summary>
    /// Table cell with weight value
    /// </summary>
    public class WeightedTableCell : TableCell
    {
        /// <summary>
        /// Weight of the cell value
        /// </summary>
        public double Weight;

        /// <summary>
        /// Weighted cell value
        /// </summary>
        public double WeightedValue { get; set; }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="value">Cell value</param>
        /// <param name="weight">Cell weight</param>
        public WeightedTableCell(double value, double weight)
        {
            this.Value = value;
            this.Weight = weight;
            this.WeightedValue = value * weight;
        }

        /// <summary>
        /// String representation of weighted value
        /// </summary>
        /// <returns>String representation of unweighted value of this cell</returns>
        public override string ToString()
        {
            return base.ToString();
        }

        /// <summary>
        /// Equality
        /// </summary>
        /// <param name="a">First cell compared</param>
        /// <param name="b">Second cell compared</param>
        /// <returns>True, if value and weight of the cells are equal. False otherwise.</returns>
        public static bool operator ==(WeightedTableCell a, WeightedTableCell b)
        {
            return a.Equals(b);
        }

        /// <summary>
        /// Inequality
        /// </summary>
        /// <param name="a">First cell compared</param>
        /// <param name="b">Second cell compared</param>
        /// <returns>False, if value and weight of the cells are equal. True otherwise.</returns>
        public static bool operator !=(WeightedTableCell a, WeightedTableCell b)
        {
            return !(a.Equals(b));
        }

        /// <summary>
        /// Equality
        /// </summary>
        /// <param name="o">Object this WeightedTableCell is compared to</param>
        /// <returns>
        /// True, if argument object is WeightedTableCell of equal value and weight.
        /// False otherwise
        /// </returns>
        public override bool Equals(Object o)
        {
            try
            {
                return ((this.Value == (o as WeightedTableCell).Value) &&
                        (this.Weight == (o as WeightedTableCell).Weight));
            }
            catch
            {
                throw new TPCIDWCFileException("Objects not mutually comparable");
            }
        }

        /// <summary>
        /// Hash code
        /// </summary>
        /// <returns>Hash code for this cell</returns>
        public override int GetHashCode()
        {
            return (this.Weight.GetHashCode() ^ this.Value.GetHashCode());
        }
    }
}
